/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.parser;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.queries.JavadocForBinaryQuery;
import org.netbeans.api.java.queries.JavadocForBinaryQuery.Result;
import org.netbeans.modules.classfile.*;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.jmiimpl.javamodel.FeatureImpl;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.URLMapper;
import org.openide.util.Utilities;

public final class ClassFileInfoUtil {
    private static final String NAME_CLASS_INIT = "<clinit>"; // NOI18N
    private static final String NAME_INIT = "<init>"; // NOI18N

    /**
     * Only for debugging reasons, used for sumarize time consumption for
     * getting parameters from javadoc.
     */
    private static long javadocTT = 0;
    private static long lastPrinted = 0;
    
    public ClassInfo createClassInfo(FileObject folder, String partName, int mods) {
        FileObject file = null;
        try {
            file = folder.getFileObject(partName.replace('.', '$'), "class"); // NOI18N
        } catch (Exception e) {
        }
        if (file == null) {
            return null;
        }
        return createClassInfo (folder, file, mods);
    }
    
    public ClassInfo createClassInfo(FileObject folder, FileObject file, int mods) {
        ClassFile clsFile;
        
        try {
            InputStream is=file.getInputStream();
            try {
                clsFile = new ClassFile(is, true);
            } finally {
                is.close();
            }
        } catch (Exception ex) {
            throw new RuntimeException("Inconsistent storage: class file for " + file.getNameExt() + " not created", ex); // NOI18N
        }
        
        String name = clsFile.getName().getExternalName();
        if (mods < 0) {
            mods = (clsFile.getAccess() & ~Access.SYNCHRONIZED) | (clsFile.isDeprecated() ? FeatureImpl.DEPRECATED : 0);
        }
        boolean isInterface = (mods & Access.INTERFACE) > 0;
        String typeSignature = clsFile.getTypeSignature();
        boolean isGeneric = typeSignature != null;
        TypeParamInfo[] tpi = null;
        NameRef[] interfaces;
        NameRef superName;

        if (isGeneric) {
            char[] sigChars = typeSignature.toCharArray();
            int[] pos = new int[] {0};
            tpi = parseTypeParams(sigChars, pos);
            superName = (NameRef)sigToTypeRef(sigChars, pos);
            ArrayList list = new ArrayList(6);
            while (pos[0] < sigChars.length) {
                list.add(sigToTypeRef(sigChars, pos));
            }
            interfaces = new NameRef[list.size()];
            list.toArray(interfaces);
        } else {
            ClassName className = clsFile.getSuperClass();
            superName = null;
            if (className != null) {
                superName = new NameRef(className.getExternalName(), null, null);
            }

            Collection coll = clsFile.getInterfaces();
            int size = coll.size();
            interfaces = new NameRef[size];
            Iterator iter = coll.iterator();            
            for (int x = 0; iter.hasNext(); x++) {
                interfaces[x] = new NameRef(((ClassName) iter.next()).getExternalName(), null, null);
            }
        }
        
        boolean isAnnotation = clsFile.isAnnotation();
        int methodCount = clsFile.getMethodCount();
        int fieldCount = clsFile.getVariableCount();
        Collection innerClasses = clsFile.getInnerClasses();
        ClassName clsName = clsFile.getName();
        List featuresList = new ArrayList (innerClasses.size() + methodCount + fieldCount);
        for (Iterator it = innerClasses.iterator(); it.hasNext();) {
            InnerClass innerClass = (InnerClass) it.next();
            ClassName outerName = innerClass.getOuterClassName();
            if (clsName.equals(outerName)) {                                
                String fn = innerClass.getName().getSimpleName();
                
                int index = fn.lastIndexOf ('.') + 1;
                if ((index > 0) && (fn.length() > index) && Character.isDigit(fn.charAt (index))) {
                    continue;
                }
        
                fn = fn.replace('.', '$');
                FileObject fo = folder.getFileObject(fn, "class"); // NOI18N
                if (fo != null) {
                    featuresList.add(createClassInfo(folder, fo, -1));
                }
            }
        }

        Iterator iter = clsFile.getMethods().iterator();
        for (int x = 0; x < methodCount; x++) {
            Method method = (Method) iter.next();
            if (!method.isSynthetic()) {
                if (isAnnotation) {
                    featuresList.add(createAttributeInfo(method));
                } else {
                    featuresList.add(createMethodInfo(name, method, file));
                }
            }
        }
        List enumConstants = new ArrayList();
        iter = clsFile.getVariables().iterator();
        for (int x = 0 ; x < fieldCount; x++) {
            Variable variable = (Variable) iter.next();
            if (!variable.isSynthetic()) {
                if (variable.isEnumConstant()) {
                    enumConstants.add(variable);
                } else {
                    featuresList.add(createFieldInfo (variable));
                }
            }
        }
        FeatureInfo[] features = new FeatureInfo[featuresList.size()];
        featuresList.toArray(features);
        AnnotationInfo[] annos = createAnnotations(clsFile.getAnnotations());
        
        if (clsFile.isEnum()) {
            ElementInfo[] constants = new ElementInfo[enumConstants.size()];
            Iterator it = enumConstants.iterator();
            for (int x = 0; it.hasNext(); x++) {
                Variable c = (Variable)it.next();
                constants[x] = new FeatureInfo(null, FeatureInfo.ENUM_CONSTANT_TYPE, (String)c.getName(), 0, null);
            }
            return new EnumInfo(
                null, EnumInfo.ENUM_TYPE, name, mods, features, interfaces, constants, annos
            );
        } else if (isAnnotation) {
            return new AnnotationTypeInfo(null, AnnotationTypeInfo.ANNOTATIONTYPE_TYPE, name, mods,
                features, annos
            );
        } else {
            return new ClassInfo(null,
                isInterface ? ClassInfo.INTERFACE_TYPE : ClassInfo.CLASS_TYPE,
                name, mods, features, superName, interfaces, tpi, annos
            );
        }
    }
    
    public FeatureInfo createMethodInfo (String className, Method method, FileObject file) {        
        String name = method.getName();
        int mods = method.getAccess() | (method.isDeprecated() ? FeatureImpl.DEPRECATED : 0);
        AnnotationInfo[] annos = createAnnotations(method.getAnnotations());
        if (NAME_CLASS_INIT.equals(name)) {
            int infoType = (mods & Access.STATIC) > 0 ? FeatureInfo.STATIC_INITIALIZER_TYPE :
                FeatureInfo.INSTANCE_INITIALIZER_TYPE;
            return new FeatureInfo (null, infoType, name, mods, annos);
        }
        
        char[] sig = method.getDescriptor().toCharArray();
        String typeSignature = method.getTypeSignature();
        char[] typeSig = typeSignature == null ? null : typeSignature.toCharArray();
        int[] pos = new int[] {1};
        ParameterInfo[] params;
        TypeParamInfo[] tpi = null;

        if (typeSignature == null) {
            params = createParamsInfo(method, sig, pos, file);
        } else {
            pos[0] = 0;
            tpi = parseTypeParams(typeSig, pos);
            List list = getParamTypes(typeSig, pos);
            Iterator paramsIter = method.getParameters().iterator();
            Iterator iter = list.iterator();
            params = new ParameterInfo[list.size()];
            for (int x = 0; iter.hasNext(); x++) {
                TypeRef type = (TypeRef)iter.next();
                boolean isVarArgs = !iter.hasNext() && method.isVarArgs();
                Parameter par = paramsIter.hasNext() ? (Parameter)paramsIter.next() : null;
                String paramName;
                AnnotationInfo[] anns;
                if (par == null) {
                    paramName = "";
                    anns = null;
                } else {
                    paramName = par.getName();
                    anns = createAnnotations(par.getAnnotations());
                }
                params[x] = new ParameterInfo (null, ParameterInfo.PARAMETER_TYPE, paramName, false, type, isVarArgs, anns);
            }
        }
        
        TypeRef type;
        int infoType;
        if (NAME_INIT.equals(name)) {
            infoType = MethodInfo.CONSTRUCTOR_TYPE;
            type = new NameRef(className, null, null);
            if (typeSignature == null) {
                sigToTypeRef(sig, pos);
            } else {
                pos[0]++;
                sigToTypeRef(typeSig, pos);
            }
        } else {
            infoType = MethodInfo.METHOD_TYPE;
            if (typeSignature == null) {
                type = sigToTypeRef(sig, pos);
            } else {
                pos[0]++;
                type = sigToTypeRef(typeSig, pos);
            }
        }
        TypeParamRef[] excNames = new TypeParamRef[0];
        CPClassInfo[] exceptions = method.getExceptionClasses();
        if (exceptions.length > 0) {
            if (typeSignature != null) {
                ArrayList list = new ArrayList();
                while (pos[0] < typeSig.length) {
                    if (typeSig[pos[0]] == '^')
                        pos[0]++; 
                    list.add(sigToTypeRef(typeSig, pos));
                }
                excNames = (TypeParamRef[]) list.toArray(excNames);
            }
            if (excNames.length == 0) {
                excNames = new TypeParamRef[exceptions.length];
                for (int x = 0; x < exceptions.length; x++) {
                    excNames[x] = new NameRef(exceptions[x].getClassName().getExternalName(), null, null);
                }
            }
        }
        return new MethodInfo (null, infoType, name, mods, type, params, excNames, tpi, annos);
    }

    public static FieldInfo createFieldInfo(Variable field) {
        int mods = field.getAccess() | (field.isDeprecated() ? FeatureImpl.DEPRECATED : 0);
        String name = field.getName();
        AnnotationInfo[] annos = createAnnotations(field.getAnnotations());
        String typeSignature = field.getTypeSignature();
        char[] signature = typeSignature != null ? typeSignature.toCharArray() : field.getDescriptor().toCharArray();
        int[] pos = new int[] {0};
        TypeRef type = sigToTypeRef(signature, pos);
        return new FieldInfo (null, FieldInfo.FIELD_TYPE, name, mods, type, FieldInfo.SINGLE_FIELD_INDEX, annos);
    }
    
    public static AttributeInfo createAttributeInfo(Method method) {
        int mods = method.getAccess() | (method.isDeprecated() ? FeatureImpl.DEPRECATED : 0);
        String name = method.getName();
        AnnotationInfo[] annos = createAnnotations(method.getAnnotations());
        String desc = method.getDescriptor();
        int index = desc.indexOf(')');
        TypeRef type = sigToTypeRef(desc.substring(index + 1).toCharArray(), new int[] {0});
        ElementValue defValue = method.getAnnotationDefault();
        AnnotationValueInfo defaultValueInfo = defValue == null ? null : new AnnotationValueInfo(
            null, getElementValueType(defValue), name, resolveElementValue(defValue)
        );
        return new AttributeInfo(null, AttributeInfo.ATTRIBUTE_TYPE, name, mods, type, defaultValueInfo, annos);
    }
    
    public static AnnotationInfo[] createAnnotations(Collection annos) {
        if (annos == null || annos.size() == 0) {
            return ElementInfo.EMPTY_ANNOTATIONS;
        }
        AnnotationInfo[] result = new AnnotationInfo[annos.size()];
        Iterator iter = annos.iterator();
        for (int x = 0; iter.hasNext(); x++) {
            result[x] = annotationToInfo((Annotation)iter.next());
        }
        return result;
    }
    
    public static AnnotationInfo annotationToInfo(Annotation anno) {
        NameRef name=new NameRef(anno.getType().getExternalName(), null, null);
        AnnotationComponent[] comps = anno.getComponents();
        AnnotationValueInfo[] values = new AnnotationValueInfo[comps.length];
        for (int x = 0; x < comps.length; x++) {
            values[x] = annotationComponentToInfo(comps[x]);
        }
        return new AnnotationInfo(null, AnnotationInfo.ANNOTATION_TYPE, name, values);
    }
    
    public static AnnotationValueInfo annotationComponentToInfo(AnnotationComponent comp) {
        int infoType = getElementValueType(comp.getValue());
        return new AnnotationValueInfo(null, infoType, comp.getName(), resolveElementValue(comp.getValue()));
    }
    
    public static int getElementValueType(ElementValue elemValue) {
        int infoType;
        if (elemValue instanceof ArrayElementValue) {
            infoType = AnnotationValueInfo.ANNOTATIONVALUE_ARRAY;
        } else if (elemValue instanceof PrimitiveElementValue) {
            infoType = AnnotationValueInfo.ANNOTATIONVALUE_STRING;
        } else if (elemValue instanceof ClassElementValue) {
            infoType = AnnotationValueInfo.ANNOTATIONVALUE_STRING;
        } else if (elemValue instanceof EnumElementValue) {
            infoType = AnnotationValueInfo.ANNOTATIONVALUE_STRING; // [PENDING]
        } else { // NestedElementValue
            infoType = AnnotationValueInfo.ANNOTATIONVALUE_ANNOTATION;
        }
        return infoType;
    }
    
    public static Object resolveElementValue(ElementValue elem) {
        if (elem instanceof ArrayElementValue) {
            ElementValue[] arrayValues = ((ArrayElementValue)elem).getValues();
            Object[] value = new Object[arrayValues.length];
            for (int x = 0; x < arrayValues.length; x++) {
                value[x] = resolveElementValue(arrayValues[x]);
            }
            return value;
        }
        if (elem instanceof PrimitiveElementValue) {
            return ((PrimitiveElementValue)elem).getValue().getValue().toString();
        }
        if (elem instanceof ClassElementValue) {
            return ((ClassElementValue)elem).getClassName().getExternalName();
        }
        if (elem instanceof EnumElementValue) {
            return ((EnumElementValue)elem).getEnumName(); // [PENDING]
        }
        return annotationToInfo(((NestedElementValue)elem).getNestedValue());
    }
        
    public static NameRef sigToNameRef(char[] signature, int[] i, NameRef parent) {
        StringBuffer name = new StringBuffer();
        for (; signature[i[0]] != ';' && signature[i[0]] != '<'; i[0]++) {
            char ch = signature[i[0]];
            if (ch=='/' || ch=='$') { 
                ch='.';
            }
            if (name.length() > 0 || ch != '.')
                name.append(ch);
        }
        
        if (signature[i[0]] == ';')
            return new NameRef(name.toString(), parent, null);
        
        i[0]++;
        ArrayList typeParams = new ArrayList(5);
        while(signature[i[0]] != '>') {
            char ch = signature[i[0]];
            if (ch == '*') {
                typeParams.add(new WildCardRef(false, null));
                i[0]++;
            } else if (ch == '+' || ch == '-') {
                i[0]++;
                TypeRef tr = sigToTypeRef(signature, i);
                typeParams.add(new WildCardRef(ch == '-', tr));
            } else {
                typeParams.add(sigToTypeRef(signature, i));
            }
        }
        i[0]++;
        TypeRef[] typeRefs = new TypeRef[typeParams.size()];
        typeParams.toArray(typeRefs);
        return new NameRef(name.toString(), parent, typeRefs);
    }
    
    public static TypeRef sigToTypeRef(char[] signature, int[] i) {
        char c = (char) signature[i[0]];
        switch (c) {
            case 'B':
                i[0]++;
                return PrimitiveTypeRef.BYTE;
            case 'C':
                i[0]++;
                return PrimitiveTypeRef.CHAR;
            case 'D':
                i[0]++;
                return PrimitiveTypeRef.DOUBLE;
            case 'F':
                i[0]++;
                return PrimitiveTypeRef.FLOAT;
            case 'I':
                i[0]++;
                return PrimitiveTypeRef.INT;
            case 'J':
                i[0]++;
                return PrimitiveTypeRef.LONG;
            case 'T':
                i[0]++;
                int pos;
                StringBuffer buf = new StringBuffer(40);
                for (pos = i[0]; (pos < signature.length) && (signature[pos] != ';'); pos++) {
                    char ch = signature[pos];
                    if (ch=='/' || ch=='$') ch='.'; 
                    buf.append(ch);
                }
                i[0] = pos + 1;     
                return new TypeParamRef(buf.toString());
            case 'L':
                i[0]++;
                NameRef nameRef = null;
                while (signature[i[0]] != ';') {
                    nameRef = sigToNameRef(signature, i, nameRef);
                }
                i[0]++;
                return nameRef;
            case 'S':
                i[0]++;
                return PrimitiveTypeRef.SHORT;
            case 'V':
                i[0]++;
                return PrimitiveTypeRef.VOID;
            case 'Z':
                i[0]++;
                return PrimitiveTypeRef.BOOLEAN;
            case '[':
                int dimCount = 0;
                while (signature[i[0]] == '[') {
                    dimCount++;
                    i[0]++;
                }
                return new ArrayRef((PrimitiveTypeRef)sigToTypeRef(signature, i), dimCount);
            case ')':
                i[0]++;
                return null;
            default:
                throw new IllegalArgumentException("Unknown signature: " + new String(signature) + ' ' + i[0]); // NOI18N
        }
    }
    
    public static TypeParamInfo[] parseTypeParams(char[] sig, int[] i) {
        Iterator iter;
        List infos = new ArrayList();
        if (sig[0] == '<') {
            i[0] = 1;
            while (sig[i[0]] != '>') {
                // identifier
                StringBuffer id = new StringBuffer();
                while (sig[i[0]] != ':') {
                    id.append(sig[i[0]]);
                    i[0]++;
                } // while
                if (sig[i[0] + 1] == ':') {
                    i[0]++;
                }
                ArrayList bounds = new ArrayList();
                while (sig[i[0]] == ':') {
                    i[0]++;
                    bounds.add(sigToTypeRef(sig, i));
                } // while
                TypeParamRef[] bds = new TypeParamRef[bounds.size()];
                iter = bounds.iterator();
                for (int x = 0; iter.hasNext(); x++) {
                    bds[x] = (TypeParamRef) iter.next();
                }
                infos.add(new TypeParamInfo(null, TypeParamInfo.TYPEPARAM_TYPE, id.toString(), bds));
            } // while
            i[0]++;
        } // if
        TypeParamInfo[] tpi = new TypeParamInfo[infos.size()];
        iter = infos.iterator();
        for (int x = 0; iter.hasNext(); x++) {
            tpi[x] = (TypeParamInfo)iter.next();
        }
        return tpi;
    }
    
    public static List getParamTypes (char[] signature, int[] pos) {
        List params = new ArrayList();
        pos[0]++;
        while (signature[pos[0]] != ')') {
            params.add(sigToTypeRef(signature, pos));
        }
        return params;
    }

    public ParameterInfo[] createParamsInfo (Method method, char[] signature, int[] pos, FileObject file) {
        List params = new ArrayList();
        Iterator paramsIter = method.getParameters().iterator();
        int notKnownNames = 0;
        while (true) {
            TypeRef parType = sigToTypeRef(signature, pos);
            if (parType != null) {
                AnnotationInfo[] parameterAnnons;
                String parameterName;
                if (paramsIter.hasNext()) {
                    Parameter par = (Parameter) paramsIter.next();
                    parameterName = par.getName();
                    parameterAnnons = createAnnotations(par.getAnnotations());
                    // We cannot get parameter's name from classfile (it is
                    // probably interface or abstract class, which does not have 
                    // parameter's name in compiled form even if it is compiled 
                    // with debug information. We will try to get it from
                    // javadoc later.
                    if (parameterName.length() == 0) {
                        notKnownNames++;
                    }
                }
                else {
                    parameterName = "";
                    parameterAnnons = null;
                }
                ParameterInfo info = new ParameterInfo (null, ParameterInfo.PARAMETER_TYPE,
                    parameterName, false, parType, false, parameterAnnons);
                params.add(info);
            } 
            else
                break;
        }
        ParameterInfo[] parInfos = (ParameterInfo[]) params.toArray(new ParameterInfo[params.size()]);
        if (method.isVarArgs()) {
            // [PENDING]
            parInfos[parInfos.length - 1] = new ParameterInfo(
                null, ParameterInfo.PARAMETER_TYPE, parInfos[parInfos.length - 1].name,
                false, parInfos[parInfos.length - 1].type, true, parInfos[parInfos.length - 1].annotations
            );
        }
        if (notKnownNames > 0) {
            if (!getNamesFromJavaDoc(method, parInfos, file)) {
                computeArtificalNames(parInfos);
            }
        }
        return parInfos;
    }

    private boolean getNamesFromJavaDoc(final Method method, final ParameterInfo[] parInfos, final FileObject file) {
        long startT = 0;
        if (JMManager.PERF_DEBUG) {
            startT = System.currentTimeMillis();
        }
        try {
            // Skip rt.jar file as we have src.zip on classpath, which
            // is enough for obtaining parameters' names.
            // todo #pf: should be probably beter to name all filesystem
            // which shouldn't be used for reading parameter names from
            // javadoc
            if (file.getFileSystem().getDisplayName().endsWith("rt.jar")) { // NOI18N
                if (JMManager.PERF_DEBUG) {
                    javadocTT += (System.currentTimeMillis() - startT);
                    if ((javadocTT / 100) > lastPrinted) {
                        System.err.println("DEBUG: createParamsInfo()#javadoc-names " + javadocTT + " ms.");
                        lastPrinted++;
                    }
                }
                return false;
            }
        } catch (FileStateInvalidException e) {
            // we do not care about it
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, e.getMessage());
            return false;
        }
        if (signature2parnames == null) {
            // fill the data for the class
            signature2parnames = fillParametersMap(file);
        }
        if (signature2parnames != null) {
            String patternStr = getSignatureKey(method.getName(), parInfos);
            String[] names = (String[]) signature2parnames.get(patternStr);
            if (names != null) {
                for (int i = 0; i < parInfos.length && i < names.length; i++) {
                    parInfos[i] = new ParameterInfo(
                        null,
                        ParameterInfo.PARAMETER_TYPE, 
                        // this constructor is being called in order to trim the 
                        // baggage, it will make a copy of string
                        new String(names[i]),
                        parInfos[i].isFinal,
                        parInfos[i].type,
                        parInfos[i].isVarArg,
                        parInfos[i].annotations
                    );
                }
            }
            if (JMManager.PERF_DEBUG) {
                long deltaT = System.currentTimeMillis() - startT;
                javadocTT += deltaT;
                System.err.println("PERF: createParamsInfo()#javadoc-names '" + method.getName() + "' took " + deltaT + " ms.");
                if ((javadocTT / 100) > lastPrinted) {
                    System.err.println("PERF: createParamsInfo()#javadoc-names " + javadocTT + " ms.");
                    lastPrinted++;
                }
            }
            return true;
        }
        return false;
    }

    private void computeArtificalNames(ParameterInfo[] parInfos) {
        Map names=new HashMap();
        final Integer zero=new Integer(0);
        int i;
        
        for (i=0;i<parInfos.length;i++) {
            TypeRef pType=parInfos[i].type;
            String name;
            Integer count;
            
            if (pType==null) {
                pType=NameRef.java_lang_Object;
            }
            name=getName(pType);
            count=(Integer)names.get(name);
            if (count!=null) {
                names.put(name,new Integer(count.intValue()+1));
                name=name.concat(count.toString());
            } else {
                names.put(name,zero);
            }
            parInfos[i] = new ParameterInfo(
                null,
                ParameterInfo.PARAMETER_TYPE, 
                name,
                parInfos[i].isFinal,
                parInfos[i].type,
                parInfos[i].isVarArg,
                parInfos[i].annotations
            );
        }
    }

    private static String getName(final TypeRef pType) {
        if (pType instanceof ArrayRef) {
            return getName(((ArrayRef)pType).parent);
        } else if (pType instanceof TypeParamRef) {
            int index;
            String name=pType.getName();
            
            index=name.lastIndexOf('.');
            if (index!=-1) {
                name=name.substring(index+1);
            }
            String loweredName = name.substring(0,1).toLowerCase().concat(name.substring(1));
            return Utilities.isJavaIdentifier(loweredName) ? loweredName : 'a' + name;
        } else if (pType instanceof PrimitiveTypeRef) {
            return pType.getName().substring(0,1);
        } else if (pType instanceof WildCardRef) { // just in case
            return getName(((WildCardRef)pType).bound);
        } else {
            throw new IllegalArgumentException("Uknow type "+pType.getClass());
        }
    }

    /**
     * Creates signature string for locating method header in map containing
     * signature as a key and array of parameters' names. (i.e. result
     * represents key in <tt>signature2parnames</tt> map.)
     * 
     * @param   methodName  method name which is looked for
     * @param   parInfo     array of parameters of the method
     * @return  string representation of signature 
     */
    private static String getSignatureKey(String methodName, ParameterInfo[] parInfo) {
        StringBuffer result = new StringBuffer(); // NOI18N
        result.append(methodName).append("("); // NOI18N
        for (int i = 0; i < parInfo.length; i++) {
            String typeName = parInfo[i].type.getName();
            result.append(typeName);
            if ((i+1) < parInfo.length) {
                result.append(", "); // NOI18N
            } else {
                result.append(")"); // NOI18N
            }
        }
        return result.toString();
    }
    
    /**
     * Method signatures in string representation are used as keys. For
     * every key, array of ordered parameters' names are in value.
     * Example of item:
     * Key: <tt>gluTessBeginPolygon(javax.media.opengl.glu.GLUtessellator, java.lang.Object)</tt>
     * Value: <tt>String[tessellator,data]</tt>
     *
     * This data are used when parameters' names are not available in class
     * or source files.
     */
    private Map/*<String, String[]>*/ signature2parnames = null;
    
    /**
     * Simple pattern for finding methods in javadoc.
     */
    private static Pattern pattern = Pattern.compile("\\<A NAME\\=\"([^\\)\"]+\\))"); // NOI18N
    
    /**
     * Searches for parameters' names in defined section. It starts from
     * <tt>offset</tt>, going through the <tt>htmlContent</tt> lines till 
     * </pre> tag found.
     *
     * @param   offset       position in string where to start
     * @param   htmlContent  content of html file
     * @return  array of parameters' names (ordered)
     */
    private static String[] obtainNames(int offset, String htmlContent) {
        // get section for the method
        int startOffset = htmlContent.indexOf("<PRE>", offset) + 5;
        if (startOffset < 5) {
            return new String[0];
        }
        int endOffset = htmlContent.indexOf(")", startOffset) + 1;
        if (endOffset <= 0) {
            return new String[0];
        }
        String preSection = htmlContent.substring(startOffset, endOffset);
        startOffset = 0; 
        endOffset = preSection.length()-1;
        startOffset = preSection.indexOf("&nbsp;", 0); // NOI18N
        List/*<String>*/ names = new ArrayList(3);
        int mark = startOffset;
        while (startOffset != -1 && startOffset < endOffset) {
            mark = startOffset + 6;
            startOffset = preSection.indexOf(",", mark); // NOI18N
            if (startOffset == -1 || startOffset >= endOffset)
                startOffset = endOffset; // NOI18N
            names.add(preSection.substring(mark, startOffset));
            startOffset = preSection.indexOf("&nbsp;", startOffset); // NOI18N
        }
        return (String[]) names.toArray(new String[0]);
    }
    
    /**
     * Fetch the entire contents of a text file, and return it in a
     * <tt>String</tt>
     *
     * @param aFile is a file which already exists and can be read.
     * @exception IOException            file cannot be read
     * @exception FileNotFoundException  file is not available
     */
    static public String getContents(FileObject aFile) 
        throws FileNotFoundException, IOException
    {
        int size = (int) aFile.getSize();
        byte[] b = new byte[size];
        InputStream is = null;
        try {
            is = aFile.getInputStream();
            is.read(b);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ex) {
                // ignore, too bad
            }
        }
        return new String(b);
    }
    
    /**
     * Looks for the javadoc file related to provided file. Consider that
     * you read file named 'jar:file:/tmp/library.jar!/foo/Foo.class'. Then
     * method will try to locate file foo/Foo.html in related javadoc entry. If
     * the entry does not exist or file does not exist in the entry, it returns 
     * null value. Otherwise it returns found html file.
     *
     * @param   file   look for related html file for this parameter
     * @return  html file object, null if javadoc is not available for parameter
    */
    private static FileObject getJavadocFile(FileObject file) {
        ClassPath cp = ClassPath.getClassPath(file, ClassPath.EXECUTE);
        if (cp == null) {
            // We does not have execute classpath for provided parameter,
            // we will not try to find javadoc file for it.
            return null;
        }
        Result res;
        try {
            FileObject cpRoot = cp.findOwnerRoot(file);
            if (cpRoot == null) return null;
            res = JavadocForBinaryQuery.findJavadoc(cpRoot.getURL());
        } catch (FileStateInvalidException e) {
            // we don't care about it, consider html file was not found.
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, e.getMessage());
            return null;
        }
        URL[] url = res.getRoots();
        if (url.length > 0) {
            for (int i = 0; i < url.length; i++) {
                FileObject rootDoc = URLMapper.findFileObject(url[i]);
                if (rootDoc != null) {
                    // we finally found javadoc root, we will try to find related
                    // html file in it.
                    String html = cp.getResourceName(file);
                    html = html.substring(0, html.length()-5).replace('$', '.') + "html"; // NOI18N
                    return rootDoc.getFileObject(html);
                }
            }
        }
        // unfortunately there is not related html file for the class file.
        return null;
    }
    
    /**
     * Fills the mapping for class file. It uses string representation 
     * of signature as a key and list of strings as a parameters names. 
     * (ordered) If the *.class file does not have related html file
     * in libary definition, return empty map. Otherwise, it fills
     * the map with data.
     *
     * @param   file  file representing the *.class file
     * @return  map of methods -> parameters names, null if there is no javadoc file
     */
    public static Map fillParametersMap(FileObject file) {
        long startT = 0;
        if (JMManager.PERF_DEBUG) {
            startT = System.currentTimeMillis();
        }
        Map/*<String, String[]>*/ pars = null;
        try {
            FileObject htmlFile = getJavadocFile(file);
            if (htmlFile == null)
                return null;
            pars = new HashMap();
            String htmlContent = getContents(htmlFile);
            for (Matcher m = pattern.matcher(htmlContent); m.find(); ) {
                int start = m.start(1);
                int end = m.end(1);
                String[] names = obtainNames(start, htmlContent);
                pars.put(htmlContent.substring(start, end), names);
            }
        } catch (FileNotFoundException ex) {
            return null;
        } catch (IOException ex) {
            return null;
        }
        if (JMManager.PERF_DEBUG) {
            long deltaT = System.currentTimeMillis() - startT;
            System.err.println("Map for file '" + file.getName() + "' was filled in " + deltaT + " ms.");
        }
        return pars;
    }
}