/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.parser;

import java.lang.ref.SoftReference;
import java.util.*;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.mdr.handlers.BaseObjectHandler;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.modules.javacore.ExclusiveMutex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaClassClassImpl;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaClassImpl;

/**
 *
 * @author  Tomas Hurka
 */
public class Scope implements ScopeMember {
    
    private Scope parentScope;
    private Map positiveCache;
    private Set negativeCache;
    private ArrayList members;
    
    /** Creates a new instance of Scope */
    Scope(Scope parent) {
        parentScope=parent;
        positiveCache=new HashMap();
        negativeCache=new HashSet();
        members=new ArrayList();
    }
    
    public Object lookup(final Object key) {
        Object val=positiveCache.get(key);
        if (val!=null)
            return val;
        if (!negativeCache.contains(key))
            val=lookupMembers(key);
        if (val==null && parentScope!=null)
            val=parentScope.lookup(key);
        return val;
    }
    
    void addMember(final ScopeMember member) {
        positiveCache.clear();
        negativeCache.clear();
        members.add(member);
    }
    
    public boolean equals(Object obj) {
        if (obj!=null && obj instanceof Scope) {
            Scope sc=(Scope)obj;
            if (!members.equals(sc.members))
                return false;
            if (parentScope==null)
                return parentScope==sc.parentScope;
            return parentScope.equals(sc.parentScope);
        }
        return false;
    }
    
    protected Object clone() {
        Scope cloned=new Scope(parentScope);
        
        cloned.members=(ArrayList)members.clone();
        return cloned;
    }
    
    private Object lookupMembers(final Object key) {
        Iterator it=members.iterator();
        Object value=null;
        
        while(it.hasNext()) {
            ScopeMember m=(ScopeMember)it.next();
            Object val=m.lookup(key);
            
            if (val==null)
                continue;
            if (value==null) {
                value=val;
                continue;
            }
            if (!value.equals(val))
                // ambiguos reference
                JMManager.getLog().log("Ambiguos reference "+key+":"+value+":"+val); // NOI18N
        }
        if (value!=null)
            positiveCache.put(key,value);
        else
            negativeCache.add(key);
        return value;
    }
    
    static Scope createMemberTypeScope(ClassDefinition jcls,MDRParser sc) {
        if (jcls instanceof ParameterizedType)
            jcls=((ParameterizedType)jcls).getDefinition();
        if (jcls instanceof UnresolvedClass)
            return new Scope(null);
        MOFID id=((BaseObjectHandler) jcls)._getMofId();
        Map cache=getCacheFor(Scope.class.getName()+"createMemberTypeScope"); // NOI18N
        Scope scope=(Scope)cache.get(id);

        if (scope==null) {
            cache.put(id,new Scope(null));
            scope=constructMemberTypeScope(jcls, sc);
            cache.put(id,scope);
        }
        return scope;
    }
    
    static Scope constructMemberTypeScope(ClassDefinition jcls,MDRParser sc) {
        List superScopes=new ArrayList(2);
        Scope superScope=null;
        Scope memberScope;
        JavaClass superJavaClass=getSuperClass(jcls,sc);
        Collection interfaces=getInterfaces(jcls,sc);
        Iterator ifaceIt=interfaces.iterator();
        
        if (superJavaClass!=null && !(superJavaClass instanceof UnresolvedClass))
            superScopes.add(createMemberTypeScope(superJavaClass,sc));
        while (ifaceIt.hasNext()) {
            JavaClass ifaceClass=(JavaClass)ifaceIt.next();
            
            if (ifaceClass!=null && !(ifaceClass instanceof UnresolvedClass))
                superScopes.add(createMemberTypeScope(ifaceClass,sc));
        }
        if (!superScopes.isEmpty()) {
            Iterator scopeIt=superScopes.iterator();

            superScope=new Scope(null);
            while(scopeIt.hasNext())
                superScope.addMember((ScopeMember)scopeIt.next());
        }
        memberScope=new Scope(superScope);
        if (jcls instanceof JavaClass)
            memberScope.addMember(new MemberClassScope((JavaClass)jcls));
        return memberScope;
    }
    
    static Scope createTypeScope(String jpck, ClassPath classPath, ElementInfo[] imports) {
        Scope packageImp=new Scope(null);
        Scope currentPackage=new Scope(packageImp);
        Scope singleImp=new Scope(currentPackage);
        Scope memberTypes=new Scope(singleImp);
        Iterator it;
        Resource rsc;
        
        packageImp.addMember(new PackageImpScope("java.lang",classPath)); // NOI18N
        currentPackage.addMember(new PackageImpScope(jpck,classPath));
        if (imports != null) {
            for (int i = 0; i < imports.length; i++) {
                ElementInfo importInfo=imports[i];
                String name=importInfo.name;
                
                if (importInfo.infoType==ElementInfo.IMPORT_ON_DEMAND_TYPE)
                    packageImp.addMember(new PackageImpScope(name,classPath));
                else
                    singleImp.addMember(new SingleImpScope(name));
            }
        }
        return singleImp;
    }

    static Scope createStaticImpScope(Resource rsc) {
        Scope packageImp=new Scope(null);
        Scope singleImp=new Scope(packageImp);
        Scope memberTypes=new Scope(singleImp);
        Object[] imports=rsc.getImports().toArray();
        JavaClassClassImpl jclsClass=(JavaClassClassImpl)((JavaModelPackage)rsc.refImmediatePackage()).getJavaClass();
        
        for(int i=0;i<imports.length;i++) {
            Import imp=(Import)imports[i];

            if (imp.isStatic()) {
                String name=imp.getName();
                
                if (imp.isOnDemand()) {
                    JavaClass javaClass =(JavaClass)jclsClass.resolveClass(name, true);
                    
                    if (javaClass!=null)
                        packageImp.addMember(createStaticImportScope(javaClass,null));
                } else {
                    int lastDot=name.lastIndexOf('.');
                    
                    if (lastDot!=-1) {
                        String className=name.substring(0,lastDot);
                        JavaClass javaClass =(JavaClass)jclsClass.resolveClass(className, true);
                        
                        if (javaClass!=null)
                            singleImp.addMember(createStaticImportScope(javaClass,name.substring(lastDot+1)));
                    }
                }
            }
        }
        return singleImp;
    }
    
    static MethodScope createMethodScope(ClassDefinition jcls) {
        if (jcls instanceof UnresolvedClass)
            return new MethodScope(null);
        MOFID id=((BaseObjectHandler) jcls)._getMofId();
        Map cache=getCacheFor(Scope.class.getName()+"createMethodScope"); // NOI18N
        MethodScope scope=(MethodScope)cache.get(id);

        if (scope==null) {
            scope=createMethodScopeImpl(jcls);
            cache.put(id,scope);
        }
        return scope;
    }

    private static MethodScope createMethodScopeImpl(ClassDefinition jcls) {
        JavaClass superJavaClass=jcls.getSuperClass();
        Iterator ifaceIt=jcls.getInterfaces().iterator();
        List superScopes=new ArrayList(2);
        Iterator scopeIt;
        MethodScope superScope;
        MethodScope memberScope;
        
        if (superJavaClass!=null)
            superScopes.add(createMethodScope(superJavaClass));
        while (ifaceIt.hasNext()) {
            JavaClass ifaceClass=(JavaClass)ifaceIt.next();
            
            if (ifaceClass!=null)
                superScopes.add(createMethodScope(ifaceClass));
        }
        superScope=new MethodScope(null);
        scopeIt=superScopes.iterator();
        while(scopeIt.hasNext())
            superScope.addMember((ScopeMember)scopeIt.next());
        memberScope=new MethodScope(superScope);
        memberScope.addMember(new MethodScopeMember(jcls));
        return memberScope;
    }

    static Scope createStaticImportScope(JavaClass jcls,String name) {
        if (jcls instanceof UnresolvedClass)
            return new Scope(null);
        return createStaticImportScopeImpl(jcls,name);
    }

    private static Scope createStaticImportScopeImpl(JavaClass jcls,String name) {
        JavaClass superJavaClass=jcls.getSuperClass();
        Iterator ifaceIt=jcls.getInterfaces().iterator();
        List superScopes=new ArrayList(2);
        Iterator scopeIt;
        Scope superScope;
        Scope memberScope;
        
        if (superJavaClass!=null)
            superScopes.add(createStaticImportScope(superJavaClass,name));
        while (ifaceIt.hasNext()) {
            JavaClass ifaceClass=(JavaClass)ifaceIt.next();
            
            if (ifaceClass!=null)
                superScopes.add(createStaticImportScope(ifaceClass,name));
        }
        superScope=new Scope(null);
        scopeIt=superScopes.iterator();
        while(scopeIt.hasNext())
            superScope.addMember((ScopeMember)scopeIt.next());
        memberScope=new Scope(superScope);
        memberScope.addMember(new StaticImportScope(jcls,name));
        return memberScope;
    }
    
    static Scope createFieldScope(ClassDefinition jcls) {
        if (jcls instanceof UnresolvedClass)
            return new Scope(null);
        MOFID id=((BaseObjectHandler) jcls)._getMofId();
        Map cache=getCacheFor(Scope.class.getName()+"createFieldScope"); // NOI18N
        Scope scope=(Scope)cache.get(id);

        if (scope==null) {
            scope=createFieldScopeImpl(jcls);
            cache.put(id,scope);
        }
        return scope;
    }
    
    private static Scope createFieldScopeImpl(ClassDefinition jcls) {
        JavaClass superJavaClass=jcls.getSuperClass();
        Iterator ifaceIt=jcls.getInterfaces().iterator();
        List superScopes=new ArrayList(2);
        Iterator scopeIt;
        Scope superScope;
        Scope memberScope;
        
        if (superJavaClass!=null)
            superScopes.add(createFieldScope(superJavaClass));
        while (ifaceIt.hasNext()) {
            JavaClass ifaceClass=(JavaClass)ifaceIt.next();
            
            if (ifaceClass!=null)
                superScopes.add(createFieldScope(ifaceClass));
        }
        superScope=new Scope(null);
        scopeIt=superScopes.iterator();
        while(scopeIt.hasNext())
            superScope.addMember((ScopeMember)scopeIt.next());
        memberScope=new Scope(superScope);
        memberScope.addMember(new MemberFieldScope(jcls));
        return memberScope;
    }

    private static Map getCacheFor(String cacheId) {
        ExclusiveMutex mutex=JMManager.getTransactionMutex();
        Map parserCache=mutex.getParserCache();
        
        SoftReference ref = (SoftReference) parserCache.get(cacheId);
        Map cache = ref == null ? null : (Map) ref.get();

        
        if (cache==null) {
            cache=new HashMap();
            parserCache.put(cacheId,new SoftReference(cache));
        }
        return cache;
    }
    
    private static JavaClass getSuperClass(ClassDefinition jcls,MDRParser sc) {
        if (sc!=null)
            return sc.getSuperClass(jcls);
        return jcls.getSuperClass();
    }
    
    private static Collection getInterfaces(ClassDefinition jcls,MDRParser sc) {
        if (sc!=null)
            return sc.getInterfaces(jcls);
        return jcls.getInterfaces();
    }

    public static Scope createTypeScope(Resource rsc,ClassPath classPath) {
        String jpck=rsc.getPackageName();
        Collection importCol=rsc.getImports();
        Iterator it=importCol.iterator();
        List imports=new ArrayList();
        
        while(it.hasNext()) {
            Import imp=(Import)it.next();
            if (!imp.isStatic()) {
                int infoType=imp.isOnDemand()?ElementInfo.IMPORT_ON_DEMAND_TYPE:ElementInfo.SINGLE_IMPORT_TYPE;

                imports.add(new ElementInfo(null, infoType, imp.getName()));
            }
        }
        return createTypeScope(jpck,classPath,(ElementInfo[])imports.toArray(new ElementInfo[imports.size()]));
    }
    
    
    public static Scope computeTypeScope(Element element) {
        Scope classScope;
        
        if (element instanceof JavaClass) {
            JavaClass javaClass=((JavaClass)element);
            ClassDefinition decl=javaClass.getDeclaringClass();
            Iterator typeParIt;
            Scope parent;
            
            if (element instanceof JavaClassImpl && ((JavaClassImpl)element).isTransient() && decl==null) { // local class
                parent=computeTypeScope(JavaModelUtil.getDeclaringFeature((Element)element.refImmediateComposite()));
            } else if (decl==null) {   // jcls is top-level class
                ClassPath cp=JavaMetamodel.getManager().getClassPath();
                
                parent=Scope.createTypeScope(element.getResource(),cp);
            } else {
                parent=computeTypeScope(decl);
            }
            classScope=new Scope(parent);
            classScope.addMember(Scope.createMemberTypeScope(javaClass,null));
            typeParIt=javaClass.getTypeParameters().iterator();
            while(typeParIt.hasNext()) {
                TypeParameter tp=(TypeParameter)typeParIt.next();
                
                classScope.addMember(new TypeParamScope(tp));
            }
        } else if (element instanceof ClassMember) {
            ClassMember cm=(ClassMember)element;
            
            classScope=computeTypeScope(cm.getDeclaringClass());
            if (cm instanceof CallableFeature) {
                CallableFeature cf=(CallableFeature)cm;
                Iterator typeParIt;
                
                classScope=new Scope(classScope);
                typeParIt=cf.getTypeParameters().iterator();
                while(typeParIt.hasNext()) {
                    TypeParameter tp=(TypeParameter)typeParIt.next();

                    classScope.addMember(new TypeParamScope(tp));
                }   
            }
        } else if (element instanceof ClassDefinition) {   // anonymous class
            Scope parent=computeTypeScope(JavaModelUtil.getDeclaringFeature(element));
            
            classScope=new Scope(parent);
            classScope.addMember(Scope.createMemberTypeScope((ClassDefinition)element,null));            
        } else { // inside block
            Feature feature=JavaModelUtil.getDeclaringFeature(element);
                
            assert feature!=null:"Invalid element "+element;
            classScope=computeTypeScope(feature);
        }
        return classScope;
    }
}
