/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.javacore.scanning;

import java.io.*;
import java.util.*;
import javax.jmi.reflect.RefObject;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.EnumConstant;
import org.netbeans.jmi.javamodel.JavaEnum;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.classfile.Access;
import org.netbeans.modules.classfile.ClassFile;
import org.netbeans.modules.classfile.ClassName;
import org.netbeans.modules.classfile.InvalidClassFormatException;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.*;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;


/**
 *
 * @author  Tomas Hurka
 */
public class ClassUpdater {    
    private Map fileObjectToClassFile;
    private Map nameToResource;
    private Map resourceToSuperCodes;    
    private Map unmodifiedResources;
    private Map resourceToClasses;
    private ResourceClassImpl resProxy;
    private JavaClassClassImpl clsProxy;
    private JavaEnumClassImpl enumProxy;
    private AnnotationTypeClassImpl annoProxy;
    private ClassIndex classIndex;
    private ResourceImpl updatedResource;
    private long indexTimestamp;
    
    /** Creates a new instance of ClassUpdater */
    public ClassUpdater(JavaModelPackage mofModel) {
        clsProxy = (JavaClassClassImpl) mofModel.getJavaClass();
        enumProxy = (JavaEnumClassImpl) mofModel.getJavaEnum();
        resProxy = (ResourceClassImpl) mofModel.getResource();
        annoProxy = (AnnotationTypeClassImpl) mofModel.getAnnotationType();
        classIndex=ClassIndex.getIndex(mofModel);
    }
 
    
    public static void updateIndex(ResourceImpl resource,FileObject fobj) {
        JavaMetamodel.getDefaultRepository().beginTrans(false);
        try {
            JavaModelPackage pck=(JavaModelPackage)resource.refOutermostPackage();
            Map cls=new HashMap();
            String fname=fobj.getNameExt();
            String prefix=fobj.getName().concat("$");
            String ext=fobj.getExt();
            FileObject[] children=fobj.getParent().getChildren();
            
            cls.put(fname,new FObjectInfo(fobj));
            for (int i=0;i<children.length;i++) {
                FileObject ch=children[i];
                
                if (ext.equals(ch.getExt()) && ch.getName().startsWith(prefix)) {
                    cls.put(ch.getNameExt(),new FObjectInfo(ch));   
                    
                }
            }
            ClassUpdater instance=new ClassUpdater(pck);
            instance.updatedResource=resource;
            instance.updateResources(Collections.EMPTY_MAP,cls,true);
        } finally {
            JavaMetamodel.getDefaultRepository().endTrans();
        }
    }
    
    public Collection updateResources(Map sources, Map classes) {
        return updateResources(sources, classes, false);
    }
    
    public Collection updateResources(Map sources, Map classes, boolean directUpdate)
    {
        Iterator iter, iter2;
        indexTimestamp=classIndex.getTimestamp();
        unmodifiedResources = new HashMap();
        fileObjectToClassFile = new HashMap();
        nameToResource = new HashMap();
        resourceToClasses = new HashMap();
        resourceToSuperCodes = new HashMap();
        iter = classes.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry entry = (Map.Entry) iter.next();
            String name = (String) entry.getKey();
            
            int index = name.indexOf('$');
            String resName = (index > -1) ? name.substring(0, index) + ".class" : name;   //NOI18N

            // ignore classes with correcponding java files in the same directory
            String srcName = resName.substring(0, resName.length() - ".class".length()) + ".java"; // NOI18N
            if (sources.containsKey(srcName)) continue;
            // ------------------------
            
            FileInfo file = (FileInfo) entry.getValue();
            ClassFile clsFile = getClassFile (file);
            if ((clsFile == null) || isAnonymous(clsFile)) {
                continue;
            }            
            Set superCodes = getSuperCodes(resName, classes);
            if (superCodes == null) {
                continue;
            }
            addFileToMap (resName, clsFile);
            addSuperCodes(superCodes, clsFile);
        } // while
        
        iter = resourceToSuperCodes.entrySet().iterator();
        for (int x = 0; iter.hasNext(); x++) {
            Map.Entry entry = (Map.Entry) iter.next();
            Set codes = (Set) entry.getValue();
            int size = codes.size();
            int[] hc = new int[size];
            iter2 = codes.iterator();
            for (int y = 0; y < size; y++) {
                hc[y] = ((Integer) iter2.next()).intValue();
            }
            ResourceImpl resource = (ResourceImpl) entry.getKey();
            classIndex.setIdentifiers(resource,hc);
        } // for

        iter = nameToResource.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry entry = (Map.Entry) iter.next();
            String resName = (String) entry.getKey();
            ResourceImpl res = (ResourceImpl) entry.getValue();
            updateClasses (resName, res, !directUpdate && !res.isInitialized());
        }
        Collection resources=new ArrayList(resourceToSuperCodes.keySet());
        resources.addAll(unmodifiedResources.values());
        return resources;
    }
    
    public void addSuperCodes(Set codes, ClassFile classFile) {
        ClassName superName = classFile.getSuperClass();
        if (superName != null) {
            codes.add (nameToHashCode(superName));
        }
        Iterator iter = classFile.getInterfaces().iterator();            
        while (iter.hasNext()) {
            codes.add (nameToHashCode((ClassName) iter.next()));
        }
    }
    
    private Set getSuperCodes(String name, Map classes) {
        if (unmodifiedResources.containsKey(name)) {
            return null;
        }
        Resource res = (Resource) nameToResource.get (name);
        if (res == null) {
            FileInfo file = (FileInfo) classes.get (name);
            if (file == null) {
                // ignore this class [PENDING]
                unmodifiedResources.put(name,null);
                JMManager.getLog().log("ClassUpdater, cannot find file object: " + name); // NOI18N
                return null;
            }
            if (updatedResource!=null) {
                assert updatedResource.getName().endsWith(name): updatedResource.getName()+" "+name; // NOI18N
                res = updatedResource;
            } else {
                res = resProxy.resolveResource(file.getPath(), true, false);
            }
            long timestamp = file.lastModified();
            if (res.getTimestamp() != timestamp || indexTimestamp<timestamp) {
                res.setTimestamp(timestamp);
                nameToResource.put (name, res);
                Set set = new HashSet();
                resourceToSuperCodes.put (res, set);
                return set;
            } else {
                unmodifiedResources.put(name,res);
                return null;
            }
        }
        return (Set) resourceToSuperCodes.get (res);
    }
    
    public String getSimpleName (ClassName clsName) {
        String name = clsName.getSimpleName();
        int index = name.lastIndexOf ('.');
        if (index > -1) {
            name = name.substring (index + 1);
        }
        return name;
    }
    
    public Integer nameToHashCode (ClassName clsName) {
        return new Integer (getSimpleName (clsName).hashCode());
    }        

    public boolean isAnonymous (ClassFile clsFile) {
        String name = clsFile.getName().getSimpleName();
        int index = name.lastIndexOf ('.') + 1;
        if (name.length() == index) {
            JMManager.getLog().log("ClassUpdater, class name ends with a dot: " + name); // NOI18N
        }
        return (index > 0) && (name.length() > index) && Character.isDigit(name.charAt (index));        
    }
    
    public ClassFile getClassFile (FileInfo file) {
        ClassFile clsFile = (ClassFile) fileObjectToClassFile.get (file);
        if (clsFile == null) {
            try {
                InputStream stream=new BufferedInputStream(file.getInputStream());
                try {
                    clsFile = new ClassFile (stream, false);
                    fileObjectToClassFile.put (file, clsFile);
                } finally {
                    stream.close();
                }
	    } catch (InvalidClassFormatException ex) {
                ErrorManager errmgr = ErrorManager.getDefault();
		errmgr.log(ErrorManager.INFORMATIONAL, 
			   "invalid class file format: " + file.getPath()); // NOI18N
            } catch (IOException ex) {
                ErrorManager errmgr = ErrorManager.getDefault();
                errmgr.annotate(ex, ErrorManager.EXCEPTION, file.getName(), null, null, null);
                errmgr.notify(ErrorManager.INFORMATIONAL, ex);
            }    
        } // if
        return clsFile;    
    }

    public void addFileToMap (String resName, ClassFile clsFile) {
        Map map = (Map) resourceToClasses.get (resName);
        if (map == null) {
            map = new HashMap ();
            resourceToClasses.put (resName, map);
        }
        map.put (clsFile.getName().getSimpleName(), clsFile);
    }
    
    public void updateClasses (String resName, ResourceImpl resource, boolean removeFeatures) {
        Map map = (Map) resourceToClasses.get (resName);
        Collection javaClasses = resource.getNakedClassifiers();
        if (javaClasses.size() > 0) {
            if (!newClassesNeeded(resource, map, removeFeatures)) {
                map.clear();
                return;
            }
        }
        Map result = new HashMap ();
        Iterator classIt=javaClasses.iterator();
        
        while(classIt.hasNext()) {
            RefObject oldJcls=(RefObject)classIt.next();            
            classIt.remove();
            oldJcls.refDelete();
        }
        while (map.size () > 0) {
            Map.Entry entry = (Map.Entry) map.entrySet().iterator().next();
            ClassFile clsFile = (ClassFile) entry.getValue();
            createJavaClass (resource, javaClasses, clsFile, map, result);
        } // while
    }
    
    private boolean newClassesNeeded(ResourceImpl resource, Map classesMap, boolean removeFeatures) {
        Collection javaClasses = resource.getNakedClassifiers();
        Map nameToClass = new HashMap();
        for (Iterator iter = javaClasses.iterator(); iter.hasNext();) {
            JavaClassImpl jc = (JavaClassImpl) iter.next();
            if (!isInIndex(jc, resource)) {
                return true;
            }
            nameToClass.put(jc.getName(), jc);
            collectInnerClasses(nameToClass, resource, jc);
        }
        if (classesMap.size() != nameToClass.size()) {
            return true;
        }
        
        for (Iterator iter = classesMap.entrySet().iterator(); iter.hasNext();) {
            Map.Entry entry = (Map.Entry) iter.next();
            ClassFile clsFile = (ClassFile) entry.getValue();
            ClassName clsName = clsFile.getName();
            String fullName = clsName.getExternalName();
            JavaClassImpl jc = (JavaClassImpl)nameToClass.get(fullName);
            if (jc == null) {
                return true;
            }
            int access = (clsFile.getAccess() & ~Access.SYNCHRONIZED);
            if (access != jc.getSourceModifiers()) {
                return true;
            }
            int declType = jc instanceof JavaEnum ? 1 : (jc instanceof AnnotationType ? 2 : 0);
            boolean isEnum = clsFile.isEnum();
            boolean isAnnotation = clsFile.isAnnotation();
            if ((declType == 1 && !isEnum) || (declType != 1 && isEnum) ||
                (declType == 2 && !isAnnotation) || (declType != 2 && isAnnotation)) {
                return true;
            }
        }
        if (removeFeatures) {
            for (Iterator iter = nameToClass.values().iterator(); iter.hasNext();) {
                removePersisted((JavaClassImpl)iter.next());
            }
        }
        return false;
    }

    private boolean isInIndex(JavaClassImpl jc, Resource res) {
        Iterator innerListIt = classIndex.getClassesByFqn(jc.getName()).iterator();
        while (innerListIt.hasNext()) {
            JavaClassImpl innerJcls = (JavaClassImpl)innerListIt.next();
            if (res.equals(innerJcls.getResource())) {
                return true;
            }
        }
        return false;
    }
    
    private void collectInnerClasses(Map nameToClass, Resource res, JavaClassImpl jc) {
        Iterator innerListIt = classIndex.getClassesByFQNPrefix(jc.getName().concat(".")).iterator(); // NOI18N
        while (innerListIt.hasNext()) {
            JavaClassImpl innerJcls = (JavaClassImpl)innerListIt.next();
            if (res.equals(innerJcls.getResource())) {
                nameToClass.put(innerJcls.getName(), innerJcls);
                collectInnerClasses(nameToClass, res, innerJcls);
            }
        }
    }

    static void removePersisted(JavaClassImpl jcls) {
        boolean changes = jcls.disableChanges;
        jcls.disableChanges = true;
        try {

            Collection fs = jcls.getPersistentContents();
            if (!fs.isEmpty()) {
                int size=fs.size();
                ClassMember features[]=(ClassMember[])fs.toArray(new ClassMember[size]);
                for (int i=0;i<size;i++) {
                    ClassMember f=features[i];
                    fs.remove(f);
                    if (!(f instanceof JavaClassImpl))
                        f.refDelete();
                }
            }
            
            fs = jcls.getPersistentTypeParameters();
            if (!fs.isEmpty()) {
                int size=fs.size();
                RefObject paramTypes[]=(RefObject[])fs.toArray(new RefObject[size]);
                for (int i=0;i<size;i++) {
                    RefObject tp=paramTypes[i];
                    fs.remove(tp);
                    tp.refDelete();
                }
            }
            
            fs = jcls.getPersistentAnnotations();
            if (!fs.isEmpty()) {
                int size=fs.size();
                RefObject annos[]=(RefObject[])fs.toArray(new RefObject[size]);
                for (int i=0;i<size;i++) {
                    RefObject anno=annos[i];
                    fs.remove(anno);
                    anno.refDelete();
                }
            }

            if (jcls instanceof JavaEnumImpl) {
                Collection ec = ((JavaEnumImpl) jcls).getPersistentConstants();
                if (!ec.isEmpty()) {
                    int cSize = ec.size();
                    EnumConstant constants[] = (EnumConstant[]) ec.toArray(new EnumConstant[cSize]);
                    for (int i = 0; i < cSize; i++) {
                        EnumConstant c = constants[i];
                        ec.remove(c);
                        c.refDelete();
                    }
                }
            }
            jcls.setPersisted(false);
        } finally {
            jcls.disableChanges = changes;
        }
    }
    
    private JavaClassImpl createJavaClass (ResourceImpl resource, Collection resClassifiers, ClassFile clsFile, Map map, Map createdClasses) {
        ClassName clsName = clsFile.getName();
        String fullName = clsName.getExternalName();
        String simpleName = clsName.getSimpleName();
        int index = simpleName.lastIndexOf('.');
        JavaClassImpl outer = null;
        if (index > -1) {
            // obtain outerclass
            String outerName = simpleName.substring (0, index);
            outer = (JavaClassImpl) createdClasses.get (outerName);
            if (outer == null) {
                ClassFile outerClassFile = (ClassFile) map.get (outerName);
                if (outerClassFile == null) {
                    map.remove (simpleName);                    
                    return null;
                }
                outer = createJavaClass (resource, resClassifiers, outerClassFile, map, createdClasses);
            }
        }        
        String pkgName = clsName.getPackage(); // [PENDING] ?? should be obtained only once (globally)        
        resource._setPackageName(pkgName);
        int access = (clsFile.getAccess() & ~Access.SYNCHRONIZED) | (clsFile.isDeprecated() ? FeatureImpl.DEPRECATED : 0);
        JavaClassImpl jc;
        if (clsFile.isAnnotation()) {
            jc = (JavaClassImpl) annoProxy.create(fullName, access, false);
        } else if (clsFile.isEnum()) {
            jc = (JavaClassImpl) enumProxy.create(fullName, access, false);
        } else {
            jc = (JavaClassImpl) clsProxy.create(fullName, access, null, null, false);
        }
        // should not add to class to the index - > it is added in the create method
        //classIndex.addClass(jc, fullName, getSimpleName(clsName));
        if (outer != null) {
            if (outer.isPersisted()) {
                outer.getPersistentContents().add(jc);
            } else {
                jc.setParentClass(outer);
            }
        } else {
            resClassifiers.add(jc);
        }
        createdClasses.put (simpleName,  jc);
        map.remove (simpleName);
        return jc;
    }
}
