/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.scanning;
import java.io.InputStream;
import java.io.BufferedInputStream;
import java.lang.reflect.Modifier;
import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.JavaEnum;
import org.netbeans.lib.java.parser.ParserTokens;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.jmiimpl.javamodel.*;
import org.netbeans.modules.javacore.parser.TokenIterator;
import org.netbeans.modules.javacore.parser.MDRParser;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.openide.ErrorManager;

/**
 *
 * @author Tomas Hurka
 */
public class JavaUpdater {
    private final JavaClassClassImpl jclsClass;
    private final JavaEnumClassImpl enumClass;
    private final AnnotationTypeClassImpl annTypeClass;
    private final ResourceClassImpl resourceClass;
    private final ClassIndex classIndex;
    private final String sourceLevel;
    private final FileScanner fileScanner;
    private static final int[] NULL_INTS=new int[0];
    
    /** Creates a new instance of JavaUpdater */
    public JavaUpdater(JavaModelPackage mofModel,String srcLevel, FileScanner fileScanner) {
        jclsClass = (JavaClassClassImpl) mofModel.getJavaClass();
        enumClass = (JavaEnumClassImpl) mofModel.getJavaEnum();
        resourceClass = (ResourceClassImpl) mofModel.getResource();
        annTypeClass = (AnnotationTypeClassImpl) mofModel.getAnnotationType();
        sourceLevel = srcLevel;
        classIndex=ClassIndex.getIndex(mofModel);
        this.fileScanner = fileScanner;
    }

    public int[] computeIndex(ResourceImpl resource,TokenIterator tokens) {
        JavaMetamodel.getDefaultRepository().beginTrans(false);
        try {
            return makeIndex(resource,tokens,false);
        } finally {
            JavaMetamodel.getDefaultRepository().endTrans();
        }
    }

    private int[] makeIndex(ResourceImpl resource,TokenIterator tokens, boolean removeFeatures) {
        int idIndexes[]=NULL_INTS;
        boolean resInited=resource.isInitialized();
        HashSet reinitToDo = new HashSet();
        List javaClasses;
        Set remainingClasses;
        Set usedClasses = new HashSet();
        
        javaClasses=resource.getPersistentClassifiers();

        remainingClasses=getAllClassesFromResource(resource,javaClasses);
        if (!resInited) {
            javaClasses.clear();
        }
        try {
            final int S_NORMAL=0;
            final int S_PACKAGE=1;
            final int S_CLASS=2;

            String id;
            IntSet ids=new IntSet();
            Iterator idIt;
            int i=0;
            int token;
            int last_token=0;
            int state=S_NORMAL;
            int modifiers=0;
            String pack="";
            Stack classes=new Stack();
            int br_level=0;
            boolean packageSet = false;

            while((token=tokens.getNextTokenType())!=0) {
                String text=null;

                if (token==ParserTokens.IDENTIFIER) {
                    int hash=tokens.getIdentifierHash();
                    ids.add(hash);
                }
		if (tokens.isDeprecated())
		    modifiers|=FeatureImpl.DEPRECATED;
                switch (token) {
                    case ParserTokens.PUBLIC:
                        modifiers|=Modifier.PUBLIC;
                        break;
                    case ParserTokens.PROTECTED:
                        modifiers|=Modifier.PROTECTED;
                        break;
                    case ParserTokens.PRIVATE:
                        modifiers|=Modifier.PRIVATE;
                        break;
                    case ParserTokens.ABSTRACT:
                        modifiers|=Modifier.ABSTRACT;
                        break;
                    case ParserTokens.STATIC:
                        modifiers|=Modifier.STATIC;
                        break;
                    case ParserTokens.FINAL:
                        modifiers|=Modifier.FINAL;
                        break;
                    case ParserTokens.SYNCHRONIZED:
                        modifiers|=Modifier.SYNCHRONIZED;
                        break;
                    case ParserTokens.NATIVE:
                        modifiers|=Modifier.NATIVE;
                        break;
                    case ParserTokens.STRICTFP:
                        modifiers|=Modifier.STRICT;
                        break;
                    case ParserTokens.TRANSIENT:
                        modifiers|=Modifier.TRANSIENT;
                        break;
                    case ParserTokens.VOLATILE:
                        modifiers|=Modifier.VOLATILE;
                        break;
                    case ParserTokens.SEMICOLON:
                        modifiers=tokens.isDeprecated() ?
			    FeatureImpl.DEPRECATED : 0;
                        break;
                    case ParserTokens.L_CURLY:
                        br_level++;
                        modifiers=tokens.isDeprecated() ?
			    FeatureImpl.DEPRECATED : 0;
                        break;
                    case ParserTokens.R_CURLY:
                        if (br_level==classes.size() && !classes.isEmpty())
                            classes.pop();
                        br_level--;
                        break;
                }
		if (last_token==ParserTokens.MONKEYS_AT && 
		    token==ParserTokens.IDENTIFIER) {
                    String s=tokens.getIdentifierText();
		    if (s.equals("Deprecated") || // NOI18N
			s.equals("java.lang.Deprecated")) // NOI18N
			// @Deprecated annotation found
			modifiers|=FeatureImpl.DEPRECATED; 
		}
                switch (state) {
                    case S_NORMAL:
                        if (token==ParserTokens.PACKAGE)
                            state=S_PACKAGE;
                        else if ((token==ParserTokens.CLASS || token==ParserTokens.INTERFACE || token==ParserTokens.ENUM) && last_token!=ParserTokens.DOT) {
                            if (!packageSet) {
                                if (JMManager.INCONSISTENCY_DEBUG) System.err.println("JavaUpdater: Setting package name of resource " + resource.getName() + " to " + pack);
                                resource._setPackageName(pack);
                                packageSet = true;
                            }
                            if (last_token==ParserTokens.MONKEYS_AT && token==ParserTokens.INTERFACE) {
                                modifiers|=MDRParser.M_ANNOTATION; // we have found an annotation
                                ids.add("Annotation".hashCode()); // NOI18N
                            }
                            state=S_CLASS;
                        }
                        break;
                    case S_PACKAGE:
			if (token==ParserTokens.IDENTIFIER)
			    text=tokens.getIdentifierText();
                        if (text!=null) {
                            if (pack.length()==0)
                                pack=text;
                            else
                                pack=pack + '.' + text; // NOI18N
                        }
                        if (token==ParserTokens.SEMICOLON) {
                            state=S_NORMAL;
                        }
                        break;
                    case S_CLASS:
			if (token==ParserTokens.IDENTIFIER)
			    text=tokens.getIdentifierText();
                        if (br_level==classes.size() && text!=null) {
                            String fqn;
                            JavaClassImpl jcls;
                            JavaClassImpl enclosingClass=null;

                            if (!classes.empty())
                                enclosingClass=(JavaClassImpl)classes.peek();
                            fqn=constructFqn(pack,classes,text);
                            if (last_token==ParserTokens.INTERFACE) {
                                modifiers|=Modifier.INTERFACE;
                            } else if (last_token == ParserTokens.ENUM) {
                                modifiers |= MDRParser.M_ENUM;
                                ids.add("Enum".hashCode()); // NOI18N
                            }
                            jcls=createJavaClass(fqn,modifiers,removeFeatures,text,remainingClasses,usedClasses,enclosingClass);
                            usedClasses.add(jcls);
                            classes.push(jcls);
                            remainingClasses.remove(jcls);
                            if (jcls.refImmediateComposite() == null) {
                                if (enclosingClass!=null) {
                                    boolean contentsInited = enclosingClass.contentsInited();
                                    boolean persisted = enclosingClass.isPersisted();
                                    if (persisted) {
                                        enclosingClass.getPersistentContents().add(jcls);
                                        if (contentsInited) {
                                            reinitToDo.add(enclosingClass);
                                        }
                                    } else {
                                        jcls.setParentClass(enclosingClass);
                                    }
                                } else {
                                    javaClasses.add(jcls);
                                }
                            }
                        }
                        state=S_NORMAL;
                }
                last_token=token;
            }
            if (!packageSet) {
                if (JMManager.INCONSISTENCY_DEBUG) System.err.println("JavaUpdater: Setting package name of resource " + resource.getName() + " to " + pack);
                resource._setPackageName(pack);
            }
            idIndexes=ids.toArray();
            // the removal has to be done always, to avoid duplicate classes in index
//            if (!resInited) {
                for (Iterator it = remainingClasses.iterator(); it.hasNext();) {
                    JavaClassImpl cls = (JavaClassImpl) it.next();
                    // if some of the classes was deleted and it contained innerclasses,
                    // these innerclasses were automatically deleted too
                    // so we need to check for validity to make sure refDelete is not called
                    // on classes that have already been deleted
                    if (cls.isValid()) {
                        Object parent = cls.refImmediateComposite();
                        if (parent == resource) {
                            javaClasses.remove(cls);
                        } else if (parent instanceof JavaClassImpl) {
                            JavaClassImpl enclosingClass = (JavaClassImpl) parent;
                            boolean contentsInited = enclosingClass.contentsInited();
                            boolean persisted = enclosingClass.isPersisted();
                            if (persisted) {
                                enclosingClass.getPersistentContents().remove(cls);
                                if (contentsInited) {
                                    reinitToDo.add(enclosingClass);
                                }
                            } else {
                                cls.setParentClass(null);
                            }
                        }
                        cls.refDelete();
                    }
                }
//            }
            if (resource.classifiersInited())
                resource.reinitClassifiers();
            for (Iterator it = reinitToDo.iterator(); it.hasNext();) {
                // if during the deletion an innerclass was deleted first (so the outerclass
                // was added to the reinitToDo) and then its outerclass was deleted too,
                // we can have an invalid object in this collection -> validity check is necessary
                JavaClassImpl cls = (JavaClassImpl) it.next();
                if (cls.isValid()) {
                    cls.reinitContents();
                }
            }
        } catch (Exception ex) {
            ErrorManager.getDefault().notify(ex);
        }
        return idIndexes;
    }

    private JavaClassImpl createJavaClass(String fqn, int modifiers,boolean removeFeatures, String simpleName,Set remainingClasses,Set usedClasses,JavaClass enclosingClass) {
        Collection classes = classIndex.getClassesByFqn(fqn);
        JavaClass jcls = null;

        if (JMManager.INCONSISTENCY_DEBUG) System.err.println("JavaUpdater: Looking for class: " + fqn);
        for (Iterator it = classes.iterator(); it.hasNext();) {
            JavaClass temp = (JavaClass) it.next();
            if ((!removeFeatures && enclosingClass != null && enclosingClass.equals(temp.refImmediateComposite()) && !usedClasses.contains(temp)) || remainingClasses.contains(temp)) {
                if (JMManager.INCONSISTENCY_DEBUG) System.err.println("JavaUpdater: Existing class found in index.");
                
                // if the temp exists test for the change of declaration type. E.g.
                // Used when user changes enum declaration to class and vice versa.
                int declType = temp instanceof JavaEnum ? 1 : (temp instanceof AnnotationType ? 2 : 0);
                if ((declType == 1 && !isEnum(modifiers)) || (declType != 1 && isEnum(modifiers)) ||
                    (declType == 2 && !isAnnotation(modifiers)) || (declType != 2 && isAnnotation(modifiers)))
                {
                    // class type changed
                    if (JMManager.INCONSISTENCY_DEBUG) System.err.println("JavaUpdater: The found class is of wrong type: " + temp.getClass().getName() + " -> deleting...");
                    // make sure the class is in remainingClasses so it gets deleted
                    // at the end
                    remainingClasses.add(temp);
                } else {
                    jcls = temp;
                }
                break;
            }
        }

        if (jcls==null) {
            if (JMManager.INCONSISTENCY_DEBUG) System.err.println("JavaUpdater: No suitable class found -> creating...");
            if (isEnum(modifiers)) {
                jcls = enumClass.create(fqn, modifiers, false);
            } else if (isAnnotation(modifiers)) {
                jcls = annTypeClass.create(fqn, modifiers, false);
            } else {
                jcls = jclsClass.create(fqn, modifiers, null, null, false);
            }
            // should not add the class to the index - it is already done in the create method
            //classIndex.addClass(jcls, fqn, simpleName);
        } else {
            if (removeFeatures) {
                ClassUpdater.removePersisted((JavaClassImpl) jcls);
            }
        }
        return (JavaClassImpl) jcls;
    }

    private Set getAllClassesFromResource(Resource res,Collection topLevelClasses) {
        Iterator topIt=topLevelClasses.iterator();
        Set allClasses=new HashSet();
        
        while(topIt.hasNext()) {
            JavaClass topJcls=(JavaClass)topIt.next();
            Iterator innerListIt=classIndex.getClassesByFQNPrefix(topJcls.getName().concat(".")).iterator(); // NOI18N

            allClasses.add(topJcls);
            while (innerListIt.hasNext()) {
                JavaClass innerJcls=(JavaClass)innerListIt.next();
                
                if (res.equals(innerJcls.getResource())) {
                    allClasses.add(innerJcls);
                }
            }
        }
        return allClasses;
    }
    
    private static boolean isEnum(int modifiers) {
        return (modifiers & MDRParser.M_ENUM) != 0;
    }
    
    private static boolean isAnnotation(int modifiers) {
        return (modifiers & MDRParser.M_ANNOTATION) != 0;
    }

    private String constructFqn(String pack,Stack names,String simpleName) {
        String name;

        if (names.isEmpty()) {
            if (pack.length()>0) {
                name=pack+'.';
            } else {
                name="";
            }
        } else {
            name=((JavaClass)names.peek()).getName()+'.';
        }
        return name.concat(simpleName);
    }

    public Collection updateResources(Map javaFiles) 
    {
        JavaMetamodel.getDefaultRepository().beginTrans(false);
        try {
            Iterator resIt=javaFiles.values().iterator();
            List resList=new ArrayList();
            long indexTimestamp;

            indexTimestamp=classIndex.getTimestamp();
            while(resIt.hasNext()) {
                FileInfo file=(FileInfo)resIt.next();
                try {
                    String name=file.getPath();
                    long timestamp=file.lastModified();
                    ResourceImpl resource=(ResourceImpl) resourceClass.resolveResource(name,true,false);

                    resList.add(resource);
                    if (resource.getTimestamp()!=timestamp || resource.isFromMemory() || indexTimestamp<timestamp) {
                        resource.setTimestamp(timestamp, false);
                        if (fileScanner != null) fileScanner.checkParseEagerly(resource);
                        InputStream stream=new BufferedInputStream(file.getInputStream());

                        try {
                            int ids[]=makeIndex(resource, new TokenIterator(stream,sourceLevel,true), !resource.isInitialized());
                            classIndex.setIdentifiers(resource,ids);
                        } finally {
                            stream.close();
                        }
                    }
                } catch (Exception ex) {
                    ErrorManager.getDefault().notify(ex);
                }
            }
            return resList;
        } finally {
            JavaMetamodel.getDefaultRepository().endTrans();
        }
    }
}
