/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.javacore.classpath;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import junit.framework.Assert;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.classpath.GlobalPathRegistry;
import org.netbeans.api.java.queries.SourceForBinaryQuery.Result;
import org.netbeans.junit.NbTestCase;
import org.netbeans.spi.java.classpath.ClassPathFactory;
import org.netbeans.spi.java.classpath.ClassPathImplementation;
import org.netbeans.spi.java.classpath.PathResourceImplementation;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.netbeans.spi.java.queries.SourceForBinaryQueryImplementation;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.filesystems.URLMapper;
import org.openide.util.Lookup;
import org.openide.util.lookup.Lookups;
import org.openide.util.lookup.ProxyLookup;

/**
 * @author  tom
 */
public class MergedClassPathImplementationTest extends NbTestCase {

    private boolean failed;
    
    // Copied from org.netbeans.api.project.TestUtil:
    static {
        // XXX replace with MockServices
        System.setProperty("org.openide.util.Lookup", Lkp.class.getName());
        Assert.assertEquals(Lkp.class, Lookup.getDefault().getClass());
    }

    public static final class Lkp extends ProxyLookup {
        private static Lkp DEFAULT;
        public Lkp() {
            Assert.assertNull(DEFAULT);
            DEFAULT = this;
            setLookup(new Object[0]);
        }
        public static void setLookup(Object[] instances) {
            ClassLoader l = Lkp.class.getClassLoader();
            DEFAULT.setLookups(new Lookup[] {
                Lookups.fixed(instances),
                Lookups.metaInfServices(l),
                Lookups.singleton(l),
            });
        }
    }
    
    private static final String ROOT = "cproot_";
    private static final int ROOT_COUNT = 6;
    private FileObject[] roots;
    
    /** Creates a new instance of MergedClassPathImplementationTest */
    public MergedClassPathImplementationTest (String name) {
        super (name);
    }
    
    protected void setUp() throws Exception {
        super.setUp();
        this.clearWorkDir();
        File f = this.getWorkDir();
        FileObject wd = getWorkDirFileObject ();
        this.roots = new FileObject[ROOT_COUNT];
        for (int i=0; i< ROOT_COUNT; i++) {
            this.roots[i] = wd.createFolder(ROOT+i);
        }        
    }
    
    protected void tearDown() throws Exception {
        super.tearDown();
    }    
    
    
    public void testMergedClassPath () throws IOException {
        ClassPath cp_1 = ClassPathSupport.createClassPath (new FileObject[] {roots[0], roots[1]});
        assertNotNull ("ClassPathSupport.createClassPath() returned null",cp_1);
        ClassPath cp_2 = ClassPathSupport.createClassPath (new FileObject[] {roots[2], roots[3]});
        assertNotNull ("ClassPathSupport.createClassPath() returned null",cp_2);
        DynamicClassPath dympl = new DynamicClassPath();
        ClassPath cp_3 = ClassPathFactory.createClassPath(dympl);
        assertNotNull ("ClassPathSupport.createClassPath() returned null",cp_3);
        GlobalPathRegistry regs = GlobalPathRegistry.getDefault();
        regs.register(ClassPath.COMPILE, new ClassPath[] {cp_1, cp_3});
        MergedClassPathImplementation mcpi = MergedClassPathImplementation.getDefault();
        ClassPath mcp = ClassPathFactory.createClassPath (mcpi);
        assertNotNull ("ClassPathSupport.createClassPath() returned null for MergedClassPathImplementation" ,mcp);
        PathResourceImplementation[] resources = mcpi.getUnresolvedRoots();
        assertEquivalent (resources, new FileObject[] {roots[0],roots[1]});
        regs.register(ClassPath.COMPILE, new ClassPath[] {cp_2});
        resources = mcpi.getUnresolvedRoots();
        assertEquivalent (resources,new FileObject[] {roots[0],roots[1],roots[2],roots[3]});
        dympl.setRoots(new FileObject[]{roots[4],roots[5]});
        resources = mcpi.getUnresolvedRoots();
        assertEquivalent (resources,new FileObject[] {roots[0],roots[1],roots[2],roots[3],roots[4],roots[5]});
        dympl.setRoots(new FileObject[]{});
        resources = mcpi.getUnresolvedRoots();
        assertEquivalent (resources,new FileObject[] {roots[0],roots[1],roots[2],roots[3]});
        dympl.setRoots(new FileObject[]{roots[2],roots[3]});
        resources = mcpi.getUnresolvedRoots();
        assertEquivalent (resources,new FileObject[] {roots[0],roots[1],roots[2],roots[3],roots[2],roots[3]});
        dympl.setRoots(new FileObject[]{});
        resources = mcpi.getUnresolvedRoots();
        assertEquivalent (resources,new FileObject[] {roots[0],roots[1],roots[2],roots[3]});
    }

    public void testDeadlock() throws Exception {
        // instantiate singleton
        final MergedClassPathImplementation mcpi = MergedClassPathImplementation.getDefault();
        Lkp.setLookup(new Object[] {
            new SourceForBinaryQueryImplementation() {
                public Result findSourceRoots(URL binaryRoot) {
                    Thread mcpiLocker = new Thread(new Runnable() {
                        public void run() {
                            synchronized(mcpi) {}
                        }
                    });
                    mcpiLocker.start();
                    try {
                        mcpiLocker.join();
                    } catch (InterruptedException ie) {
                        ie.printStackTrace();
                        failed = true;
                    }
                    return null;
                }
            }
        });
        final DynamicClassPath dympl = new DynamicClassPath();
        ClassPath cp = ClassPathFactory.createClassPath(dympl);
        GlobalPathRegistry regs = GlobalPathRegistry.getDefault();
        regs.register(ClassPath.COMPILE, new ClassPath[] {cp});
        dympl.setRoots(new FileObject[]{roots[0]});
        assertFalse(failed);
    }
    
    private FileObject getWorkDirFileObject () throws IOException {
        FileObject fos = FileUtil.toFileObject(this.getWorkDir());
        if (fos != null) {
            return fos;
        }
        else {
            throw new IllegalStateException ("Can not resolve FileObject for home dir");
        }
    }
    
    private static void assertEquivalent(List entries, FileObject[] roots) throws IOException {
        assertNotNull("Entries are null", entries);
        assertNotNull("Roots are null", roots);
        SortedSet/*<String>*/ expected = new TreeSet(), actual = new TreeSet();
        for (int i = 0; i < roots.length; i++) {
            expected.add(roots[i].getURL().toExternalForm());
        }
        Iterator it = entries.iterator();
        while (it.hasNext()) {
            ClassPath.Entry entry = (ClassPath.Entry) it.next();
            actual.add(entry.getURL().toExternalForm());
        }
        assertEquals("Correct entry URLs", expected, actual);
    }
    
    private static void assertEquivalent (PathResourceImplementation[] resources, FileObject[] roots) {
        assertNotNull("Entries are null", resources);
        assertNotNull("Roots are null", roots);
        assertEquals("Different length",resources.length,roots.length);
        for (int i=0; i<resources.length; i++) {
            URL[] urls = resources[i].getRoots();
            for (int j=0; j<urls.length; j++) {
                FileObject fo = URLMapper.findFileObject(urls[j]);
                assertNotNull (fo);
                assertEquals ("Invalid root",roots[i],fo);
            }
        }
    }
    
    private static class DynamicClassPath implements ClassPathImplementation {
        
        public PropertyChangeSupport support;
        private List roots = Collections.EMPTY_LIST;
        
        public DynamicClassPath () {
            this.support = new PropertyChangeSupport (this);
        }
        
        public void setRoots (FileObject[] roots) throws IOException {
            List newRoots = new ArrayList ();
            for (int i=0; i< roots.length; i++) {
                PathResourceImplementation impl = ClassPathSupport.createResource(roots[i].getURL());
                newRoots.add(impl);
            }
            this.roots = newRoots;
            this.support.firePropertyChange(PROP_RESOURCES,null,null);
        }
        
        public void addPropertyChangeListener(PropertyChangeListener listener) {
            this.support.addPropertyChangeListener (listener);
        }
        
        public List getResources() {
            return Collections.unmodifiableList(this.roots);
        }
        
        public void removePropertyChangeListener(PropertyChangeListener listener) {
            this.support.removePropertyChangeListener (listener);
        }        
    }
}
