/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * ClassMemberModel.java
 *
 * Created on September 18, 2004, 9:33 PM
 */

package org.netbeans.modules.java.navigation;

import javax.jmi.reflect.RefFeatured;
import javax.swing.event.ChangeEvent;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.api.mdr.events.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.Constructor;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.java.JavaDataObject;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.java.navigation.actions.FilterSubmenuAction;
import org.netbeans.modules.java.navigation.actions.SortActionSupport;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.java.navigation.base.FiltersDescription;
import org.netbeans.modules.java.navigation.base.FiltersManager;
import org.netbeans.modules.java.navigation.strings.WeightedStringImpl;
import org.openide.loaders.*;
import org.openide.util.*;

import javax.swing.*;
import java.awt.event.*;
import java.lang.reflect.*;
import java.util.*;
import org.netbeans.modules.java.navigation.actions.GenerateJavadocAction;
import org.netbeans.modules.java.navigation.actions.OpenAction;
import org.netbeans.modules.java.navigation.actions.OverrideAction;
import org.netbeans.modules.java.navigation.jmi.Hacks;
import org.netbeans.modules.java.navigation.jmi.JUtils;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;
import org.netbeans.modules.java.navigation.spi.Reorderable;
import org.netbeans.modules.java.navigation.spi.StringFilter;
import org.netbeans.modules.java.navigation.spi.RelatedItemListModelSupport;

/**
 * Model for java class members.
 *
 * @author Tim Boudreau, Dafe Simonek
 */
public final class ClassMemberModel extends RelatedItemListModelSupport implements MDRChangeListener, Reorderable, FiltersManager.FilterChangeListener {

    /** constants for defined filters */
    private static final String SHOW_NON_PUBLIC = "show_non_public";
    private static final String SHOW_STATIC = "show_static";
    private static final String SHOW_FIELDS = "show_fields";
    private static final String SHOW_INHERITED = "show_inherited";

    /** JMI repository to get data about class we represent */
    private static final MDRepository repo = JavaModel.getJavaRepository();
    
    /** Filters and their state. Note the static, there are only one shared
     * filters instance for all models */
    /** package private for tests */ static FiltersManager filters = null;

    /**
     * Set of objects we are listening to
     */
    private final HashSet listeningTo = new HashSet ();
    /**
     * JavaDataObject we represent
     */
    private JavaDataObject dob = null;
    /**
     * The main class object we represent
     */
    private JavaClass clazz = null;

    /** True when sorting by positions in the source, false when 
     * sorting alphabetically by names */
    private static boolean naturalSort = false;
    
    /** List of 1 element shown when no members are displayable */
    private List noMembersAvail;
    
    /** True when model is used inside InheritanceTreeModel, false otherwise */
    private boolean isInheritance = false;
    
    /** helps to invoke tooltip in async way */
    private JUtils.TipHackInvoker tipInvoker;
    
    /** asociated related item listener impl */
    private RelatedItemListener relatedL;
    
    private static final boolean singleClick = Boolean.getBoolean ("nb.navigator.singleclick");

    /**
     * Constructor used by Java Navigation ModelProvider
     */
    ClassMemberModel (JavaDataObject jdo, JUtils.TipHackInvoker tipInvoker, RelatedItemListener relatedL) {
        super(new ClassMemberRelatedItemProvider(jdo), singleClick, true, true);
        this.dob = jdo;
        this.isInheritance = false;
        this.tipInvoker = tipInvoker;
        this.relatedL = relatedL;
    }

    /**
     * Constructor used to embed an instance in an InheritanceTreeModel
     */
    ClassMemberModel (JavaClass clazz) {
        super(null, false, true, true);
        this.clazz = clazz;
        this.isInheritance = true;
    }

    void setDataObject (JavaDataObject jdo) {
        this.dob = jdo;
    }

    public String getTooltip (Object o) {
        return null;
    }
    
    public String getTooltip (Object tooltipFor, int x, int y) {
        ItemPaintingData itemData = (ItemPaintingData)tooltipFor;
        return JUtils.getTooltip(itemData.getWString(), itemData.getJMIData(),
                                  clazz, x, y, tipInvoker);
    }

    public Action getDefaultAction (Object o) {
        if (!(o instanceof ItemPaintingData)) o = null; //CCE if user clicks a please wait node
        Object jmiData = o != null ? ((ItemPaintingData)o).getJMIData() : null;
        return OpenAction.createOpenAction(jmiData);
    }
    
    public Action[] getActions (Object o) {
        Object jmiData = null;
        if (o instanceof ItemPaintingData) {
            jmiData = ((ItemPaintingData)o).getJMIData();
        }
        boolean wrap = false;
        if (jmiData != null) {
            JUtils.isWrapper(jmiData);
            jmiData = JUtils.unwrap(jmiData);
        }

        if (jmiData == null || (jmiData instanceof Method && wrap)) {
            return new Action[]{
                new SortActionSupport.SortByNameAction(this),
                new SortActionSupport.SortBySourceAction(this),
                null,
                new FilterSubmenuAction(filtersInstance())
            };
        } else {
            return new Action[]{
                new GenerateJavadocAction((ClassMember)jmiData),
                null,
                new SortActionSupport.SortByNameAction(this),
                new SortActionSupport.SortBySourceAction(this),
                null,
                new FilterSubmenuAction(filtersInstance())
            };
        }
    }
    
    public List getSearchResults (String partial) {
        if ( getList () == null ) {
            return Collections.EMPTY_LIST;
        }
        partial = partial.trim();
        StringFilter exactFilter = StringFilter.create(partial);
        StringFilter insensitiveFilter = StringFilter.create((partial.toUpperCase()));
        ArrayList results = new ArrayList();
        ArrayList caseInsensitive = new ArrayList();
        String s;
        ItemPaintingData pData;

        for ( Iterator i = getList ().iterator (); i.hasNext (); ) {
            pData = (ItemPaintingData)i.next();
            s = pData.getWString().toString();
            int dotIndex = s.lastIndexOf(".");
            if (dotIndex != -1) {
                // search into inner classes also (prefixed by class names)
                s = s.substring(dotIndex + 1);
            }
            // try exact matches first, then case insensitive if exact fails
            if (exactFilter.match(s)) {
                results.add(pData);
            } else if (insensitiveFilter.match(s.toUpperCase())) {
                caseInsensitive.add(pData);
            }
        }
        results.addAll(caseInsensitive);
        
        return Collections.unmodifiableList ( results );
    }

    public Icon getIcon (Object o) {
        return ((ItemPaintingData)o).getIcon();
    }

    protected WeightedString assembleName (WeightedString str, Object o) {
        return ((ItemPaintingData)o).getWString();
    }
        
    protected boolean isObjectValid (Object o) {
        // no check needed as now we get all JMI objects of list under read
        // transaction, in which we first check main object validity
        return true;
    }
    
    /** Overriden to find item correctly also when jmi Element is passed in,
     * as it the case from RelatedItemProviders 
     */
    public int indexOf (Object obj) {
        if (obj instanceof Element) {
            return findItemIndexForElement((Element)obj);
        }
        return super.indexOf(obj);
    }
    
    /** Finds ItemPaintingData item for given JMI element or null if nothing
     is found */
    private int findItemIndexForElement (Element elem) {
        List items = getList();
        ItemPaintingData curItem = null;
        int result = 0;
        for (Iterator iter = items.iterator(); iter.hasNext(); result++) {
            curItem = (ItemPaintingData)iter.next();
            if (elem.equals(curItem.getJMIData())) {
                return result;
            }
        }
        // not found
        return -1;
    }

    protected void stopListening () {
        if ( !listeningTo.isEmpty () ) {
            for ( Iterator i = listeningTo.iterator (); i.hasNext (); ) {
                ( (MDRChangeSource) i.next () ).removeListener ( this );
            }
            listeningTo.clear ();
        }
        if (relatedL != null) {
            getRelatedItemProviderSupport().removeRelatedItemListener(relatedL);
        }
        
    }

    protected void startListening () {
        if (relatedL != null) {
            try {
                getRelatedItemProviderSupport().addRelatedItemListener(relatedL);
            } catch (TooManyListenersException ex) {
                ex.printStackTrace();
            }
        }
        // other listening in loadContents
    }

    private boolean includeInners = true;
    void setIncludeInners (boolean val) {
        includeInners = val;
    }

    /** Loads content of asociated java class from JMI and transforms into
     * painting-friendly data, which are returned.
     *
     * @return List of ItemPaintingData
     */
    public List loadContents () {
        List result = null;

        ((JMManager) JMManager.getManager()).waitScanFinished();
        repo.beginTrans(false);
        try {
            fireBusyChange(true);
            // check validity
            if ( this.clazz != null && !this.clazz.isValid () ) {
                this.clazz = null;
            }
            
            JavaClass clazz = this.clazz;
            JavaClass[] jc = new JavaClass[]{clazz};
            boolean showInherited = filtersInstance().isSelected(SHOW_INHERITED);

            if (isInheritance) {
                if (clazz != null) {
                    result = JUtils.getClassMembers ( clazz, showInherited, includeInners );
                }
            } else {
                result = JUtils.getClassMembers ( dob, showInherited, jc, includeInners );
            }
            // class is empty or invalid 
            if (result == null) {
                return getNoMembersAvail();
            }
            
            if ( jc[ 0 ] != this.clazz && jc[ 0 ] != null ) {
                this.clazz = jc[ 0 ];
            }
            
            // exclude unwanted elements (xxx what about static initializers?)
            for (Iterator i=result.iterator(); i.hasNext();) {
                Object o = JUtils.unwrap(i.next());
                if (!(o instanceof Method) && !(o instanceof Constructor)
                    && !(o instanceof Field) && !(o instanceof Attribute) ) {
                    i.remove();
                }
            }

            result = filter(this.clazz, result);

            Object[] toSort = result.toArray();
            Arrays.sort ( toSort, naturalSort ? (Comparator) new NaturalSortComparator () :
                    (Comparator) new ClassMemberComparator ( clazz ) );
            result = Arrays.asList ( toSort );

            result = buildMembersForPainting(result);
            
            if (result.isEmpty()) {
                result = getNoMembersAvail();
            }

            startListeningTo ( Hacks.getResourceForDataObject ( dob ) );

        } finally {
            repo.endTrans(false);
            fireBusyChange(false);
        }

        return result;
    }
    
    /** Examine JMI members data and create list of data needed for painting.
     *
     * @return List of ItemPaintingData
     */
    private List buildMembersForPainting (List jmiMembers) {
        List result = new ArrayList(jmiMembers.size());
        WeightedString wString = null;
        Object jmiData = null;
        
        for (Iterator iter = jmiMembers.iterator(); iter.hasNext(); ) {
            jmiData = iter.next();
            result.add(new ItemPaintingData(
                    buildWString(jmiData), JUtils.iconFor(jmiData), jmiData)
            );
        }
        
        return result;
    }
    
    /** Returns 1-element list, used for indication that no members are
     * available for showing (because of filter or empty class..)
     */ 
    private List getNoMembersAvail () {
        if (noMembersAvail == null) {
            noMembersAvail = new ArrayList(1);
            WeightedString wString = new WeightedStringImpl();
            wString.startMarkupRun(WeightedString.DEEMPHASIZED);
            wString.append(NbBundle.getMessage(ClassMemberModel.class,
                    "LBL_NoMembersAvail"), 0.9f);
            noMembersAvail.add(new ItemPaintingData(wString, null, null));
        }
        return noMembersAvail;
    }

    /** Creates and fills weighted string from given jmi data object */
    private WeightedString buildWString (Object jmiData) {
        WeightedString result = new WeightedStringImpl();
        
        if ( jmiData instanceof ClassMember && !( (ClassMember) jmiData ).isValid () ) {
            result.startMarkupRun ( WeightedString.ERROR_EMPHASIS );
            result.append ( NbBundle.getMessage ( ClassMemberModel.class,
                    "LBL_Invalid" ), 0.9f ); //NOI18N
        } else {
            boolean useName = dob == null;
            if (!useName && !isInheritance && jmiData instanceof ClassMember && 
                ((ClassMember) jmiData).getDeclaringClass() instanceof JavaClass) {
                //XXX check this in JUtils
                JavaClass declClass = (JavaClass)((ClassMember)jmiData).getDeclaringClass();
                useName = declClass != clazz || declClass.isInner();
            }
            JUtils.extractName(result, jmiData, clazz, useName);
        }
        
        return result;
    }

    
    /** Filters given list of JMI members and returns filtered subset that
     * satisfies filtering conditions.
     *
     * @return filtered List of memebers
     */
    private List filter (JavaClass headClass, List members) {
        FiltersManager f = filtersInstance();
        
        boolean showStatic = f.isSelected(SHOW_STATIC);
        boolean showNonPublic = f.isSelected(SHOW_NON_PUBLIC);
        boolean showFields = f.isSelected(SHOW_FIELDS);
        
        // show everything == no filtering
        if (showStatic && showNonPublic && showFields) {
            return members;
        }
        // do filtering
        List result = new ArrayList ();
        boolean isHeadPublic = (headClass.getModifiers() & Modifier.PUBLIC) != 0;
        Object curElem, unwrappedElem;
        ClassMember cm;
        
        for (Iterator i = members.iterator(); i.hasNext(); ) {
            curElem = i.next();
            unwrappedElem = JUtils.unwrap(curElem);
            if (unwrappedElem instanceof ClassMember) {
                cm = (ClassMember) unwrappedElem;
                // show non public if desired
                if (!showNonPublic) {
                    if (!isHeadPublic || !isPartOfPublicAPI(cm)) {
                        continue;
                    }
                }
                // show static if desired
                if (!showStatic && ((cm.getModifiers () & Modifier.STATIC ) != 0 )) {
                    continue;
                }
                // show fields if desired
                if (!showFields && (cm instanceof Field)) {
                    continue;
                }
                result.add(cm);
            }
        }
        
        return result;
    }
    

    /** Decides if given class member is part of public API or not.
     *
     * @return true if given member is part of public API, false otherwise
     */
    private boolean isPartOfPublicAPI (ClassMember cm) {
        boolean isPublic = (cm.getModifiers() & Modifier.PUBLIC) != 0;
        boolean isProtected = (cm.getModifiers() & Modifier.PROTECTED) != 0;
        // exclude non public members
        if (!isPublic && !isProtected) {
            return false;
        }
        // exclude members from innerclasses
        RefFeatured parentRef = cm.refImmediateComposite();
        JavaClass jc;
        int jcModifs;
        boolean isJCPublic, isJCProtected, isJCFinal;
        while (parentRef instanceof JavaClass) {
            jc = (JavaClass)parentRef;
            jcModifs = jc.getModifiers();
            isJCPublic = (jcModifs & Modifier.PUBLIC) != 0;
            isJCProtected = (jcModifs & Modifier.PROTECTED) != 0;
            if (!isJCPublic && !isJCProtected) {
                // exclude members defined in not publicly accessible class
                return false;
            }
            if (isProtected && isJCProtected && 
                ((jcModifs & Modifier.FINAL) != 0)) {
                // exclude protected members defined in protected,
                // but final class
                return false;
            }
            parentRef = jc.refImmediateComposite();
        }
        
        return true;
    }
    
    

/* XXX - string filter temporarily disabled...
    private List stringFilterMembers (List members) {
        StringFilter sf = (StringFilter) filter;

        //BEWARE!  Calling remove() on the real list can actually delete the
        //methods from the source file!!!
        List l = new ArrayList ();

        for ( Iterator i = members.iterator (); i.hasNext (); ) {
            Object o = i.next ();
            o = JUtils.unwrap ( o );

            ClassMember cm = (ClassMember) o;
            if ( sf.match ( cm.getName () ) ) {
                l.add ( cm );
            }
        }
        return l;
    }
 */

    private void startListeningTo (Object r) {
        if ( r != null && r instanceof MDRChangeSource && !listeningTo.contains ( r ) ) {
            listeningTo.add ( r );
            ( (MDRChangeSource) r ).addListener ( this );
        } else if ( clazz != null && clazz instanceof MDRChangeSource && !listeningTo.contains ( clazz ) ) {
            listeningTo.add ( clazz );
            ( (MDRChangeSource) clazz ).addListener ( this );
        }
    }

    /** Reactions to changes from MDR.
     * XXX [dafe] - need to be checked with MDR guys to apply more strict
     * conditions.
     */
    public void change (MDRChangeEvent e) {
        if ( !( e instanceof AttributeEvent ) ) {
            return;
        }
        change ();
    }

    public boolean isReorderable () {
        return naturalSort;
    }

    /** Sets natural (as in source) or alphabetical sort of elements
     * @param value true when natural sort should be set, false when alphabetical
     * sort should be set
     */
    public void setNaturalSort (boolean value) {
        if (naturalSort == value) {
            return;
        } 
        naturalSort = value;
        // perform change
        fireChange();
        change();
    } 
    
    /** @return true if natural sort is enabled (sort as in source), false 
     * when alphabetical source is enabled
     */
    public boolean isNaturalSort () {
        return naturalSort;
    }

    public String move (Object o, int newLocation) {
        Object jmiData = ((ItemPaintingData)o).getJMIData();
        jmiData = JUtils.unwrap(jmiData);

        ClassMember toMove = (ClassMember) jmiData;
        ClassMember after = (ClassMember) ( (ItemPaintingData)getElementAt(newLocation) ).getJMIData();

        String result = JUtils.move ( toMove, after );
        if ( result == null ) {
            change ();
        }
        return result;
    }

    public JComponent getFilters () {
        FiltersManager f = filtersInstance();
        
        f.hookChangeListener(this);
        return f.getComponent();
    }

    /** implementation of FilterChangeListener, reacts to changes in filters
     * state */
    public void filterStateChanged(ChangeEvent e) {
        if (isActive()) {
            change();
        }
    }
    
    /** Accessor for filters manager, note it is called from various threads
     * (AWT through getFilters and non-AWT through loadContents)
     */
    private synchronized static FiltersManager filtersInstance () {
        if (filters == null) {
            filters = createFilters();
        }
        return filters;
    }
    
    /** Creates filter descriptions and filters itself */
    private static FiltersManager createFilters () {
        FiltersDescription desc = new FiltersDescription();
        
        desc.addFilter(SHOW_INHERITED,
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowInherited"),     //NOI18N
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowInheritedTip"),     //NOI18N
                false,
                new ImageIcon (Utilities.loadImage("org/netbeans/modules/java/navigation/resources/filterHideInherited.png")), //NOI18N
                null
        );
        desc.addFilter(SHOW_FIELDS,
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowFields"),     //NOI18N
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowFieldsTip"),     //NOI18N
                true,
                new ImageIcon (Utilities.loadImage("org/netbeans/modules/java/navigation/resources/filterHideFields.gif")), //NOI18N
                null
        );
        desc.addFilter(SHOW_STATIC,
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowStatic"),     //NOI18N
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowStaticTip"),     //NOI18N
                true,
                new ImageIcon (Utilities.loadImage("org/netbeans/modules/java/navigation/resources/filterHideStatic.png")), //NOI18N
                null
        );
        desc.addFilter(SHOW_NON_PUBLIC,
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowNonPublic"),     //NOI18N
                NbBundle.getMessage(ClassMemberModel.class, "LBL_ShowNonPublicTip"),     //NOI18N
                true,
                new ImageIcon (Utilities.loadImage("org/netbeans/modules/java/navigation/resources/filterHideNonPublic.png")), //NOI18N
                null
        );
        
        return FiltersDescription.createManager(desc);
    }
    
    /** Purely data class, encapsulation of data that are needed for 
     * painting of one member item. Contains also original JMI data object
     * from were data for painting were constructed from.
     */
    static final class ItemPaintingData {
        
        private final WeightedString wString;
        private final Icon icon;
        private final Object jmiData;
        

        public ItemPaintingData (WeightedString wString, Icon icon, Object jmiData) {
            this.wString = wString;
            this.icon = icon;
            this.jmiData = jmiData;
        }
        
        public WeightedString getWString () {
            return wString;
        }
        
        public Icon getIcon () {
            return icon;
        }
        
        public Object getJMIData () {
            return jmiData;
        }
        
        public boolean equals (Object o) {
            if (o == this) {
                return true;
            } else {
                return jmiData == null ? false :
                    jmiData.equals(o);
            }
        }
        
        public int hashCode() {
            return jmiData == null ? System.identityHashCode(this) : 
                jmiData.hashCode();
        }
        
    } // end of ItemPaintingData
    
    
}
