/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.navigation;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.KeyStroke;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;
import javax.swing.event.MenuKeyEvent;
import javax.swing.event.MenuKeyListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.tree.TreePath;
import org.netbeans.modules.java.navigation.base.NavigatorJList;
import org.netbeans.modules.java.navigation.spi.DisplayProvider;
import org.netbeans.modules.java.navigation.spi.NavigatorTreeModel;
import org.openide.util.Utilities;


/**
 * Controller for InheritanceTreePanelUI. Handles user actions of inheritance 
 * tree panel UI.
 *
 * @author Dafe Simonek
 */
final class InheritanceTreeController extends MouseAdapter implements TreeExpansionListener {
    
    /** UI of members navigator panel */ 
    private InheritanceTreePanelUI panelUI;
    /** Tree - main content of panelUI */
    private JTree tree;

    public InheritanceTreeController (InheritanceTreePanelUI panelUI) {
        this.panelUI = panelUI;
        this.tree = panelUI.getContent();
        initActions();
        initListeners();
    }
    
    private void initActions () {
        // action on ENTER key
        Action keyAction = new OpenByKeyboardAction();
        tree.getInputMap(JComponent.WHEN_FOCUSED).put(
                KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0, true), "open"); //NOI18N
        tree.getActionMap().put("open", keyAction); //NOI18N

        // popup menu via keyboard Shift+F10
        Action showPopupAction = new ShowPopupAction();
        tree.getInputMap(JComponent.WHEN_FOCUSED).put(
                KeyStroke.getKeyStroke(KeyEvent.VK_F10, KeyEvent.SHIFT_DOWN_MASK, true),
                "popup" ); //NOI18N
        tree.getActionMap().put("popup", showPopupAction); //NOI18N

    }
    
    private void initListeners () {
        tree.addMouseListener(this);
        tree.addTreeExpansionListener(this);
    }
    
    /** Shows popup menu on given list at given point */
    private static void showPopup (JTree tree, Point p) {
        // obtain current element
        DisplayProvider dp = (DisplayProvider)tree.getModel();
        TreePath path = tree.getPathForLocation(p.x, p.y);
        Object elem = null;
        if (path != null) {
            elem = path.getLastPathComponent();
            tree.scrollRectToVisible(tree.getPathBounds(path));
        }
        
        JPopupMenu m = createPopupMenuForItem(tree, dp, elem);
        if (m != null) {
            // XXX - TBD m.addPopupMenuListener(this);
            m.show(tree, p.x, p.y);
        }
    }

    /** Runs default model action on double click (or single click eventually) */ 
    public void mouseClicked (MouseEvent e) {
        // react only on left mouse button click
        if (!SwingUtilities.isLeftMouseButton(e)) {
            return;
        }
        // obtain clicked element
        Point p = e.getPoint ();
        Object clicked = null;
        TreePath path = tree.getPathForLocation(p.x, p.y);
        if (path != null) {
            clicked = path.getLastPathComponent();
        } else {
            // no element clicked
            return;
        }        
        // perform default action from model (DisplayProvider)
        DisplayProvider dp = (DisplayProvider)tree.getModel();
        if (dp.isDefaultActionInstant() || e.getClickCount() > 1) {
            Action act = dp.getDefaultAction(clicked);
            if (act != null && act.isEnabled()) {
                ActionEvent ae = new ActionEvent(InheritanceTreeController.this,
                                ActionEvent.ACTION_PERFORMED, "click"); //NO18N
                act.actionPerformed(ae);
            }
        }
    }
    
    /** Overriden to show popup if triggered */
    public void mousePressed (MouseEvent e) {
        if (e.isPopupTrigger()) {
            showPopup((JTree)e.getSource(), e.getPoint());
            e.consume();
        }
    }

    /** Overriden to show popup if triggered */
    public void mouseReleased (MouseEvent e) {
        if (e.isPopupTrigger()) {
            showPopup((JTree)e.getSource(), e.getPoint());
        }
    }
    
    /***** Impl of TreeExpansionListener, to notify tree model properly ******/
    
    public void treeCollapsed (TreeExpansionEvent e) {
        NavigatorTreeModel tr = (NavigatorTreeModel)tree.getModel();
        tr.notifyExpandedState(e.getPath (), false);
    }

    public void treeExpanded (TreeExpansionEvent e) {
        NavigatorTreeModel tr = (NavigatorTreeModel)tree.getModel();
        tr.notifyExpandedState(e.getPath (), true);
    }
    
    
    
    /************* Utility methods used also by ClassMemberController *********/
    
    /** Actually shows popup menu for given Component, 
     * DisplayProvider intance, element context on given point.
     */
    static JPopupMenu createPopupMenuForItem (Component comp, DisplayProvider dp, 
                                        Object elem) {
        // obtain actions
        Action def = dp.getDefaultAction(elem);
        Action[] other = dp.getActions(elem);
        int actionCount = (other == null ? 0 : other.length) +
                          (def == null ? 0 : 1); 
        if (actionCount == 0) {
            // no actions, no popup
            return null; 
        }

        // insert default action on first place
        Action[] allActions;
        if (def != null) {
            allActions = new Action[actionCount];
            allActions[0] = def;
            if (other != null) {
                System.arraycopy(other, 0, allActions, 1, other.length);
            }
        } else {
            allActions = other;
        }
        
        // create popup from actions and return 
        return Utilities.actionsToPopup(allActions, comp);
    }

    /** Runs given action, surrounding actions' processing with busy 
     * cursor on given component
     */
    static void runWithWaitCursor (JComponent curComp, Action a, ActionEvent evt) {
        Cursor old = curComp.getCursor ();
        curComp.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        try {
            a.actionPerformed(evt);
        } finally {
            curComp.setCursor(old);
        }
    }
    
    
    /* Action that runs default action of list model on currently selected item
     */
    private class OpenByKeyboardAction extends AbstractAction {
        
        public boolean isEnabled () {
            return findDefaultAction() != null;
        }

        /** Returns default action for current selection or null */
        private Action findDefaultAction () {
            JTree tree = InheritanceTreeController.this.tree;
            TreePath path = tree.getSelectionPath();
            Object sel = path == null ? null : path.getLastPathComponent();
            
            if (sel != null) {
                Action a = ((DisplayProvider)tree.getModel()).getDefaultAction(sel);
                if (a != null && a.isEnabled()) {
                    return a;
                }
            }
            
            return null;
        }

        /** Runs found default action if possible */
        public void actionPerformed (ActionEvent evt) {
            Action a = findDefaultAction();
            if (a != null) {
                runWithWaitCursor(tree, a, evt);
            }
        }
        
    } // end of OpenByKeyboardAction
    
    
    /** Shows popup for currently selected item */
    private final class ShowPopupAction extends AbstractAction {
        
        public void actionPerformed (ActionEvent ae) {
            TreePath path = InheritanceTreeController.this.tree.getSelectionPath();
            if (path != null) {
                Rectangle r = tree.getPathBounds(path);
                Point p = new Point(r.x + 10, r.y + r.height - 3);
                showPopup(tree, p);
            }
        }
    }

    /*
    private class ToggleTapPanelsAction extends AbstractAction {
        public void actionPerformed (ActionEvent ae) {
            boolean val = filtersPanel.isExpanded ();
            synchronized (getTreeLock ()) {
                filtersPanel.setExpanded ( !val );
            }
            revalidate ();
            repaint ();
        }
    }*/

    
}
