/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.navigation.base;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.image.BufferedImage;
import javax.swing.ComboBoxModel;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTree;
import javax.swing.ListCellRenderer;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.plaf.FontUIResource;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeCellRenderer;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;
import org.netbeans.modules.java.navigation.spi.ModelProvider;
import org.netbeans.modules.java.navigation.spi.DisplayProvider;
import org.netbeans.modules.java.navigation.spi.RelatedItemProvider;
import org.netbeans.modules.java.navigation.spi.AbstractModel;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;

/**
 * A cell renderer for Navigator, which renders a WeightedCharacterIterator and knows about secondary selections and
 * such.
 *
 * @author Tim Boudreau
 */
public final class CellRenderer extends DefaultTreeCellRenderer implements ListCellRenderer, TreeCellRenderer {
    
    /** colors for background and foreground */
    private static Color unfocusedSelFg;
    private static Color unfocusedSelBg;

    private boolean inSetContainerWidth = false;

    private void setContainerWidth (JComponent jc) {
        if ( inSetContainerWidth ) {
            //Viewport UI will cause a stack overflow if this is being called
            //inside getPreferredSize();
            return;
        }
        inSetContainerWidth = true;
        try {
            setContainerWidth ( jc.getWidth () );
        } finally {
            inSetContainerWidth = false;
        }
    }

    private int containerWidth = 50;
    private boolean widthChanged = true;
    private int scrollPaneWidth = -1;

    private void setContainerWidth (int i) {
        if ( containerWidth != i ) {
            containerWidth = i;
            widthChanged = true;
        }
    }
    
    private void setScrollPaneWidth (JComponent jc) {
        Container scrollPane = jc.getParent();
        if (scrollPane != null) {
            scrollPane = SwingUtilities.getAncestorOfClass(
                            JScrollPane.class, jc.getParent());
        }
        if (scrollPane != null) {
            int newWidth = scrollPane.getWidth();
            widthChanged = scrollPaneWidth != newWidth;
            if (widthChanged) {
                scrollPaneWidth = newWidth;
            }
        } else {
            widthChanged = scrollPaneWidth != -1;
            if (widthChanged) {
                scrollPaneWidth = -1;
            }
        }
    }

    private int charCount = 20;
    private int charWidth = 8;

    private void calcCharCount (Graphics g, FontMetrics fm) {
        int calcWidth = scrollPaneWidth == -1 ? containerWidth : scrollPaneWidth;
        int wid =  calcWidth - getIconTextGap(); //Make room for icon
        String test = "abcdefghkmnopqrstuvwxyzABCDEFG";
        int swidth = fm.stringWidth ( test );
        charWidth = swidth / test.length ();
        charCount = Math.max ( 5, ( wid / charWidth ) );
        widthChanged = false;
    }

    private boolean isSelected () {
        return selected;
    }

    private boolean isParentFocused () {
        return focused && !renderingCombo;
    }

    private void setSelected (boolean val) {
        selected = val;
    }

    private void setLead (boolean val) {
        lead = val;
    }

    private void setParentFocused (boolean val) {
        focused = val;
    }

    private void setRenderingCombo (boolean val) {
        renderingCombo = val;
    }

    private static final Color relatedColor = new Color ( 255, 240, 205 );
    private static final Color relatedBorder = new Color ( 255, 215, 207 );

    public Color getBackground () {
        return renderingCombo && !isSelected () ? null : super.getBackground ();
    }

    public Color getBackgroundNonSelectionColor () {
        if (renderingCombo) {
            return isSelected() ? getBackgroundSelectionColor() :
                null;
        } else {
            return super.getBackgroundNonSelectionColor ();
        }
    }

    private RelatedItemProvider provider = null;

    private void setRelatedItemProvider (RelatedItemProvider p) {
        this.provider = p;
    }

    private boolean related;

    private boolean isRelated () {
        return related && !isSelected();
    }

    private void setRelated (boolean val) {
        related = val;
    }

    private boolean primary = false;

    private void setPrimary (boolean val) {
        primary = val;
    }

    private boolean isTree () {
        return tree;
    }

    private void setTree (boolean val) {
        tree = val;
    }

    void setShowDragFeedback (boolean val) {
        showDragFeedback = val;
    }

    private boolean isShowDragFeedback () {
        return showDragFeedback;
    }

    private boolean showDragFeedback = false;
    private boolean renderingCombo = false;
    private boolean selected = false;
    private boolean lead = false;
    private boolean focused = false;
    private boolean tree = false;

    public Component getListCellRendererComponent (JList jList, Object obj, int idx, boolean sel, boolean lead) {
        str = null;
        setSelected ( sel );
        setLead ( lead );
        setParentFocused ( jList.hasFocus () );
        if ( jList.getModel () instanceof DisplayProvider ) {
            if ( jList.getModel () instanceof RelatedItemProvider ) {
                setRelatedItemProvider ( (RelatedItemProvider) jList.getModel () );
            }
            configureFrom ( (DisplayProvider) jList.getModel (), obj );
            setContainerWidth ( jList );
            if (shouldAbbreviate()) {
                setScrollPaneWidth(jList);
            }
        } else if ( obj instanceof ModelProvider ) {
            setText ( ( (ModelProvider) obj ).getDisplayName () );
            // #53688: HIE don't want icons
            //setIcon ( ( (ModelProvider) obj ).getIcon () );
            setIcon ( null );
        } else {
            setText ( obj == null ? "" : obj.toString () );
            setIcon ( null );
        }
        setRenderingCombo ( jList.getModel () instanceof ComboBoxModel );
        if ( sel ) {
            setBackground ( getBackgroundSelectionColor () );
            setForeground ( getTextSelectionColor () );
        } else {
            setBackground ( getBackgroundNonSelectionColor () );
            setForeground ( getTextNonSelectionColor () );
        }
        setFont ( jList.getFont () );
        setTree ( false );
        return this;
    }

    /**
     * Workaround for JDK bug #6272764.
     */
    public void setFont(Font font) {
	if (font instanceof FontUIResource) {
	    font = new Font(font.getName(), font.getStyle(), font.getSize());
        }
        super.setFont(font);
    }
    
    public Component getTreeCellRendererComponent (JTree jTree, Object obj, boolean sel, boolean exp, boolean leaf,
                                                   int row, boolean lead) {
        str = null;
        setSelected ( sel );
        setLead ( lead );
        setParentFocused ( jTree.hasFocus () );
        setRenderingCombo ( false );
        if ( jTree.getModel () instanceof DisplayProvider ) {
            if ( jTree.getModel () instanceof RelatedItemProvider ) {
                setRelatedItemProvider ( (RelatedItemProvider) jTree.getModel () );
            }
            configureFrom ( (DisplayProvider) jTree.getModel (), obj );
            setContainerWidth ( jTree );
        } else {
            setText ( obj == null ? "" : obj.toString () );
            setIcon ( null );
        }
        if ( sel ) {
            setBackground ( getBackgroundSelectionColor () );
            setForeground ( getTextSelectionColor () );
        } else {
            setBackground ( getBackgroundNonSelectionColor () );
            setForeground ( getTextNonSelectionColor () );
        }
        setFont ( jTree.getFont () );
        setTree ( true );
        return this;
    }

    private void configureFrom (DisplayProvider dp, Object obj) {
        if ( AbstractModel.isWaitMarker( obj ) ) {
            setText ( NbBundle.getMessage ( CellRenderer.class, "MSG_PleaseWait" ) ); //NOI18N
            setIcon ( WAIT_ICON );
        } else if ( AbstractModel.isInvalidMarker( obj ) ) {
            setText ( NbBundle.getMessage ( CellRenderer.class, "MSG_Invalid" ) ); //NOI18N
            setIcon ( null );
        } else {
            setIcon ( dp.getIcon ( obj ) );
            setWeightedString ( dp.getName ( obj ) ); //XXX
        }
        if ( provider != null ) {
            setRelated ( provider.isRelated ( obj ) );
            setPrimary ( provider.isPrimary ( obj ) );
        } 
    }

    private static Graphics scratch = new BufferedImage ( 2, 2, BufferedImage.TYPE_INT_ARGB ).getGraphics ();

    public Dimension getPreferredSize () {
        if (isTree()) {
            boolean pad = false;
            if (str != null) {
                setText ( str.toString() );
                int markup = str.allMarkupTypes();
                if ((markup & WeightedString.BOLD) != 0 || (markup & WeightedString.ITALIC) != 0) {
                    pad = true;
                }
            }
            Dimension d = super.getPreferredSize();
            d.width += 1;
            if (pad) {
                //Make a little room for italic or boldface to make the
                //string wider
                d.width = (int) (((float) d.width) * 1.15f);
            }
            return d;
        }
        String s = str == null ? getText () : str.toString ();
        if ( s == null ) {
            return new Dimension ( 16, 100 );
        }
        Font f = getFont ();
        if ( f == null ) {
            f = UIManager.getFont ( "controlFont" ); //NOI18N
        }
        FontMetrics fm = scratch.getFontMetrics ( f );
        int w = ( getIcon () != null ? getIcon ().getIconWidth () + getIconTextGap () : 0 )
                + fm.stringWidth ( s );

        w += 8; //Fudge factor for bold fonts

        Dimension result = new Dimension ( w, Math.max ( 16, fm.getHeight () ) );
        if (isTree()) {
            result.width = containerWidth - getLocation().x + (getIcon() != null ? getIcon().getIconWidth() + getIconTextGap() : 0);
        }
        return result;
    }

    private WeightedString str = null;

    private void setWeightedString (WeightedString as) {
        this.str = as;
    }

    public void paint (Graphics g) {
        if ( str == null ) {
            // regular painting without weighted strings
            super.paint ( g );
        } else {
            // painting with weighted strings
            Icon ic = getIcon ();
            Insets ins = getInsets ();
            int x = ins.left;
            if ( ic != null ) {
                x += ic.getIconWidth () + getIconTextGap ();
            }
            boolean selected = isSelected();
            boolean parentFocused = isParentFocused();
            
            if ( isShowDragFeedback () && selected ) {
                g.setColor ( getForeground () );
                g.drawLine ( 0, 0, containerWidth, 0 );
            } else if ( isRelated () ) {
                g.setColor ( relatedColor );
                g.fillRoundRect ( 2, 1, containerWidth - 5, getHeight () - 2, 16, 16 );
                g.setColor ( relatedBorder );
                g.drawRoundRect ( 2, 1, containerWidth - 5, getHeight () - 2, 16, 16 );
            } else {
                if (parentFocused) {
                    g.setColor(selected ? getOurBackgroundSelectionColor() : getOurBackgroundNonSelectionColor());
                    g.fillRect( isTree() ? x : 0, 0, containerWidth - ((isTree() ? x : 0) + 1), getHeight () - 1);
                    if (selected) {
                        Color borderSel = getBorderSelectionColor();
                        if (borderSel != null) {
                            g.setColor (borderSel);
                            if (isTree()) {
                                g.drawRect (x, 0, getWidth()-(x + 1), getHeight() -1);
                            } else {
                                g.drawRect ( 0, 0, containerWidth - 1, getHeight () - 1 );
                            }
                        }
                    }
                } else {
                    Color bgNonSelection = getOurBackgroundNonSelectionColor();
                    g.setColor(selected ? getUnfocusedSelectionBackground(bgNonSelection) : bgNonSelection);
                    g.fillRect( isTree() ? x : 0, 0, containerWidth - ((isTree() ? x : 0) + 1), getHeight () - 1);
                }
            }

            if ( ic != null ) {
                ic.paintIcon ( this, g, ins.left, ins.top );
            }
            // text
            g.setFont ( getFont () );
            FontMetrics fm = g.getFontMetrics ();
            if ( widthChanged ) {
                calcCharCount ( g, fm );
            }
            int fheight = g.getFontMetrics ().getHeight ();
            int y = ins.top + g.getFontMetrics ().getMaxAscent ();
            int h = getHeight () - ( ins.top + ins.bottom );
            if ( fheight < h ) {
                y += ( h - fheight ) / 2;
            }
            if (parentFocused) {
                g.setColor(selected ? getOurTextSelectionColor() : getOurTextNonSelectionColor());
            } else {
                g.setColor(selected ? getUnfocusedSelectionForeground() : getOurTextNonSelectionColor());
            }
            
            int cc = shouldAbbreviate() ? charCount : Integer.MAX_VALUE;

            str.getPainter ().paint ( g, x, y, str, cc, !isSelected () );
        }
        setRelatedItemProvider ( null );
        setRelated ( false );
        setPrimary ( false );
    }
    
    /** @return true when abbreviations should be used for this renderer,
     * false otherwise
     */
    private boolean shouldAbbreviate () {
        if (isTree()) {
            return false;
        }
        return Boolean.getBoolean("navigator.string.abbrevs");
    }
    
    private Color getOurBackgroundSelectionColor () {
        Color result = getBackgroundSelectionColor();
        if (result == null) {
            result = Color.blue;
        }
        return result;
    }
    
    private Color getOurBackgroundNonSelectionColor () {
        Color result = getBackgroundNonSelectionColor();
        if (result == null) {
            result = Color.white;
        }
        return result;
    }
    
    private Color getOurTextSelectionColor () {
        Color result = getTextSelectionColor();
        if (result == null) {
            result = Color.white;
        }
        return result;
    }
    
    private Color getOurTextNonSelectionColor () {
        Color result = getTextNonSelectionColor();
        if (result == null) {
            result = Color.black;
        }
        return result;
    }
    
    /** Get the system-wide unfocused selection background color */
    private static Color getUnfocusedSelectionBackground(final Color bgNonSelection) {
        if (unfocusedSelBg == null) {
            //allow theme/ui custom definition
            unfocusedSelBg = UIManager.getColor("nb.explorer.unfocusedSelBg"); //NOI18N
            if (unfocusedSelBg == null) {
                //try to get standard shadow color
                unfocusedSelBg = UIManager.getColor("controlShadow"); //NOI18N
                if (unfocusedSelBg == null) {
                    //Okay, the look and feel doesn't suport it, punt
                    unfocusedSelBg = Color.lightGray;
                }
                //Lighten it a bit because disabled text will use controlShadow/
                //gray
                if (!bgNonSelection.equals(unfocusedSelBg.brighter())) { // NOI18N
                    unfocusedSelBg = unfocusedSelBg.brighter();
                }
            }
        }
        return unfocusedSelBg;
    }

    /** Get the system-wide unfocused selection foreground color */
    private static Color getUnfocusedSelectionForeground() {
        if (unfocusedSelFg == null) {
            //allow theme/ui custom definition
            unfocusedSelFg =
            UIManager.getColor("nb.explorer.unfocusedSelFg"); //NOI18N
            if (unfocusedSelFg == null) {
                //try to get standard shadow color
                unfocusedSelFg = UIManager.getColor("textText"); //NOI18N
                if (unfocusedSelFg == null) {
                    //Okay, the look and feel doesn't suport it, punt
                    unfocusedSelFg = Color.BLACK;
                }
            }
        }
        return unfocusedSelFg;
    }
    
    public void firePropertyChange (String s, Object a, Object b) {
        //do nothing - performance
    }

    private static final Icon WAIT_ICON = new ImageIcon ( Utilities.loadImage (
            "org/netbeans/modules/java/navigation/resources/wait.gif" ) ); //NOI18N
}
