/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.navigation.jmi;

import java.lang.ref.WeakReference;
import javax.swing.SwingUtilities;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.java.JavaDataObject;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.openide.*;
import org.openide.filesystems.FileObject;
import org.openide.loaders.*;
import org.openide.text.PositionBounds;
import org.openide.util.*;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;
import java.lang.reflect.Modifier;
import java.util.*;
import java.util.List;
import org.openide.util.RequestProcessor;


/**
 * Utility methods for handling java classes.
 *
 * @author Tim Boudreau
 */
public final class JUtils {
    //Fallback icon
    private static final Icon defIcon = new ImageIcon ( Utilities.loadImage (
            "org/netbeans/modules/java/navigation/resources/public.gif" ) ); //NOI18N

    //Map for icons
    private static final HashMap icons = new HashMap ( 15 );

    //Weights to use in constructing weighted strings
    private static final float NAME_IMPORTANCE = 1.0f;
    private static final float PARAMS_PAREN_IMPORTANCE = 0.8f;
    private static final float PARAMETER_TYPE_IMPORTANCE = 0.25f;
    private static final float PARAMETER_NAME_IMPORTANCE = 0.1f;
    private static final float FIELD_TYPE_IMPORTANCE = 0.3f;

    /** constants for javadoc htmlization */
    private static final int IN_DESCRIPTION = 0;
    private static final int IN_PARAM = 1;
    private static final int IN_RETURN = 2;
    private static final int IN_THROWS = 3;
    private static final int IN_OTHER = 99;
    

    /**
     * If true, dragging an inner class member to an outer class will
     * ask the user if they want to move the inner class. 
     */
    private static boolean askMoveInnerClass = true;
    
    /** JMI repository to get data about class we represent */
    private static final MDRepository repo = JavaModel.getJavaRepository();
    
    /** holds last object for which the tooltip was built */
    private static WeakReference lastTooltipForRef;
    /** task that calculates tooltip */
    private static RequestProcessor.Task tooltipTask;
    /** holds last tooltip text */
    private static String lastTooltip;
    /** data lock for tooltip calculations */
    private static final Object TOOLTIP_DATA_LOCK = new Object();
    

    /**
     * Never instantiate
     */
    private JUtils () {
    }


    //---------------------------
    //--- Icon handling >>


    /**
     * Get an icon for an object.  If it is an instance of Wrapper, it will 
     * be dimmed (this is used for inherited methods that are
     * not really present)
     *
     * return icon proper for given object or null when object is not valid
     */
    public static Icon iconFor (Object o) {
        String key = "";
        boolean washOut = false;
        if ( o instanceof Wrapper ) {
            o = ( (Wrapper) o ).getElement ();
            washOut = true;
        }
        // validity check
        if (o instanceof Element && !((Element)o).isValid()) {
            return null;
        }
        if ( o instanceof Method ) {
            Method m = (Method) o;
            boolean stat = ( m.getModifiers () & Modifier.STATIC ) != 0;

            if ( ( m.getModifiers () & Modifier.PUBLIC ) != 0 ) {
                key = stat ? "methodStPublic" : "methodPublic"; //NOI18N
            } else if ( ( m.getModifiers () & Modifier.PROTECTED ) != 0 ) {
                key = stat ? "methodStProtected" : "methodProtected"; //NOI18N
            } else if ( ( m.getModifiers () & Modifier.PRIVATE ) != 0 ) {
                key = stat ? "methodStPrivate" : "methodPrivate"; //NOI18N
            } else {
                key = stat ? "methodStPackage" : "methodPackage"; //NOI18N
            }
        } else if ( o instanceof Constructor ) {
            Constructor c = (Constructor) o;

            if ( ( c.getModifiers () & Modifier.PUBLIC ) != 0 ) {
                key = "constructorPublic"; //NOI18N
            } else if ( ( c.getModifiers () & Modifier.PROTECTED ) != 0 ) {
                key = "constructorProtected"; //NOI18N
            } else if ( ( c.getModifiers () & Modifier.PRIVATE ) != 0 ) {
                key = "constructorPrivate"; //NOI18N
            } else {
                key = "constructorPackage"; //NOI18N
            }
        } else if ( o instanceof Field ) {
            Field f = (Field) o;
            boolean stat = ( f.getModifiers () & Modifier.STATIC ) != 0;

            if ( ( f.getModifiers () & Modifier.PUBLIC ) != 0 ) {
                key = stat ? "variableStPublic" : "variablePublic"; //NOI18N
            } else if ( ( f.getModifiers () & Modifier.PROTECTED ) != 0 ) {
                key = stat ? "variableStProtected" : "variableProtected"; //NOI18N
            } else if ( ( f.getModifiers () & Modifier.PRIVATE ) != 0 ) {
                key = stat ? "variableStPrivate" : "variablePrivate"; //NOI18N
            } else {
                key = stat ? "variableStPackage" : "variablePackage"; //NOI18N
            }
        } else if ( o instanceof JavaClass ) {
            JavaClass jc = (JavaClass) o;

            if ( jc.isInterface () ) {
                key = "interface";
            } else {
                key = "class2";
            }
        } else if ( o instanceof DataObject ) {
            key = "class";
        }
        Icon icon = (Icon) icons.get ( washOut ? key + "_w" : key ); //NOI18N
        if ( icon == null ) {
            String name = "org/netbeans/modules/java/navigation/resources/" + key + //NOI18N
                    ".gif"; //NOI18N
            Image img = Utilities.loadImage ( name );
            if ( washOut ) {
                img = washOutImage ( img );
            }
            if ( img != null ) {
                icon = new ImageIcon ( img );
                icons.put ( washOut ? key + "_w" : key, icon ); //NOI18N
            } else {
                icon = defIcon;
            }
        }
        return icon;
    }


    /**
     * Generate a lightened version of an image
     */
    private static Image washOutImage (Image i) {
        Image result = toBufferedImage ( i );
        Graphics2D g = (Graphics2D) result.getGraphics ();
        g.setComposite ( AlphaComposite.getInstance ( AlphaComposite.SRC_ATOP, 0.40f ) );

        g.fillRect ( 0, 0, 16, 16 );
        return result;
    }


    /**
     * Image management junk stolen from IconManager
     */
    private static final Image toBufferedImage (Image img) {
        new javax.swing.ImageIcon ( img );
        java.awt.image.BufferedImage rep =
                createBufferedImage ( img.getWidth ( null ), img.getHeight ( null ) );

        java.awt.Graphics g = rep.createGraphics ();
        g.drawImage ( img, 0, 0, null );
        g.dispose ();
        img.flush ();
        return rep;
    }


    /**
     * More image management junk stolen from IconManager
     */
    private static final BufferedImage createBufferedImage (int width, int height) {

        ColorModel model = GraphicsEnvironment.getLocalGraphicsEnvironment ().
                getDefaultScreenDevice ().
                getDefaultConfiguration ().
                getColorModel ( java.awt.Transparency.BITMASK );

        BufferedImage result = new java.awt.image.BufferedImage ( model,
                model.createCompatibleWritableRaster ( width, height ),
                model.isAlphaPremultiplied (), null );

        return result;
    }


    //--- << Icon handling ------
    //---------------------------
    //--- Element display name handling >>


    /**
     * Construct the display name for a ClassMember or other object that can be displayed in Navigator.
     */
    public static void extractName (WeightedString as, Object o, JavaClass declaringClass, boolean useName) {
        o = unwrap(o);
        if ( o instanceof Constructor ) {
            Constructor c = (Constructor) o;
            extractConstructorName ( as, c, declaringClass, useName );
        } else if ( o instanceof Method ) {
            Method m = (Method) o;
            extractMethodName ( as, m, declaringClass, useName );
        } else if ( o instanceof Field ) {
            Field f = (Field) o;
            extractFieldName ( as, f, declaringClass, useName );
        } else if ( o instanceof Parameter ) {
            Parameter p = (Parameter) o;
            extractParameterName ( as, p, 0 );
        } else if ( o instanceof DataObject ) {
            as.append ( ( (DataObject) o ).getName (), NAME_IMPORTANCE );
        } else if ( o instanceof JavaClass ) {
            JavaClass jc = (JavaClass) o;
            as.append ( jc.getSimpleName (), NAME_IMPORTANCE );
        } else if ( o instanceof NamedElement ) {
            as.append(((NamedElement)o).getName(), NAME_IMPORTANCE);
        } else {
            as.append ( o.toString (), NAME_IMPORTANCE );
        }
    }

    static final boolean EXTRA_DECORATIONS = Boolean.getBoolean("nb.navigator.decorate");

    private static void extractMethodName (WeightedString as, Method m, JavaClass declaringClass, boolean useName) {
        JavaClass parent = useName && m.refImmediateComposite () instanceof JavaClass ?
                (JavaClass) m.refImmediateComposite () : null;

        if ( useName && parent != declaringClass && parent != null ) {
            String s = parent.getSimpleName ();
            if ( s.length () > 0 ) {
                as.startMarkupRun ( as.DEEMPHASIZED );
                as.append ( s + ".", 0.7f ); //NOI18N
                as.endMarkupRun ();
            }
        }

        boolean isDeprecated = m.isDeprecated();
        if (isDeprecated) {
            as.startMarkupRun ( as.STRIKETHROUGH );
        }

// [dafe] XXX - commented out, because it is soooo sloooow
// I'll have to find another way, perhaps doing this always lazily later        
        if (EXTRA_DECORATIONS) {
            if ( Hacks.isOverride ( m ) ) {
                as.startMarkupRun ( as.BOLD );
            }
            if (( m.getModifiers() & Modifier.FINAL ) != 0) {
                as.startMarkupRun ( as.ITALIC );
            }
        }

        as.append ( m.getName (), NAME_IMPORTANCE );
        
        // parameters - always include parenthesis, even if params empty
        as.append ( "(", PARAMS_PAREN_IMPORTANCE ); //NOI18N
        
        List params = m.getParameters ();
        if ( !params.isEmpty () ) {
            for ( Iterator i = params.iterator (); i.hasNext (); ) {
                extractParameterName ( as, (Parameter) i.next (), isDeprecated ? as.STRIKETHROUGH : 0 );
                if ( i.hasNext () ) {
                    as.append ( ",", PARAMETER_TYPE_IMPORTANCE + 0.1f ); //NOI18N
                }
            }
        }
        
        as.append ( ")", PARAMS_PAREN_IMPORTANCE ); //NOI18N
        
    }

    private static void extractFieldName (WeightedString as, Field f, JavaClass declaringClass, boolean useName) {

        JavaClass parent = useName && f.refImmediateComposite () instanceof JavaClass ?
                (JavaClass) f.refImmediateComposite () : null;

        if ( useName && parent != null && parent != declaringClass ) {
            String s = parent.getSimpleName ();
            as.startMarkupRun ( as.DEEMPHASIZED );
            as.append ( s + ".", 0.7f ); //NOI18N
            as.endMarkupRun ();
        }

        if ( f.isDeprecated () ) {
            as.startMarkupRun ( as.STRIKETHROUGH );
        }
        
        as.append ( f.getName (), NAME_IMPORTANCE );
        
        as.startMarkupRun ( as.DEEMPHASIZED );
        if ( f.getType () != null ) {
            StringBuffer sb = new StringBuffer();
            convertToSimpleName(f.getType().getName(), sb);
            as.append ( " " + sb.toString(), FIELD_TYPE_IMPORTANCE ); //NOI18N
        }
    }


    private static void extractConstructorName (WeightedString as, Constructor c, JavaClass declaringClass,
                                               boolean useName) {
        if ( c.getDeclaringClass () == null ) {
            //For mysterious reasons, this can happen.
            return;
        }
        JavaClass parent = useName && c.refImmediateComposite () instanceof JavaClass ?
                (JavaClass) c.refImmediateComposite () : null;

        if ( useName && parent != null && parent != declaringClass ) {
            as.startMarkupRun ( as.DEEMPHASIZED );
            ClassDefinition decl = c.getDeclaringClass ();
            String name = decl instanceof JavaClass ?
                    ( (JavaClass) decl ).getSimpleName () : decl.getName ();

            append ( as, name + ".", '.', 0.7f ); //NOI18N
            as.endMarkupRun ();
        }

        List l = c.getParameters ();
        boolean isDeprecated = c.isDeprecated();
        if ( isDeprecated ) {
            as.startMarkupRun ( as.STRIKETHROUGH );
        }
        append ( as, c.getDeclaringClass ().getName (), '.', NAME_IMPORTANCE ); //NOI18N

        // parameters - always include parenthesis, even if params empty
        as.append ( "(", PARAMS_PAREN_IMPORTANCE ); //NOI18N
        if ( !l.isEmpty () ) {
            for ( Iterator i = l.iterator (); i.hasNext (); ) {
                extractParameterName ( as, (Parameter) i.next (), isDeprecated ? as.STRIKETHROUGH : 0 );
                if ( i.hasNext () ) {
                    as.append ( ",", PARAMETER_TYPE_IMPORTANCE + 0.1f ); //NOI18N
                }
            }
        }
        as.append ( ")", PARAMS_PAREN_IMPORTANCE ); //NOI18N
    }

    /**
     * Extract the name of a parameter, with appropriate weighting, into a WeightedString.
     */
    private static void extractParameterName (WeightedString as, Parameter p, int mask) {
        if (!p.isValid()) {
            return;
        }
        as.startMarkupRun(as.DEEMPHASIZED | mask);
        as.append(getParameterType(p), PARAMETER_TYPE_IMPORTANCE);
        if (p.isVarArg()) {
            as.append(" ...", PARAMETER_NAME_IMPORTANCE);
        }
        String name = p.getName();
        if (name != null && !"".equals(name)) {
            as.append(" " + name, PARAMETER_NAME_IMPORTANCE);
        }
        as.endMarkupRun();
    }

    
    /** Returns string describing given parameter's type (in short form)
     */
    private static String getParameterType (Parameter p) {
        // try typename first
        TypeReference typeRef = p.getTypeName();
        if (typeRef != null) {
            return typeRef.getName();
        }
        // when source not available, we have to use following:
        String fullName = p.getType().getName();
        int lastDot = fullName.lastIndexOf('.');
        if (lastDot > 0) {
            return fullName.substring(lastDot + 1);
        }
        return fullName;
    }
    

    /**
     * Convenience method to strip a string to the last occurance of some character, and append it to a WeightedString
     */
    private static void append (WeightedString as, String target, char last, float importance) {
        int start = target.lastIndexOf ( last ) + 1;
        if ( start != -1 && start < target.length () ) {
            char[] c = new char[ target.length () - start ];
            target.getChars ( start, target.length (), c, 0 );
            as.append ( c, importance );
        } else {
            as.append ( target, importance );
        }
    }



    //--- << Element display name handling ------
    //-------------------------------------------
    //--- Element tooltip generation >>

    /**
     * Generate a tooltip for a class member we want to display.
     * For performance reasons, only text of given weighted string is returned
     * if full tooltip with javadoc is not precalculated. Full tooltip is
     * calculated in background thread and tooltip is refreshed after calculation
     * is done.
     */
    public static String getTooltip (WeightedString wString, Object tooltipFor,
                                     JavaClass jc, int x, int y, TipHackInvoker invoker) {
        
        synchronized (TOOLTIP_DATA_LOCK) {
            Object lastTooltipFor = lastTooltipForRef == null ? null : lastTooltipForRef.get();
            if (lastTooltipFor == tooltipFor) {
                // tooltip already calculated and cached
                if (lastTooltip != null) {
                    return lastTooltip;
                }
                // no further activity, because tooltip is just being calculated
                return "";
            }
            // cancel previous now invalid task
            if (tooltipTask != null) {
                boolean cancelled = tooltipTask.cancel();
                tooltipTask = null;
            }
            lastTooltipForRef = new WeakReference(tooltipFor);
            // clear previous result
            lastTooltip = null;
        }
        // start full tooltip calculation in request processor
        TooltipCalculator tc = new TooltipCalculator(tooltipFor, jc, x, y, invoker);
        synchronized (TOOLTIP_DATA_LOCK) {
            tooltipTask = RequestProcessor.getDefault().post(tc);
        }
        
        return "";
    }

    /** calculates tooltip and invokes tooltip refresh */
    private static class TooltipCalculator implements Runnable {
        
        private Object tooltipFor;
        private JavaClass jc;
        private int x, y;
        private TipHackInvoker invoker;
        
        TooltipCalculator (Object tooltipFor, JavaClass jc, int x, int y,
                            TipHackInvoker invoker) {
            this.tooltipFor = tooltipFor;
            this.jc = jc;
            this.x = x;
            this.y = y;
            this.invoker = invoker;
        }
        
        /** actually calculates tooltip for given item */
        public void run () {
            String result;
            repo.beginTrans(false);
            try {
                result = getTooltip(tooltipFor, jc);
            } finally {
                repo.endTrans(false);
            }
            
            synchronized (TOOLTIP_DATA_LOCK) {
                tooltipTask = null;
                // cancel if not needed (tooltip for another object was requested later)
                if ((lastTooltipForRef != null) && (lastTooltipForRef.get() != tooltipFor)) {
                    return;
                }
                lastTooltip = result;
            }
            // invoke tooltip
            SwingUtilities.invokeLater(new Runnable () {
                public void run () {
                    invoker.invokeTip(x, y);
                }
            });
        }
    }
    
    /** Interface for async invoke of tooltip set hack. Really stupid,
     * we should get rid of tooltips and favor regular popup window ASAP.
     */
    public interface TipHackInvoker {
        
        public void invokeTip (int x, int y);
        
    }
    
    /**
     * Generate a tooltip for a class member we want to display
     */
    // XXX - make private
    public static String getTooltip (Object o, JavaClass jc) {
        Object maybeWrapper = o;
        o = unwrap(o);
        // validity check
        if (o instanceof Element && !((Element)o).isValid()) {
            return null;
        }
        
        StringBuffer sb = new StringBuffer ( 50 );
        if ( o instanceof Method ) {
            createMethodTooltip ( (Method) o, sb, jc );
        } else if ( o instanceof Field ) {
            createFieldTooltip ( (Field) o, sb, jc );
        } else if ( o instanceof Constructor ) {
            createConstructorTooltip ( (Constructor) o, sb, jc );
        } else if ( o instanceof DataObject ) {
            FileObject fob = ((DataObject) o).getPrimaryFile();
            if (fob != null) {
                return fob.getPath();
            } else {
                return ((DataObject) o).getName();
            }
        } else if ( o instanceof JavaClass ) {
            createClassTooltip((JavaClass)o, sb, jc);
        }
        if (o instanceof ClassMember) {
            Object comp = ((ClassMember) o).refImmediateComposite();
            if (comp instanceof JavaClass) {
                sb.append (" on " + ((JavaClass) comp).getName());
            }
        }
        return sb.toString ();
    }

    /**
     * Appends a message to a tooltip about what inner class the method occurs on, if any.
     */
    private static void includeInnerMessage (org.netbeans.jmi.javamodel.Element e, StringBuffer sb, JavaClass jc) {
        JavaClass parent = e.refImmediateComposite () instanceof JavaClass ?
                ( (JavaClass) e.refImmediateComposite () ) : null;
        if ( parent != null && parent.isInner () ) {
            StringBuffer clzz = new StringBuffer ();
            while ( parent != jc && parent != null ) {
                clzz.insert ( 0, parent.getSimpleName () );
                parent = parent.refImmediateComposite () instanceof JavaClass ?
                        ( (JavaClass) parent.refImmediateComposite () ) : null;
            }
            sb.append ( NbBundle.getMessage ( JUtils.class, "FMT_InnerClassTip", //NOI18N
                    clzz ) );
            sb.append ( "<br>" ); //NOI18N
        }
    }
    
    /**
     * Appends a message to a tooltip about what a method overrides.
     */
    private static void includeOverrideMessage (Method m, StringBuffer sb) {
        if ( Hacks.isOverride ( m ) ) {
            Collection c = JavaModelUtil.getOverriddenMethods(m);
            if (c == null) {
                return;
            }
            JavaClass superClass = m.getDeclaringClass().getSuperClass();
            if (superClass == null) {
                return; 
            }
            Method orig = superClass.getMethod(m.getName(), getParameterTypeList(m), false);
            boolean isImplementing = orig == null || 
                                    (orig.getModifiers() & Modifier.ABSTRACT) != 0;

            StringBuffer names = new StringBuffer(30);
            
            for (Iterator i=c.iterator(); i.hasNext();) {
                Method me = (Method) i.next();
                ClassDefinition cd = me.getDeclaringClass();
                names.append (cd.getName());
                if (i.hasNext()) {
                    names.append (", ");
                }
            }
            
            sb.append ( "<br><hr>" ); //NOI18N

            sb.append ( NbBundle.getMessage ( JUtils.class, isImplementing ?
                    "FMT_ImplementsTip" : "FMT_OverrideTip", //NOI18N
                    new Object[]{m.getName (), names.toString() } ) );
        }
    }
    
    
    /* XXX [dafe] copied from editor/NbJMICompletionJavadoc.
     * Should be merged in future when we will use one popup component
     * both for editor and navigator javadoc
     */
    private static void includeDocumentedAnnotations(AnnotableElement element, StringBuffer sb) {
        for (Iterator it = element.getAnnotations().iterator(); it.hasNext();) {
            Annotation ann = (Annotation)it.next();
            AnnotationType type = ann.getType();
            if (type != null) {
                type = (AnnotationType)getSourceForBinary(type);
                for (Iterator itt = type.getAnnotations().iterator(); itt.hasNext();) {
                    AnnotationType meta = (AnnotationType)getSourceForBinary(((Annotation)itt.next()).getType());
                    if (meta != null && "java.lang.annotation.Documented".equals(meta.getName())) { // NOI18N
                        sb.append("@" + getTypeName(type)); // NOI18N
                        List values = ann.getAttributeValues();
                        if (!values.isEmpty()) {
                            try {
                                StringBuffer vsb = new StringBuffer();
                                for (Iterator ittt = values.iterator(); ittt.hasNext();) {
                                    AttributeValue av = (AttributeValue)ittt.next();
                                    vsb.append(av.getDefinition().getName());
                                    vsb.append('='); //NOI18N
                                    PositionBounds bounds = JavaMetamodel.getManager().getElementPosition(av.getValue());
                                    if (bounds != null) {
                                        vsb.append(bounds.getText());
                                    } else {
                                        vsb = null;
                                        break;
                                    }
                                    if (ittt.hasNext())
                                        vsb.append(", "); //NOI18N
                                }
                                if (vsb != null) {
                                    sb.append('('); // NOI18N
                                    sb.append(vsb);
                                    sb.append(')'); // NOI18N
                                }
                            } catch (Exception ex) {
                                // ignoring, better to not include part of the
                                // massage then fail
                            }
                        }
                        sb.append("<br>"); // NOI18N
                    }
                }
            }
        }
    }
    
   
    private static String getTypeName(Type typ) {
        String result = null;
        if (typ instanceof Array) {
            result = getTypeName(((Array)typ).getType()) + "[]"; // NOI18N
        }
        if (typ instanceof AnnotationType || (typ instanceof ParameterizedType &&
                ((ParameterizedType)typ).getDefinition() instanceof AnnotationType)) {
            result = "@" + ((JavaClass)typ).getSimpleName(); // NOI18N
        }
        if (typ instanceof JavaClass) {
            result = ((JavaClass)typ).getSimpleName();
        }
        result = result.replaceAll("<", "&lt;"); // NOI18N
        return result.replaceAll(">", "&gt;"); // NOI18N
    }
    
    // end of code copied from editor/NbJMICompletionJavadoc
    


    /**
     * Gets the list of parameter Types from a method, from its list of Parameters.
     */
    private static List getParameterTypeList (Method m) {
        List params = new ArrayList ();

        for ( Iterator i = m.getParameters ().iterator ();
              i.hasNext ();
              params.add ( ( (Parameter) i.next () ).getType () ) )
            ;

        return params;
    }

    private static void createMethodTooltip (Method m, StringBuffer sb, JavaClass jc) {
        sb.append ( "<html>" ); //NOI18N

        getModifiers ( m.getModifiers (), sb );
        if ( m.getType ().getName () != null ) {
            htmlizeTypeName ( m.getType ().getName (), sb, false );
        } else {
            htmlizeTypeName ( "void", sb, false );
        }
        sb.append ( ' ' ); //NOI18N
        sb.append ( "<b><font color=#7C0000>" ); //NOI18N
        sb.append ( m.getName () );
        sb.append ( "</b></font>" ); //NOI18N

        appendParameterList ( m.getParameters (), sb );
        
        sb.append ( "<br>" ); //NOI18N
        
        List annots = m.getAnnotations();
        if (annots != null && annots.size() > 0) {
            includeDocumentedAnnotations(m, sb);
        }

        if ( jc != null ) {
            includeInnerMessage ( m, sb, jc );
        }

        String javadoc = m.getJavadocText ();
        if (javadoc == null || javadoc.length() == 0) {
            //#51450 - if no javadoc, look for super method with javadoc
            Method method4javadoc = findSuperWithJavadoc (m);
            if (method4javadoc != null) {
                javadoc = method4javadoc.getJavadocText();
            }
        }
        if ( javadoc != null && javadoc.length () > 0 ) {
            sb.append ( "<hr>" ); //NOI18N
            sb.append ( htmlize ( javadoc ) );
        }
        includeOverrideMessage ( m, sb );
        sb.append ( "</html>" ); //NOI18N
    }
    
    /**
     * If a method has no javadoc, tries to find a super or interface method
     * that does, for generating tooltips. 
     */
    private static Method findSuperWithJavadoc (Method m) {
        Object o = m.refImmediateComposite();
        if (o instanceof JavaClass) {
            JavaClass jc = (JavaClass) o;
            JavaClass parent = jc.getSuperClass();
            if (parent != null) {
                Method parM = parent.getMethod(m.getName(), 
                        obtainParamTypes(m.getParameters()), true);
                if (parM != null) {
                    String txt = parM.getJavadocText();
                    if (txt != null && txt.trim().length() > 0) {
                        return parM;
                    } else {
                        return findSuperWithJavadoc (parM);
                    }
                }
            }
            for (Iterator i= jc.getInterfaces().iterator(); i.hasNext();) {
                Object o1 = i.next();
                if (o1 instanceof JavaClass) {
                    JavaClass iface = (JavaClass) o1;
                    Method inM = iface.getMethod(m.getName(), 
                            obtainParamTypes(m.getParameters()), true);
                    if (inM != null) {
                        String s = inM.getJavadocText();
                        if (s != null && s.length() > 0) {
                            return inM;
                        }
                    }
                }
            }
        }
        return null;
    }
    
    /** Gets List<org.netbeans.jmi.javamodel.Parameter> and converts it to the list of their types.
     * @return List<org.netbeans.jmi.javamodel.Type>
     */
    private static List obtainParamTypes (List params) {
        int paramCount = params.size();
        if (paramCount == 0) {
            return params;
        }
        List result = new ArrayList(paramCount);
        for (Iterator iter = params.iterator(); iter.hasNext(); ) {
            result.add(((Parameter)iter.next()).getType());
        }
        return result;
    }

    /**
     *  Takes a list of Parameter objects, htmlizes them and inserts them in
     *  a StringBuffer for constructing a tooltip.
     *
     *  @param l A list of Parameters
     *  @param sb A StringBuffer
     */
    private static void appendParameterList (List l, StringBuffer sb) {
        sb.append ( " ( " ); //NOI18N
        int paramsLen = sb.length () - 61;
        HashSet used = null;
        for ( Iterator i = l.iterator (); i.hasNext (); ) {
            Parameter p = (Parameter) i.next ();
            String tname = p.getType ().getName ();
            paramsLen += tname.length ();
            if ( paramsLen + tname.length () + ( p.getTypeName () == null ? 0 : p.getTypeName ().getName ().length () ) > 120 ) {
                sb.append ( "<br>&nbsp;&nbsp;   " ); //NOI18N
                paramsLen = 0;
            }

            htmlizeTypeName ( tname, sb, false );
            sb.append ( ' ' );
            if ( p.getName () != null ) {
                sb.append ( "<font color=#B200B2>" ); //NOI18N
                sb.append ( p.getName () );
                sb.append ( "</font>" ); //NOI18N
                paramsLen += p.getName ().length ();
            } else {
                if ( used == null ) {
                    used = new HashSet ();
                }
                String n = generateBelievableParameterName ( p, used );
                sb.append ( "<font color=#B200B2>" ); //NOI18N
                sb.append ( n );
                sb.append ( "</font>" ); //NOI18N
                paramsLen += n.length ();
            }
            if ( i.hasNext () ) {
                sb.append ( ", " ); //NOI18N
                paramsLen += 2;
            }
        }
        sb.append ( " )" ); //NOI18N
    }


    /**Tries to separate the class name from the fully qualified name. Does not handle generics.
     *
     *@param fullyQualifiedName a fully qualified name to separate
     *@param appendTo a StringBuffer where the result should be appended
     */
    /*package private for tests*/ static void separateClassName(String fullyQualifiedName, StringBuffer appendTo) {
        int lastDot = fullyQualifiedName.lastIndexOf('.');
        
        if (lastDot != (-1)) {
            appendTo.append(fullyQualifiedName.substring(lastDot + 1));
        } else {
            appendTo.append(fullyQualifiedName);
        }
    }
    
    /**Tries convert a fully qualified class name into the simple class.
     * Handles generics.
     *
     *@param fullyQualifiedName a fully qualified name to process
     *@param appendTo a StringBuffer where the result should be appended
     */
    /*package private for tests*/ static void convertToSimpleName(String fullyQualifiedName, StringBuffer appendTo) {
         int genericsStart = fullyQualifiedName.indexOf('<');

	 if (genericsStart == (-1)) {
             // no generics...
	     separateClassName(fullyQualifiedName, appendTo);
             return;
         }
         
         // #65975: defensive, don't try to extract generics if syntax is incorrect 
         if (fullyQualifiedName.charAt(fullyQualifiedName.length() - 1) != '>') {
             separateClassName(fullyQualifiedName, appendTo);
             return;
         }

         // hadle generics
         separateClassName(fullyQualifiedName.substring(0, genericsStart), appendTo);
         appendTo.append("<");

         int paramStart = genericsStart + 1;

         while (paramStart < fullyQualifiedName.length()) {
             int paramEnd = paramStart;
             int balance = 0;

             while (paramEnd < fullyQualifiedName.length() && (fullyQualifiedName.charAt(paramEnd) != ',' || balance > 0)) {
                 switch (fullyQualifiedName.charAt(paramEnd)) {
                     case '<':
                         balance++;
                         break;
                     case '>':
                         balance--;
                         break;
                 }
                 paramEnd++;
             }

             String genericString = fullyQualifiedName.substring(paramStart, paramEnd);
             int space = genericString.lastIndexOf(' ');

             if (space == (-1)) {
                 convertToSimpleName(genericString, appendTo);
             } else {
                 appendTo.append(genericString.substring(0, space));
                 appendTo.append(' ');
                 convertToSimpleName(genericString.substring(space + 1), appendTo);
             }

             paramStart = paramEnd + 1;

             if (paramStart < fullyQualifiedName.length())
                 appendTo.append(", "); //NOI18N
	 }
    }
    
    /**
     *  Takes a type name and adds color highlighting to the segment after the
     * last "."
     *  @param s 
     *  @param sb 
     *  @param highlight 
     */
    private static void htmlizeTypeName (String s, StringBuffer sb, boolean highlight) {
        if (highlight)
            sb.append ( "<b><font color=#77000>" ); //NOI18N
	    
        int start = sb.length();
        
	convertToSimpleName(s, sb);
        
        while (start < sb.length()) {
            switch (sb.charAt(start)) {
                case '<':
                    sb.replace(start, start + 1, "&lt;");
                    break;
                case '>':
                    sb.replace(start, start + 1, "&gt;");
                    break;
            }
            start++;
        }
	
	if (highlight)
            sb.append ( "</font></b>" ); //NOI18N
    }

    /**
     * Parses javadoc text and formats it for use in tooltips.
     * Recognizes the <code>@param</code>, <code>@return</code> and
     * <code>@throws</code> javadoc tags and formats them. Other javadoc
     * tags are ignored.
     *
     *  @param s Plain javadoc text
     */
    private static String htmlize (String s) {
        StringBuffer sb = new StringBuffer ( 256 );
        StringBuffer paragraph = new StringBuffer();
        // javadoc starts with a description
        int state = IN_DESCRIPTION;
        // parse by lines
        StringTokenizer t = new StringTokenizer(s, "\n");  //NOI18N
        while (t.hasMoreTokens()) {
            String line = t.nextToken().trim();
            line = extractLinks(line);
            // lines that don't start with '@' are simply appended to the previous text
            if (!line.startsWith("@")) {  //NOI18N
                paragraph.append(line + " ");   //NOI18N
                continue;
            }
            // new tag is found, wrap and flush the previous paragraph
            // and clear the buffer
            wrapText(paragraph.toString(), sb);
            paragraph.delete(0, paragraph.length());
            // extract the javadoc tag
            int pos = 1;
            while (pos < line.length() && Character.isJavaIdentifierPart(line.charAt(pos)))
                pos++;
            // no tag? skip
            if (pos == 1)
                continue;
            // recognize the tag and remember the current context
            int newState = IN_DESCRIPTION;
            String tag = line.substring(1, pos);
            if (tag.equals("param"))   //NOI18N
                newState = IN_PARAM;
            else if (tag.equals("return"))  //NOI18N
                newState = IN_RETURN;
            else if (tag.equals("throws") || tag.equals("exception"))  //NOI18N
                newState = IN_THROWS;
            else
                newState = IN_OTHER;
            
            // produce definition list html for known javadoc tags
            if (state == IN_DESCRIPTION && state != IN_OTHER) {
                sb.append("<dl>");     //NOI18N
            }
            // if changing context, generate the section header (definition term)
            if (newState != state) {
                sb.append("<dt><b>");      //NOI18N
                switch (newState) {
                    case IN_PARAM:
                        sb.append(NbBundle.getMessage ( JUtils.class, "FMT_ParametersTerm" ) + ":");  //NOI18N
                        break;
                    case IN_RETURN:
                        sb.append(NbBundle.getMessage ( JUtils.class, "FMT_ReturnsTerm" ) + ":");  //NOI18N
                        break;
                    case IN_THROWS:
                        sb.append(NbBundle.getMessage ( JUtils.class, "FMT_ThrowsTerm") + ":");  //NOI18N
                        break;
                    default:
                        break;
               }
               
                sb.append("</b>");     //NOI18N
                sb.append("<dd>");     //NOI18N
            } else if (newState != IN_DESCRIPTION && newState != IN_OTHER) {
                sb.append("<dd>");     //NOI18N
            }

            // Do special, standard doclet-like formatting for params javadoc.
            // Display the parameter names as &lt;code&gt; and append '-'.
            // (Might look better in plain text.)
            if (newState == IN_PARAM) {
                paragraph.append("<code>");   //NOI18N
                // Bug #68241: handle param declarations without any specifiers
                int pos2 = Math.min(pos + 1, line.length());
                while (pos2 < line.length() && Character.isWhitespace(line.charAt(pos2)))
                    pos2++;
                pos = pos2;
                while (pos2 < line.length() && Character.isJavaIdentifierPart(line.charAt(pos2)))
                   pos2++;
                paragraph.append(line.substring(pos, pos2));
                paragraph.append("</code> - ");   //NOI18N
                pos = pos2;
            }
            // Append the tag description to the buffer. Ignore the description
            // of unrecognized javadoc tags.
            if (newState != IN_OTHER) {
                paragraph.append(line.substring(pos) + " ");    //NOI18N
            }
            state = newState;
        }
        // wrap and flush the last paragraph
        wrapText(paragraph.toString(), sb);
        if (state !=IN_DESCRIPTION) {
            sb.append("</dl>");     //NOI18N
        }
        
        return sb.toString ();
    }
            
    /**
     * Accepts javadoc text and inserts &lt;br&gt; to line wrap it reasonably
     * for use in tooltips. 
     */
    private static void wrapText(String s, StringBuffer sb) {
        int pos = 0;
        int htmlBreakPos, nextBreak, tokenLength;
        StringTokenizer t = new StringTokenizer(s, " \t\r\n");  //NOI18N
        while (t.hasMoreTokens()) {
            String w = t.nextToken().toLowerCase();
            tokenLength = w.length();
            // TODO: ignore html tags when calculating the line length
            htmlBreakPos = findHtmlBreakPos(w);
            nextBreak = htmlBreakPos == -1 ? tokenLength : htmlBreakPos;
            if (pos + nextBreak > 90) {
                sb.append("<br>");      //NOI18N
                pos = 0;
            }
            sb.append(w + " ");     //NOI18N
            if (htmlBreakPos == -1) {
                pos += tokenLength + 1;
            } else {
                // #69666: handle html breaks that affect wrapping   
                // note that hardcoded number 4 is just estimate of html breaks length
                pos = Math.max(0, tokenLength - nextBreak - 7);
            }
            if (w.endsWith("</br>") || w.endsWith("</p>")) {
                pos = 0;
            }
        }
    }
    
    /** Finds and returns position of html breaks such as &lt;br&gt; or
     * &lt;p&gt; if there are any. <b>Given string must be lower case.</b>
     * @return position of first html break found, or -1 if no html break in
     * given string
     */
    private static int findHtmlBreakPos (String s) {
        int result = s.indexOf("<p>");
        if (result == -1) {
            result = s.indexOf("<br>");
        }
        return result;
    }

    private static void createFieldTooltip (Field f, StringBuffer sb, JavaClass jc) {
        sb.append ( "<html>" ); //NOI18N
        getModifiers ( f.getModifiers (), sb );
        sb.append ( f.getType ().getName () );
        sb.append ( ' ' ); //NOI18N
        sb.append ( "<b><font color=#0000B2>" ); //NOI18N
        sb.append ( f.getName () );
        sb.append ( "</font></b>" ); //NOI18N

        if ( jc != null ) {
            includeInnerMessage ( f, sb, jc );
        }
        String javadoc = f.getJavadocText ();
        if ( javadoc != null && javadoc.length () > 0 ) {
            sb.append ( "<br>" ); //NOI18N
            sb.append ( "<hr>" ); //NOI18N
            sb.append ( htmlize ( javadoc ) );
        }
        sb.append ( "</html>" ); //NOI18N
    }

    private static void createConstructorTooltip (Constructor m, StringBuffer sb, JavaClass jc) {
        String javadoc = m.getJavadocText ();
        sb.append ( "<html>" ); //NOI18N
        getModifiers ( m.getModifiers (), sb );
        sb.append ( "<b><font color=#B28C00>" ); //NOI18N
        sb.append ( ( (JavaClass) m.refImmediateComposite () ).getSimpleName () );
        sb.append ( "</font></b>" ); //NOI18N
        appendParameterList ( m.getParameters (), sb );

        if ( jc != null ) {
            includeInnerMessage ( m, sb, jc );
        }

        if ( javadoc != null && javadoc.length () > 0 ) {
            sb.append ( "<br>" ); //NOI18N
            sb.append ( "<hr>" ); //NOI18N
            sb.append ( htmlize ( javadoc ) );
        }
        sb.append ( "</html>" ); //NOI18N
    }
    
    private static void createClassTooltip(JavaClass source, StringBuffer sb, JavaClass jc) {
        String javadoc = source.getJavadocText ();
        sb.append("<html>"); //NOI18N
        sb.append(source.getName());
        if (javadoc != null && javadoc.length () > 0) {
            sb.append ( "<br>" ); //NOI18N
            sb.append ( "<hr>" ); //NOI18N
            sb.append ( htmlize ( javadoc ) );
        }
        sb.append ( "</html>" ); //NOI18N
    }
    

    /**
     *  Get the modifier text (i.e. "public static final synchronized") for
     * a class member.
     *
     *  @param m The modifiers mask
     *  @param sb A stringbuffer to put the result in
     */
    private static void getModifiers (int m, StringBuffer sb) {
        boolean hasFirst = false;
        if ( ( m & Modifier.PUBLIC ) != 0 ) {
            sb.append ( "public" ); //NOI18N
            hasFirst = true;
        } else if ( ( m & Modifier.PROTECTED ) != 0 ) {
            sb.append ( "protected" ); //NOI18N
            hasFirst = true;
        } else if ( ( m & Modifier.PRIVATE ) != 0 ) {
            sb.append ( "private" ); //NOI18N
            hasFirst = true;
        }
        if ( ( m & Modifier.ABSTRACT ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "abstract" ); //NOI18N
            hasFirst = true;
        }
        if ( ( m & Modifier.STATIC ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "static" ); //NOI18N
            hasFirst = true;
        }
        if ( ( m & Modifier.FINAL ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "final" ); //NOI18N
            hasFirst = true;
        }
        if ( ( m & Modifier.NATIVE ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "native" ); //NOI18N
            hasFirst = true;
        }
        if ( ( m & Modifier.SYNCHRONIZED ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "synchronized" ); //NOI18N
            hasFirst = true;
        }
        if ( ( m & Modifier.TRANSIENT ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "transient" );
            hasFirst = true;
        }
        if ( ( m & Modifier.VOLATILE ) != 0 ) {
            if ( hasFirst ) {
                sb.append ( ' ' ); //NOI18N
            }
            sb.append ( "volatile" ); //NOI18N
            hasFirst = true;
        }
        if ( hasFirst ) {
            sb.append ( ' ' ); //NOI18N
        }
    }



    //--- << Element tooltip generation  ------
    //-----------------------------------------
    //--- >>  Parameter name generation  ------


    /**
     * Generates a better parameter name than "param1" for cases where we need to generate a method signature, but there
     * is no source to get the original parameter name from.
     */
    public static String generateBelievableParameterName (Parameter p, Set used) {
        String tname = p.getType ().getName ();
        String result;
        boolean usedClassName = false;
        if ( tname.endsWith ( "Listener" ) ) { //NOI18N
            result = "l"; //NOI18N
        } else if ( tname.equals ( "java.lang.Object" ) ) { //NOI18N
            result = "o"; //NOI18N
        } else if ( tname.equals ( "java.lang.Class" ) ) { //NOI18N
            result = "clazz"; //NOI18N
        } else if ( tname.endsWith ( "Stream" ) ) { //NOI18N
            result = "stream"; //NOI18N
        } else {
            result = tname.toLowerCase ();
            int bkt = tname.indexOf ( "[" ); //NOI18N
            if ( bkt > 0 ) {
                tname = tname.substring ( 0, bkt - 1 );
            }

            int idx = result.lastIndexOf ( "." ); //NOI18N
            if ( idx != -1 ) {
                result = result.substring ( idx + 1 );
            }
            usedClassName = true;
        }
        if ( result.trim ().length () == 0 ) {
            result = "value"; //NOI18N
        }
        int bkt = result.indexOf ( "[" ); //NOI18N
        if ( bkt > 0 ) {
            result = result.substring ( 0, bkt );
            if ( usedClassName ) {
                result += "s"; //NOI18N
            }
        }
        if ( !Utilities.isJavaIdentifier ( tname ) ) {
            result = new String ( new char[ tname.charAt ( 0 ) ] );
        }
        if ( isPrimitiveTypeName ( result ) ) {
            result = new String ( new char[]{result.charAt ( 0 )} );
        }
        if ( result.length () > 7 ) {
            result = tryMakeAcronym ( result );
        }

        String test = result;
        int revs = 0;
        while ( used.contains ( test ) ) {
            revs++;
            test = result + revs;
        }
        result = test;
        used.add ( result );
        return result;
    }

    /**
     * Determine if a string matches a java primitive type.  Used in generating reasonable variable names.
     */
    public static boolean isPrimitiveTypeName (String typeName) {
        return (
                //Whoa, ascii art!
                "void".equals ( typeName ) || //NOI18N
                "int".equals ( typeName ) || //NOI18N
                "long".equals ( typeName ) || //NOI18N
                "float".equals ( typeName ) || //NOI18N
                "double".equals ( typeName ) || //NOI18N
                "short".equals ( typeName ) || //NOI18N
                "char".equals ( typeName ) || //NOI18N
                "boolean".equals ( typeName ) ); //NOI18N
    }

    /**
     * Try to create an acronym-style variable name from a string - i.e., "JavaDataObject" becomes "jdo".
     */
    private static String tryMakeAcronym (String s) {
        char[] c = s.toCharArray ();
        StringBuffer sb = new StringBuffer ();
        for ( int i = 0; i < c.length; i++ ) {
            if ( Character.isUpperCase ( c[ i ] ) ) {
                sb.append ( c[ i ] );
            }
        }
        if ( sb.length () > 1 ) {
            return sb.toString ().toLowerCase ();
        } else {
            return s;
        }
    }



    //--- << Parameter name generation  -----------
    //---------------------------------------------
    //--- >> Building lists of class members ------


    /**
     * Get all the members of a class.
     *
     * @param dob - A JavaDataObject to search for members on
     * @param recurse - If true, also include all inherited members
     * @param jc - A hack to return the first located JavaClass object
     * @param includeInners - Whether or not to include all members of any inner classes found
     * @return members of a class or null when class is empty.
     */
    public static List getClassMembers (JavaDataObject dob, boolean recurse, JavaClass[] jc, boolean includeInners) {
        List result = null;
        Resource r = Hacks.getResourceForDataObject ( dob );

        JavaClass clazz;
        if ( r != null && r.isValid () ) {
            JavaModel.setClassPath(r);
            boolean moreThanOne = false;
            for ( Iterator i = r.getClassifiers ().iterator (); i.hasNext (); ) {
                clazz = (JavaClass) i.next ();
                if ( jc[ 0 ] == null ) {
                    jc[ 0 ] = clazz;
                }

                if ( result == null ) {
                    result = clazz.getFeatures ();
                } else {
                    if ( !moreThanOne ) {
                        List oldResult = result;
                        result = new ArrayList ();
                        result.addAll ( oldResult );
                        moreThanOne = true;
                    }
                    result.addAll ( clazz.getFeatures () );
                }
                if ( includeInners ) {
                    result = addInnerClassMembers ( result, recurse );
                }
                if ( recurse ) {
                    result = addInheritedMembers ( clazz, result, clazz );
                }
            }
            if (result != null) {
                //MDR can have duplicates, we don't want them
                result = new ArrayList ( new HashSet ( result ) ); 
            }
        }
        return result;
    }

    /**
     * @return members of a class or null when class is empty.
     */
    public static List getClassMembers (JavaClass clazz, boolean recurse, boolean includeInners) {
        JavaModel.setClassPath(clazz.getResource());
        
        List result = clazz.getFeatures ();
        if ( includeInners ) {
            result = addInnerClassMembers ( result, recurse );
        }
        if ( recurse ) {
            result = addInheritedMembers ( clazz, result, clazz );
        }
        if (result != null) {
            //MDR can have duplicates, we don't want them
            result = new ArrayList ( new HashSet ( result ) ); 
        }
        return result;
    }    

    /**
     * Iterate a list of objects, and recursively add all inner class members of any JavaClass objects found.  This
     * method also weeds out Initializer elements, which we don't represent in the GUI.
     *
     * @param l A list which may contain some JavaClass elements
     * @param inherited include inherited class members for all inner classes
     */
    public static List addInnerClassMembers (List l, boolean inherited) {
        //Create a new list - iterator.remove() on the real list of features
        //will literally delete them from the source file!
        if ( !( l instanceof ArrayList ) && !( l instanceof LinkedList ) ) {
            l = new LinkedList ( l );
        }
        ArrayList subs = null;
        for ( Iterator i = l.iterator (); i.hasNext (); ) {
            Object o = i.next ();
            if ( o instanceof Initializer ) {
                i.remove ();
            } else if ( o instanceof JavaClass ) {
                if (subs == null) {
                    subs = new ArrayList();
                }
                subs.add (o);
                i.remove();
            }
        }
        if (subs != null) {
            for (Iterator i=subs.iterator(); i.hasNext();) {
                JavaClass jc = (JavaClass) i.next();
                l = addInnerClassMembers ( jc, l, inherited );
            }
        }
        return l;
    }

    /**
     * Add the inner class members of the passed JavaClass to the passed list (or a copy that will be returned).
     *
     * @param clazz - The class whose inner classes members will be located
     * @param l - A list which may contain the top level members of the class
     * @param recurse - whether or not to also obtain objects for all inherited methods of inner classes
     */
    private static List addInnerClassMembers (JavaClass clazz, List l, boolean recurse) {
        List nue = clazz.getFeatures ();

        for ( Iterator i = nue.iterator (); i.hasNext (); ) {
            Object o = i.next ();
            if ( o instanceof Initializer ) {

            } else if ( o instanceof JavaClass ) {
                l = addInnerClassMembers ( (JavaClass) o, l, recurse );
            } else {
                l.add ( o );
            }
        }
        // XXX [dafe] no inherited members for inner classes, tough to display clearly,
        // methods may be multiplied for complex classes with lot of innerclasses
        // (JTable as worst example)
        // return recurse ? addInheritedMembers ( clazz, l, clazz ) : l;
        return l;
    }

    /**
     * Add member elements to the list for methods, etc. that are inherited but not actually present on the passed
     * class.
     *
     * @param clazz - The class in question
     * @param l - A list that may already contain some other class members
     * @param owner - The class which does not have the members that will be put in the list, but with which they should
     * be associated
     * @return the same list passed in, or another one with the old one's contents and added inner member items
     */
    private static List addInheritedMembers (ClassDefinition clazz, List l, JavaClass owner) {
        if ( !( l instanceof ArrayList ) ) {
            l = new ArrayList ( l );
        }
        ClassDefinition curr = clazz;
        if ((clazz instanceof JavaClass) && !((JavaClass)clazz).isInterface()) {
            // get superclass members (not for interfaces, because in JMI each
            // interface returns java.lang.Object as its superclass)
            ClassDefinition binary = clazz.getSuperClass();
            if (binary == null) {
                return l;
            }
            // try to get source and use it, binary as fallback
            ClassDefinition source = getSourceForBinary(binary);
            curr = source == null ? binary : source;
            addItemMembers(curr, l, owner);
            // recurse to superclass
            addInheritedMembers(curr, l, owner);
        }
        // retrieve members from implemented and super interfaces
        for (Iterator i = curr.getInterfaces().iterator(); i.hasNext(); ) {
            Object o = i.next();
            if (o instanceof JavaClass) {
                addItemMembers((JavaClass)o, l, owner);
                addInheritedMembers((JavaClass) o, l, owner);
            }
        }
        return l;
    }

    /** Adds displayable members of given class/interface into given list */
    private static void addItemMembers (ClassDefinition clazz, List l, JavaClass owner) {
        for ( Iterator i = clazz.getFeatures ().iterator (); i.hasNext (); ) {
            Object o = i.next ();
            if ( isDisplayableInheritedMember ( o, owner ) ) {
                l.add ( new Wrapper ( (Element) o, owner ) );
            }
        }
    }
    

    /** Returns source for given class definition if exists
     * @return source class definition of null if no source can be found
     */
    public static ClassDefinition getSourceForBinary (ClassDefinition clazz) {
        return ((JMManager)JavaMetamodel.getManager()).getSourceElementIfExists(clazz);
    }

    /**
     * Weed out elements from a superclass that aren't useful to show
     * (things that are already overridden, super-super-constructors).
     */
    private static boolean isDisplayableInheritedMember (Object o, JavaClass owner) {
        if (!(o instanceof Method || o instanceof Field ||
            o instanceof Constructor)) {
            return false;
        }
        
        ClassMember cm = (ClassMember)o;
        Object ownerPkg = owner.refImmediatePackage();
        Object memberPkg = cm.refImmediatePackage();
        
        boolean allowPackage = ownerPkg == null && memberPkg == null ||
            ownerPkg != null && ownerPkg.equals ( memberPkg );
        
        int mods = cm.getModifiers();
        
        boolean result = (((mods & Modifier.PUBLIC) != 0) || 
                   ((mods & Modifier.PROTECTED) != 0)) ||
                   (allowPackage && ((mods & Modifier.PRIVATE) == 0)
                   || cm.refImmediateComposite() == owner);
        
        if (result && o instanceof Method) {
            result = !isOverridenAnywhere(owner, (Method)o);
        }
        return result;
    }


    /**
     * We use Wrapper objects to associate a class that does not implement a method with a method it inherits, so that
     * we can have multiple unique items for, e.g., Object.hashCode() as a part of both an outer and an inner class in
     * the same list.
     * <p>
     * This method resolves the underlying element if necessary.
     */
    public static Object unwrap (Object o) {
        if ( o instanceof JUtils.Wrapper ) {
            o = ( (JUtils.Wrapper) o ).getElement ();
        }
        return o;
    }
    
    public static boolean isWrapper (Object o) {
        return o instanceof Wrapper;
    }

    /**
     * Wrapper so we can display inherited class members in a list model.  I.e., given classes A and A.B, you don't want
     * two items "Object.hashCode" in the member list, you want "A.hashCode" and "B.hashCode".  So this class wraps an
     * element and provides a fake declaring class for it.
     */
    private static final class Wrapper {
        final org.netbeans.jmi.javamodel.Element element;
        final JavaClass clazz;

        public Wrapper (org.netbeans.jmi.javamodel.Element element, JavaClass clazz) {
            this.element = element;
            this.clazz = clazz;
        }

        public org.netbeans.jmi.javamodel.Element getElement () {
            return element;
        }

        public JavaClass getOwner () {
            return clazz;
        }

        public String toString () {
            return clazz.getName () + "." + element;
        }

        public int hashCode () {
            return ( clazz.getName ().hashCode () * 31 ) ^ element.hashCode ();//(clazz.hashCode() * 31) ^ element.hashCode();
        }

        public String getClassName () {
            return clazz.getSimpleName ();
        }

        public boolean equals (Object o) {
            if ( o instanceof Wrapper ) {
                Wrapper w = (Wrapper) o;
                return w.element == element && w.clazz == clazz;
            } else {
                return false;
            }
        }
    }


    //--- << Building lists of class members ------
    //---------------------------------------------
    //--- >> Misc utility stuff -------------------


    public static boolean overridesMethod (JavaClass target, Method m) {
        if (target == null) {
            return false;
        }
        List params = new ArrayList ();
        for ( Iterator i = m.getParameters ().iterator (); i.hasNext (); params.add ( ( (Parameter) i.next () ).getType () ) ) ;
        try {
            return target.getMethod ( m.getName (), params, false ) != null;
        } catch (NullPointerException npe) {
            NullPointerException npe2 = new NullPointerException ("NPE: target " + target + " method "  + m);
            ErrorManager.getDefault().annotate (npe2, npe);
            throw npe2;
        }
    }
    
    /** Finds out whether given method is overriden by any class between
     * (and including) given target class and subclass of declaring class of given method.
     * #param m A method in question
     * #param target Class from which the search starts.
     * @return true when method which overrides given method is found,
     * false otherwise.
     */
    private static boolean isOverridenAnywhere (JavaClass target, Method m) {
        JavaClass decClass = (JavaClass)m.getDeclaringClass();
        // prepare param list of method
        List params = m.getParameters();
        List paramTypes = new ArrayList(params.size());
        for (Iterator i = params.iterator(); i.hasNext(); ) {
            paramTypes.add(((Parameter)i.next()).getType());
        }
        String mName = m.getName();
        String decClassName = decClass.getName();
        // go through superclass chain
        for (JavaClass i = target; i != null && !decClassName.equals(i.getName()); i = i.getSuperClass()) {
            if (i.getMethod(mName, paramTypes, false) != null) {
                return true;
            }
        }
        return false;
    }


    /**
     * Move an Element from one place to a position after another element. Can (if ever implemented) involve
     * refactoring.
     * <p>
     * Note: Does not currently handle guarded blocks.
     *
     * @param toMove - an element to move
     * @param after - an element in some class somewhere, which this one should be positioned after
     * @return null if success, a localized message if failure
     */
    public static String move (ClassMember toMove, ClassMember after) {
        if ( toMove.equals ( after ) ) {
            return NbBundle.getMessage ( JUtils.class, "MSG_CanMoveToSelf" ); //NOI18N
        }
        if ( toMove.refImmediateComposite () instanceof JavaClass &&
                after.refImmediateComposite () instanceof JavaClass ) {

            JavaClass toMoveClass = (JavaClass) toMove.refImmediateComposite ();
            JavaClass targetClass = (JavaClass) after.refImmediateComposite ();

            if ( toMoveClass.equals ( targetClass ) ) {
                repositionElementWithinClass ( toMove, after, toMoveClass );
            } else {
                if ( toMoveClass.isInner () &&
                        toMoveClass.refImmediateComposite () == targetClass ) {

                    return askMoveEntireInnerClass ( targetClass, toMoveClass,
                            after, toMove );
                } else {
                    return NbBundle.getMessage ( JUtils.class, "MSG_CantMove" ); //NOI18N
                }
            }
        } else {
            //XXX Sometimes happens, who knows what it is. Come up with a
            //reasonable message
            return NbBundle.getMessage ( JUtils.class, "MSG_CantMove" ); //NOI18N
        }
        return null;
    }

    private static void repositionElementWithinClass (Element toMove, Element after, JavaClass clazz) {
        List features = clazz.getFeatures ();

        int toMoveIdx = features.indexOf ( toMove );
        int afterIdx = features.indexOf ( after );
        if ( toMoveIdx != -1 && afterIdx != -1 ) {
            features.remove ( toMoveIdx );
            if ( afterIdx > toMoveIdx ) {
                afterIdx--;
            }
            features.add ( afterIdx, toMove );
        }
    }

    /**
     *  Retrieves a method or field name, or constructs a constructor name
     * (constructors don't have names).
     *
     *  @param o A ClassMember of some sort
     */
    private static String qualifiedNameOf (Object o) {
        if ( o instanceof Constructor ) {
            return NbBundle.getMessage ( JUtils.class, "FMT_Constructor", //NOI18N
                    ( (Constructor) o ).getDeclaringClass ().getName () );
        } else {
            return ( (Feature) o ).getName ();
        }
    }

    /**
     *  Called when the user attempts to drag an inner class member in between
     * members of the outer class.  Pops up a dialog asking if the whole inner
     * class should be moved to the new spot.
     * 
     *  @param target The class owning the inner class
     *  @param class2move The inner class
     *  @param after The member of the outer class it should be inserted after
     *  @param triedToMove The inner class member that was dragged
     */
    private static String askMoveEntireInnerClass (JavaClass target, JavaClass class2move, Element after,
                                                   Element triedToMove) {
        boolean canMove = !askMoveInnerClass;
        if ( !canMove ) {
            String title = NbBundle.getMessage ( JUtils.class,
                    "TITLE_MoveInnerClass" ); //NOI18N

            String msg = NbBundle.getMessage ( JUtils.class,
                    "MSG_AskMoveInnerClass", new Object[]{//NOI18N
                        qualifiedNameOf ( triedToMove ),
                        class2move.getSimpleName ()} );

            JPanel msgPanel = new JPanel ();
            msgPanel.setLayout ( new BorderLayout () );
            JTextArea jta = new JTextArea ( msg );
            jta.setOpaque ( false );
            jta.setEditable ( false );
            jta.setBorder ( BorderFactory.createEmptyBorder () );
            JCheckBox cb = new JCheckBox ( NbBundle.getMessage ( JUtils.class,
                    "MSG_AlwaysDoThis" ) ); //NOI18N

            msgPanel.add ( jta, BorderLayout.CENTER );
            msgPanel.add ( cb, BorderLayout.SOUTH );

            NotifyDescriptor nd = new NotifyDescriptor ( msgPanel, title,
                    NotifyDescriptor.YES_NO_OPTION,
                    NotifyDescriptor.QUESTION_MESSAGE,
                    null, NotifyDescriptor.YES_OPTION );

            canMove = NotifyDescriptor.YES_OPTION.equals ( 
                DialogDisplayer.getDefault ().notify ( nd ) );

            askMoveInnerClass = cb.isSelected (); //XXX persist

            if ( !canMove ) {
                return ""; //NOI18N
            }
        }

        List feat = target.getContents ();
        int newLocation = feat.indexOf ( after );

        int oldIdx = -1;
        for ( Iterator i = feat.iterator (); i.hasNext (); ) {
            oldIdx++;
            Object oh = i.next ();
            if ( oh.equals ( class2move ) ) {
                break;
            }
            ;
        }
        if ( oldIdx == -1 || oldIdx == feat.size () || newLocation == -1 ) {
            return NbBundle.getMessage ( JUtils.class, "MSG_CouldNotLocateClass", //NOI18N
                    class2move.getSimpleName () );
        }

        feat.remove ( oldIdx );
        try {
            if ( oldIdx < newLocation ) {
                newLocation--;
            }
            feat.add ( newLocation, class2move );
        } catch ( Exception e ) {
            feat.add ( oldIdx, class2move );
            String msg = NbBundle.getMessage ( JUtils.class, "MSG_MoveFailed" ); //NOI18N
            ErrorManager.getDefault ().annotate ( e, msg );
            ErrorManager.getDefault ().notify ( ErrorManager.USER, e );
            return msg;
        }
        return null;
    }

    private static final String LINK_TAG_START = "{@link ";
    private static final char LINK_TAG_END = '}';
            
    private static String extractLinks(String string) {
        int linkIndex = string.indexOf(LINK_TAG_START);
        // do nothing if link not found
        if (linkIndex == -1) {
            return string;
        }
        
        StringBuffer result = new StringBuffer(string.length() + 1);
        int startPos = 0;
        int endPos = -1;
        while (linkIndex != -1) {
            result.append(string.substring(startPos, linkIndex));
            endPos = string.indexOf(LINK_TAG_END, linkIndex);
            if (endPos != -1) {
                // ending '}' found, append link target
                result.append(string.substring(linkIndex + 7, endPos));
                startPos = endPos + 1;
            } else {
                // ending '}' not found, so just ignore '{@link' and step out
                result.append(string.substring(linkIndex + 7));
                break;
            }
            linkIndex = string.indexOf(LINK_TAG_START, endPos);
        }
        // #66530: add last chunk after link tag
        if (endPos != -1 && endPos + 1 < string.length()) {
            result.append(string.substring(endPos + 1));
        }
        return result.toString();
    }

}
