/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * RelatedItemProviderSupport.java
 *
 * Created on September 23, 2004, 10:25 PM
 */

package org.netbeans.modules.java.navigation.spi;

import org.openide.util.*;
import org.openide.windows.*;

import javax.swing.*;
import java.beans.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * Abstract implementation of RelatedItemProvider.  Usage:
 * <p/>
 * Override <code>findPrimaryItem()<code> to return whatever object should appear to be selected in the Navigator GUI.
 * <p/>
 * Override <code>collectRelatedItems()</code> to return a collection of items which should be highlighted in the GUI as
 * related to this item.
 * <p/>
 * If you only desire to update the selection in the Navigator when the 
 * activated node changes, subclass ActiveNodeSupport, and return 
 * Collections.EMPTY_SET from <code>collectRelatedItems</code>.
 *
 * @author Tim Boudreau
 * @see RelatedItemProvider
 */
public abstract class RelatedItemProviderSupport implements RelatedItemProvider {
    private RelatedItemListener listener = null;
    private static final RequestProcessor rp = new RequestProcessor ( "Navigator RelatedItemProviderSupport gatherer" ); //NOI18N
    private Updater updater = null;
    private Collection list = Collections.EMPTY_LIST;
    private Object primary = null;

    /**
     * An instance of RelatedItemProvider that should be substituted as the
     * source property of events we fire.
     */
    private RelatedItemProvider eventDelegate = this;

    protected RelatedItemProviderSupport () {
    }

    /**
     * Hook for RelatedItemListModelSupport to set itself as the delegate. 
     */
    final void setEventDelegate (RelatedItemProvider eventDelegate) {
        this.eventDelegate = eventDelegate;
    }

    /**
     * Fetch the list of all items that are related to the primary item, and should be displayed as such the GUI.  May
     * return null.
     *
     * @param primary The return value from an immediately preceding call to <code>findPrimaryItem()</code>.  If that
     * method can return null, the implementation of this method must be able to handle a null argument.
     * @see #findPrimaryItem
     */
    protected abstract Collection collectRelatedItems (Object primary);

    /**
     * Fetch the primary item, which should be the selection in the GUI. If the primary item changes, call change() to
     * fire an event.  May return null.
     */
    protected abstract Object findPrimaryItem ();

    /**
     * Begin listening to whatever this provider should listen to in order to fire changes.
     */
    protected abstract void startListening ();

    /**
     * Stop listening to whatever this provider should listen to in order to fire changes.  Called when this
     * RelatedItemProvider is no longer connected with any GUI.
     */
    protected abstract void stopListening ();

    /**
     * Call this method after a change event has happened that will change the contents.  This will cause
     * collectRelatedItems() and findPrimaryItem() to be called again (on a non-AWT thread), and changes fired if need
     * be.
     */
    protected final void change () {
        if ( !isActive () ) {
            return;
        }
        updater.start ();
    }

    public final synchronized void addRelatedItemListener (RelatedItemListener l)
            throws TooManyListenersException {

        if ( listener != null ) throw new TooManyListenersException ();
        if ( l == null ) throw new NullPointerException ();
        listener = l;
        start ();
    }

    public final synchronized void removeRelatedItemListener (RelatedItemListener l) {
        listener = null;
        stop ();
    }

    /**
     * Determine if this provider is currently being listened to by anything.
     */
    protected final boolean isActive () {
        return listener != null;
    }

    /**
     * Returns true if the collection returned from the last call to collectRelatedItems contains the passed object.
     *
     * @param o An object which may be related to the primary object
     */
    public final boolean isRelated (Object o) {
        assert SwingUtilities.isEventDispatchThread () : "Not the AWT thread"; //NOI18N
        return isActive () && list != null ? list.contains ( o ) : false;
    }
    
    public final void reset() {
        assert SwingUtilities.isEventDispatchThread () : "Not the AWT thread"; //NOI18N
        list = Collections.EMPTY_LIST;
    }

    /**
     * Returns true if the object returned from the last call to findPrimaryItem was non-null and equals the passed
     * object.
     *
     * @param o An object which may be related to the primary object
     */
    public final boolean isPrimary (Object o) {
        assert SwingUtilities.isEventDispatchThread () : "Not the AWT thread"; //NOI18N
        return primary == null || !isActive () ? false : primary.equals ( o );
    }

    /**
     * Get an unmodifiable version of the list last provided by collectRelatedItems. May only be called on the event
     * dispatch thread.
     */
    protected Collection getList () {
        assert SwingUtilities.isEventDispatchThread ();
        return list == null ? (Collection) Collections.EMPTY_SET :
                Collections.unmodifiableCollection ( list );
    }

    /**
     * Get the item that should appear as the selection in the view, which the related items are related to, as provided
     * by the last call to findPrimaryItem. May only be called on the event dispatch thread.
     */
    public final Object getPrimaryItem () {
        return primary;
    }

    /**
     * Replaces the contents of the old list with the passed list, and the old
     * primary object with the new one.  Called in the EDT from Updater.run().
     */
    private void setContents (Collection l, Object primary) {
        assert SwingUtilities.isEventDispatchThread ();
        Collection oldList = this.list;
        this.list = l;
        Object oldPrimary = this.primary;
        this.primary = primary;
        if ( ( oldList != null && this.list != null && !oldList.equals ( this.list ) ) || oldPrimary != primary ) {
            fire ( oldList, this.list, oldPrimary, primary );
        }
    }

    /** Fire a RelatedItemEvent */
    private void fire (Collection old, Collection nue, Object oldPrimary, Object primary) {
        assert SwingUtilities.isEventDispatchThread ();
        if ( !isActive () ) {
            return;
        }
        RelatedItemEvent evt = new RelatedItemEvent ( eventDelegate, old, nue, oldPrimary, primary );
        if ( ( nue == null || nue.isEmpty () ) && primary == null ) {
            listener.itemsCleared ( evt );
        } else {
            listener.itemsChanged ( evt );
        }
    }

    /**
     * Starts the updater to do the first fetch of related items in the request
     * processor thread. 
     */
    private void start () {
        assert updater == null : "Start called twice";
        updater = new Updater ();
        updater.start ();
    }

    /**
     * Stop listening and firing events, deinitialize
     */
    private void stop () {
        if (updater != null) {
            updater.cancel ();
        }
        updater = null;
        primary = null;
        list = null;
        stopListening ();
    }

    /**
     * A runnable which will cause the related items list to be recreated and
     * the primary item re-found.
     */
    private final class Updater implements Runnable {
        boolean enqueued = false;
        boolean cancelled = false;
        boolean firstRun = true;

        Object newPrimary = null;
        Collection newItems = null;

        void start () {
            if ( enqueued ) {
                return;
            }
            enqueued = true;
            rp.post ( this );
        }

        void cancel () {
            cancelled = true;
        }

        public void run () {
            if ( rp.isRequestProcessorThread () ) {
                if ( !isActive () ) {
                    return;
                }
                if ( firstRun ) {
                    firstRun = false;
                    startListening ();
                }
                synchronized (this) {
                    newPrimary = findPrimaryItem ();
                    newItems = collectRelatedItems ( newPrimary );
                }
                try {
                    SwingUtilities.invokeAndWait ( this );
                } catch ( InterruptedException e ) {
                    return;
                } catch ( InvocationTargetException e ) {
                    return;
                }
            } else if ( SwingUtilities.isEventDispatchThread () ) {
                if ( !isActive () ) {
                    return;
                }

                Collection nue = null;
                Object prim = null;

                synchronized (this) {
                    nue = newItems;
                    prim = newPrimary;
                }

                setContents ( newItems, newPrimary );
                newItems = null;
                newPrimary = null;
                enqueued = false;
            } else {
                assert false : "Wrong thread"; //NOI18N
            }
        }



    }

    /**
     * RelatedItemProviderSupport subclass which tracks the current or activated Node and offers methods for providing a
     * primary object and related objects based on the current activated node context.
     * To use, simply override <code>collectRelatedItems</code> and 
     * <code>getPrimaryItemFromNodes</code>.
     */
    public static abstract class ActiveNodeSupport extends RelatedItemProviderSupport {
        private PropertyChangeListener pcl = null;
        private final boolean useCurrentNodes;
        private static final Object PCL_LOCK = new Object();

        /**
         * Create a new ActiveNodeSupport.
         *
         * @param useCurrentNodes - determines whether the activated or current node selection should be tracked.
         * @see org.openide.windows.TopComponent.Registry
         */
        protected ActiveNodeSupport (boolean useCurrentNodes) {
            this.useCurrentNodes = useCurrentNodes;
        }

        /**
         * Begins listening to TopComponent.Registry
         */
        protected final void startListening () {
            // [dafe] startListening and stopListening may be called
            // both from EQ thread and outside EQ thread, so syncing is needed,
            // see IZ #61448
            synchronized (PCL_LOCK) {
                pcl = new PCL ();
                TopComponent.getRegistry ().addPropertyChangeListener ( pcl );
                pcl.propertyChange ( null );
            }
        }

        /**
         * Stops listening to TopComponent.Registry
         */
        protected final void stopListening () {
            synchronized (PCL_LOCK) {
                TopComponent.getRegistry ().removePropertyChangeListener ( pcl );
                pcl = null;
            }
        }

        /**
         * Delegates to the version of this method which takes a node argument.
         */
        protected final Collection collectRelatedItems () {
            return collectRelatedItems ( getNodes () );
        }

        /**
         * Collect related items based on the last known activated or current node set.
         *
         * @param primary The return value from getPrimaryItemFromNodes
         */
        protected abstract Collection collectRelatedItems (Object primary);

        /**
         * Get the primary object (that would occur in a navigator model) which is derived from the activated or current
         * node.
         */
        protected abstract Object getPrimaryItemFromNodes (org.openide.nodes.Node[] n);

        protected final Object findPrimaryItem () {
            return getPrimaryItemFromNodes ( getNodes () );
        }

        private org.openide.nodes.Node[] getNodes () {
            return useCurrentNodes ?
                    TopComponent.getRegistry ().getCurrentNodes () :
                    TopComponent.getRegistry ().getActivatedNodes ();
        }

        /** A listener on global node activation */
        private final class PCL implements PropertyChangeListener {
            public void propertyChange (PropertyChangeEvent evt) {
                if ( ( useCurrentNodes && ( evt == null ||
                        TopComponent.Registry.PROP_CURRENT_NODES.equals ( evt.getPropertyName () ) ) ) ||
                        !useCurrentNodes && ( evt == null ||
                        TopComponent.Registry.PROP_ACTIVATED_NODES.equals ( evt.getPropertyName () ) ) ) {

                    change ();
                }
            }
        }
    }

}
