/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * StringFilter.java
 *
 * Created on September 12, 2004, 3:33 PM
 */

package org.netbeans.modules.java.navigation.spi;

import java.util.*;

/**
 * A convenience filter for user-typed filter values, which matches strings starting with the filter text and accepts
 * '*' as a wildcard character, and a leading '!' character to invert the filter.
 *
 * @author Tim Boudreau
 */
public final class StringFilter {
    final String s;

    private StringFilter (String s) {
        this.s = s;
    }

    /**
     * Create a new StringFilter with the specified search text
     */
    public static StringFilter create (String filter) {
        return new StringFilter ( filter );
    }

    /**
     * Get the text of the filter.  Note that this will not include the leading '!' if this is an inverted filter -
     * check <code>isInverseFilter</code> for that.
     */
    public String getText () {
        return isInverseFilter () ? s.substring ( 1 ) : s;
    }

    /**
     * Returns true if the filter logic should be inverted, and it will only 
     * match non-matches for its (non-leading-!)
     * text.
     */
    public boolean isInverseFilter () {
        return s.startsWith ( "!" ); //NOI18N
    }

    /**
     * See if a string matches the filter, considering wildcards and inversion.
     * @param toMatch A string to match, which may begin with ! to invert the
     *        filter and may contain one or more *s as wildcards
     */
    public boolean match (String toMatch) {
        String s = this.s;
        boolean invert = false;
        if ( s.startsWith ( "!" ) ) { //NOI18N
            invert = true;
            s = s.substring ( 1 );
        }
        boolean result;
        if ( s.trim ().length () == 0 ) {
            result = true;
        } else if ( s.indexOf ( "*" ) == -1 ) { //NOI18N
            result = toMatch.startsWith ( s );
        } else if ( s.indexOf ( "*" ) == s.length () - 1 ) { //NOI18N
            result = toMatch.startsWith ( s.substring ( s.length () - 1 ) );
        } else {
            result = wildcardMatch ( toMatch );
        }
        if ( invert ) {
            result = !result;
        }
        return result;
    }

    /**
     * Match using wildcards '*' characters.
     */
    private boolean wildcardMatch (String toMatch) {
        int pos = 0;
        StringTokenizer tok = new StringTokenizer ( s, "*" ); //NOI18N
        while ( tok.hasMoreTokens () ) {
            String curr = tok.nextToken ();
            int nupos = toMatch.indexOf ( curr, pos );
            if ( nupos == -1 ) {
                return false;
            }
            pos = nupos;
        }
        return true;
    }

    public String toString () {
        return s;
    }
}
