/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
/*
 * WeightedString.java
 *
 * Created on September 20, 2004, 2:32 PM
 */

package org.netbeans.modules.java.navigation.spi.strings;

import javax.swing.*;
import org.netbeans.modules.java.navigation.strings.WeightedStringImpl;

/**
 * A string holder which can be assembled from segments.  Each added segment is given a floating point importance value
 * between 0.0f and 1.0f.  Supports logical markup runs which can span segments. Typical usage looks like:
 * <pre>
 * WeightedString s = WeightedString.instance();
 * s.startRunOfType (WeightedString.DEEMPHASIZED);
 * s.append ("SomeInnerClassName", 0.3f);
 * s.endRun();
 * s.append ("TheInnerClass", 1.0f);
 * s.startRunOfType (WeightedString.DEEMPHASIZED_MORE);
 * s.append (" extends Foo");
 * </pre>
 *
 * @author Tim Boudreau
 */
public abstract class WeightedString {
    private static WeightedStringImpl instance = new WeightedStringImpl ();
    /**
     * Bitmask for a run of characters which are struck through.
     */
    public static final int STRIKETHROUGH = 1;
    /**
     * Bitmask for a run of characters which are bold.
     */
    public static final int BOLD = 2;
    /**
     * Bitmask for a run of characters which are deemphasized (usually gray, not black, text)
     */
    public static final int DEEMPHASIZED = 4;
    /**
     * Bitmask for a run of characters which are deemphasized more (usually light gray, not black, text)
     */
    public static final int DEEMPHASIZED_MORE = 8;
    /**
     * Uses a foreground color to emphasize the text.  Since appropriate colors will vary by theme/look and feel, this
     * avoids explicit color definitions (which could be invisible on some backgrounds) and provides a generic way of
     * specifying that a non-default color should be used.
     */
    public static final int COLOR_EMPHASIS_1 = 16;
    /**
     * Uses a foreground color to emphasize the text.  Since appropriate colors will vary by theme/look and feel, this
     * avoids explicit color definitions (which could be invisible on some backgrounds) and provides a generic way of
     * specifying that a non-default color should be used.
     */
    public static final int COLOR_EMPHASIS_2 = 32;
    /**
     * Uses a foreground color to emphasize the text.  Since appropriate colors will vary by theme/look and feel, this
     * avoids explicit color definitions (which could be invisible on some backgrounds) and provides a generic way of
     * specifying that a non-default color should be used.
     */
    public static final int COLOR_EMPHASIS_3 = 64;
    /**
     * Bitmask for a run of characters which should be rendered in italics.
     */
    public static final int ITALIC = 128;
    /**
     * Bitmask for a run of characters which should be underlined
     */
    public static final int UNDERLINE = 256;
    /**
     * Bitmask for a run of characters which should be rendered in a color indicating an error (usually red).
     */
    public static final int ERROR_EMPHASIS = 512;

    protected WeightedString () {
    }

    /**
     * Returns a shared, re-initialized instance of WeightedString if called on the event dispatch thread, otherwise
     * returns a new WeightedString instance.  Do not retain or reuse the instance returned by this method - instead,
     * call this method to obtain a new instance each time one is needed.
     */
    public static final WeightedString instance () {
        if ( SwingUtilities.isEventDispatchThread () ) {
            instance.clear ();
            return instance;
        } else {
            return new WeightedStringImpl ();
        }
    }


    /**
     * Append a String to this WeightedString, marking it with the specified importance.  The importance will be used to
     * determine if any segments or characters in segments should be elided or skipped entirely to optimally use the
     * available space.
     *
     * @param s a string
     * @param importance an importance weight between 0.0f (least important) and 1.0f (most important).
     */
    public abstract void append (String s, float importance);

    /**
     * Append an array of characters to this WeightedString, marking it with the specified importance.  The importance
     * will be used to determine if any segments or characters in segments should be elided or skipped entirely to
     * optimally use the available space.
     *
     * @param str a character array
     * @param importance an importance weight between 0.0f (least important) and 1.0f (most important).
     */
    public abstract void append (char[] str, float importance);

    /**
     * The maximum importance that has been set for any segment
     */
    public abstract float maxImportance ();

    /**
     * Get a <code>WeightedStringPainter</code> which can paint this weighted string into a graphics context.
     */
    public abstract WeightedStringPainter getPainter ();

    /**
     * Set one or more text attributes, such as STRIKETHROUGH on any text appended until <code>endRun</code> is called.
     * The value is a bitmask of logical styles, such as UNDERLINE and STRIKETHROUGH, as defined above.
     * <p/>
     * If this method is called multiple times before adding any text, the bitmask of styles will be the logical OR of
     * all the calls at the current position.
     * <p/>
     * In the case that multiple color-affecting styles in the bitmask (COLOR_EMPHASIS1-3, ERROR_EMPHASIS, DEEMPHASIS,
     * DEEMPHASIS_MORE), it is undefined which will take precedence.
     * <p/>
     * Attributes are not continued - that is, if you set BOLD at position 0, and then set ITALIC at position 4,
     * characters after position 4 are not bold and italic, but just italic.  The bitmask passed at any point should
     * contain all the styles you want used for subsequently added text.
     *
     * @param bitmask - any of the logical markup styles defined in this class
     */
    public abstract void startMarkupRun (int bitmask);

    /**
     * Complete a run started with <code>startRunOfType</code>; segments subsequently added will be displayed without
     * markup.
     */
    public abstract void endMarkupRun ();
    
    /**
     * Get the merged bitmask of all markup types used in this
     * weighted string.  This enables a few optimizations in preferred
     * size calculations.
     */
    public abstract int allMarkupTypes ();
    
}
