/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/*
 * Abbreviations.java
 *
 * Created on August 18, 2002, 2:07 PM
 */

package org.netbeans.modules.java.navigation.strings;


import java.util.*;
import org.netbeans.modules.java.navigation.spi.strings.Abbreviator;

/**
 * Intelligently handles abbreviating strings, weighting characters by their importance.
 *
 * @author Tim Boudreau
 */
public final class AbbreviatorImpl extends Abbreviator {
    private float[] buf = new float[ 50 ];
    private float maxWeight = 0;
    private float minWeight = 0;
    private final int[] lengths = new int[ 13 ];

    public AbbreviatorImpl () {
    }

    /**
     * Process an array of characters into a float[] array of weights.
     */
    public float[] process (char[] c, float importance) {
        buf = ensureCapacity ( buf, c.length );
        Arrays.fill ( lengths, 0 );
        float[] result = iterChars ( c, buf, importance, false );
        if ( allUpper ) {
            result = iterChars ( c, buf, importance, true );
        }

        return result;
    }

    public int[] getLengthsForLastRun () {
        return (int[]) lengths.clone ();
    }

    public float getMaxWeightForLastRun () {
        return maxWeight;
    }

    public float getMinWeightForLastRun () {
        return minWeight;
    }

    private boolean allUpper = false;

    private float[] iterChars (char[] c, float[] weights, float importance, boolean ignoreUpper) {
        allUpper = true;
        maxWeight = 0;

        float[] result = buf;
        char prev = '*';

        boolean lastWasUpper = false;
        boolean lastWasPlosive = false;
        boolean lastWasSiliblant = false;
        boolean lastWasVowel = false;
        boolean lastWasFricative = false;
        boolean lastWasPunctuation = true;

        for ( int i = 0; i < c.length; i++ ) {
            boolean upper = Character.isUpperCase ( c[ i ] );
            //An initial C will be hard
            boolean plosive = isPlosiveOrGlottal ( prev, c[ i ] ) || ( ( c[ i ] == 'c' || c[ i ] == 'C' ) && ( lastWasPunctuation && ( i < c.length - 1 && c[ i + 1 ] != 'h' && c[ i + 1 ] != 'H' ) ) );
            boolean siliblant = isSiliblant ( prev, c[ i ] );
            boolean fricative = isFricative ( c[ i ] );
            boolean vowel = isVowel ( c[ i ] );
            boolean punctuation = !Character.isLetter ( c[ i ] );

            allUpper &= upper;

            if ( !ignoreUpper && ( upper && !lastWasUpper ) ) {
                result[ i ] = lastWasPunctuation || i == 0 ? 1.0f : 0.95f;
            } else if ( plosive && !lastWasPlosive ) {
                result[ i ] = 0.8f;
                if ( lastWasSiliblant && ( !lastWasUpper || ignoreUpper ) ) {
                    result[ i - 1 ] = 0.75f;
                }
            } else if ( siliblant ) {
                result[ i ] = 0.60f;
                if ( isDipthongStart ( prev ) && ( !lastWasUpper || ignoreUpper ) ) {
                    result[ i - 1 ]++;
                }
            } else if ( isDipthong ( prev, c[ i ] ) ) {
                result[ i ] = 0.5f;
                result[ i - 1 ] = 0.65f;
            } else if ( isPalatal ( c[ i ] ) ) {
                result[ i ] = lastWasPlosive ? 0.5f : 0.3f;
            } else if ( isVowel ( c[ i ] ) ) {
                result[ i ] = lastWasVowel ? 0.125f : lastWasPunctuation ? 0.76f : 0.2f;
            } else if ( isFricative ( c[ i ] ) ) {
                result[ i ] = 0.50f;
            } else if ( !Character.isLetter ( c[ i ] ) ) {
                result[ i ] = 1.0f;
            } else {
                result[ i ] = 0.45f;
            }

            if ( prev == c[ i ] ) {
                result[ i ] -= 0.1f;
            }

            if ( lastWasPunctuation ) {
                result[ i ] = Math.min ( result[ i ] + 0.45f, 1.0f );
            }

            lastWasUpper = upper;
            lastWasFricative = fricative;
            lastWasSiliblant = siliblant;
            lastWasVowel = vowel;
            lastWasPlosive = plosive;
            lastWasPunctuation = punctuation;

            maxWeight = Math.max ( result[ i ], maxWeight );
            minWeight = Math.min ( result[ i ], minWeight );

            result[ i ] += ( importance - 0.5f ) / 9f;

            for ( int j = (int) ( ( result[ i ] * 10 ) - 1 ); j >= 0; j-- ) {
                lengths[ j ]++;
            }

            prev = c[ i ];
        }
        result[ 0 ] = Math.max ( 1.0f, result[ 0 ] + 0.2f );
        return result;
    }

    private float[] ensureCapacity (float[] i, int size) {
        if ( i.length >= size ) {
            return i;
        } else {
            return new float[ size + ( size / 2 ) ];
        }
    }

    private boolean isVowel (char c) {
        //in our world, y is a vowel
        return c == 'a' || c == 'e' || c == 'i' || c == 'o' || c == 'u' || c == 'y' ||
                c == 'A' || c == 'E' || c == 'I' || c == 'O' || c == 'U' || c == 'Y';
    }

    private boolean isPlosiveOrGlottal (char prev, char c) {
        boolean wasN = ( prev == 'n' || prev == 'N' ) && Character.isUpperCase ( prev ) == Character.isUpperCase ( c );
        return c == 'p' || c == 't' || c == 'b' || c == 'd' || ( c == 'g' && !wasN ) || c == 'k' || c == 'q' ||
                c == 'P' || c == 'T' || c == 'B' || c == 'D' || ( c == 'G' && !wasN ) || c == 'K' || c == 'Q';
    }

    private boolean isFricative (char c) {
        return c == 'f' || c == 'j' || c == 'h' || c == 'v' ||
                c == 'F' || c == 'J' || c == 'H' || c == 'V';
    }

    private boolean isSiliblant (char prev, char c) {
        return c == 's' || c == 'z' || c == 'X' ||
                c == 'S' || c == 'Z' || c == 'x' || isDipthong ( prev, c );
    }

    private boolean isDipthongStart (char c) {
        return c == 's' || c == 'c' || c == 't' || c == 'p' ||
                c == 'S' || c == 'C' || c == 'T' || c == 'P';
    }

    private boolean isDipthongEnd (char c) {
        return c == 'h' || c == 'h';
    }

    private boolean isPalatal (char c) {
        return c == 'm' || c == 'n' || c == 'r' || c == 'l' ||
                c == 'M' || c == 'N' || c == 'R' || c == 'L';
    }

    private boolean isDipthong (char prev, char curr) {
        return isDipthongStart ( prev ) && isDipthongEnd ( curr );
    }
}