/*
* The contents of this file are subject to the terms of the Common Development
* and Distribution License (the License). You may not use this file except in
* compliance with the License.
*
* You can obtain a copy of the License at http://www.netbeans.org/cddl.html
* or http://www.netbeans.org/cddl.txt.
* 
* When distributing Covered Code, include this CDDL Header Notice in each file
* and include the License file at http://www.netbeans.org/cddl.txt.
* If applicable, add the following below the CDDL Header, with the fields
* enclosed by brackets [] replaced by your own identifying information:
* "Portions Copyrighted [year] [name of copyright owner]"
*
* The Original Software is NetBeans. The Initial Developer of the Original
* Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
* Microsystems, Inc. All Rights Reserved.
*/
/*
 * WeightedStringPainter.java
 *
 * Created on September 20, 2004, 11:26 AM
 */

package org.netbeans.modules.java.navigation.strings;


import java.awt.*;
import java.util.*;
import org.netbeans.modules.java.navigation.spi.strings.WeightedStringPainter;
import org.netbeans.modules.java.navigation.spi.strings.WeightedString;

/**
 * Class which can paint a weighted string into a graphics context.
 *
 * @author Tim Boudreau
 */
final class WeightedStringPainterImpl implements WeightedStringPainter {
//    private boolean[] skips = new boolean[ 20 ];
    private int currSeg = 0;
    private boolean willAbbreviate = false;

    // <list of AbbrevInfo
    private ArrayList abbrevInfos = new ArrayList(20);
    
    private static class AbbrevInfo {
        boolean isSkip = false;
        char[] abbrev = null;
    }

    
    /**
     * Creates a new instance of WeightedStringPainter
     */
    WeightedStringPainterImpl () {
    }

    public void paint (Graphics g, int x, int y, WeightedString ws, int maxChars, boolean useColors) {
        doPaint ( g, x, y, (WeightedStringImpl) ws, maxChars, useColors );
    }

    private void doPaint (Graphics g, int x, int y, WeightedStringImpl ws, int maxChars, boolean useColors) {
        willAbbreviate = configureFrom ( ws, maxChars );
        Font defFont = g.getFont ();
        Color defFg = g.getColor ();
        if ( ( ws.allMarkupTypes () & ws.BOLD ) != 0 ) {
            //Fudge the spacing for bold fonts - chars are wider
            maxChars = Math.max ( 2, maxChars - ( maxChars / 3 ) );
        }
        // antialias if swing.aatext is enabled
        Map rhints = (Map)(Toolkit.getDefaultToolkit().getDesktopProperty("awt.font.desktophints")); //NOI18N
        Graphics2D g2 = (Graphics2D)g;
        if (rhints == null && Boolean.getBoolean("swing.aatext")) { //NOI18N
             g2.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
        } else if (rhints != null) {
            g2.addRenderingHints(rhints);
         }
        
        
        char[] seg = first ( ws );
        int types = 0;
        while ( seg != null ) {
            types = ws.getMarkupFor ( index () );

        
            setupGraphics ( g, types, defFont, defFg, useColors );

            FontMetrics fm = g.getFontMetrics ();

            g.drawChars ( seg, 0, seg.length, x, y );
            int dist = g.getFontMetrics ().charsWidth ( seg, 0, seg.length );

            if ( ( types & WeightedString.UNDERLINE ) != 0 ) {
                g.drawLine ( x, y + 1, x + dist, y + 1 );
            }
            if ( ( types & WeightedString.STRIKETHROUGH ) != 0 ) {
                int ly = y - ( (fm.getAscent() - 2) / 2 );
                g.setColor(defFg);
                g.drawLine ( x, ly, x + dist, ly );
            }

            x += dist;
            seg = next ( ws );
        }
    }

    private void setupGraphics (Graphics g, int type, Font defFont, Color defFg, boolean useColors) {
        if ( type == 0 ) {
            g.setColor ( defFg );
            g.setFont ( defFont );
        }
        Font f = g.getFont ();
        if ( ( type & WeightedStringImpl.BOLD ) != 0 ) {
            if ( ( type & WeightedStringImpl.ITALIC ) != 0 ) {
                if ( f.getStyle () != ( Font.BOLD | Font.ITALIC ) ) {
                    g.setFont ( g.getFont ().deriveFont ( Font.BOLD | Font.ITALIC ) );
                }
            } else {
                if ( f.getStyle () != Font.BOLD ) {
                    g.setFont ( g.getFont ().deriveFont ( Font.BOLD ) );
                }
            }
        } else if ( ( type & WeightedStringImpl.ITALIC ) != 0 ) {
            if ( f.getStyle () != Font.ITALIC ) {
                g.setFont ( g.getFont ().deriveFont ( Font.ITALIC ) );
            }
        } else {
            if ( g.getFont ().getStyle () != Font.PLAIN ) {
                g.setFont ( g.getFont ().deriveFont ( Font.PLAIN ) );
            }
        }
        //XXX define in UIManager
        if ( useColors ) {
            if ( ( type & WeightedStringImpl.DEEMPHASIZED ) != 0 ) {
                g.setColor ( Color.GRAY );
            } else if ( ( type & WeightedStringImpl.DEEMPHASIZED_MORE ) != 0 ) {
                g.setColor ( Color.LIGHT_GRAY );
            } else if ( ( type & WeightedStringImpl.ERROR_EMPHASIS ) != 0 ) {
                g.setColor ( Color.RED );
            } else if ( ( type & WeightedStringImpl.COLOR_EMPHASIS_1 ) != 0 ) {
                g.setColor ( Color.BLUE );
            } else if ( ( type & WeightedStringImpl.COLOR_EMPHASIS_2 ) != 0 ) {
                g.setColor ( new Color (0, 150, 0));
            } else if ( ( type & WeightedStringImpl.COLOR_EMPHASIS_3 ) != 0 ) {
                g.setColor ( new Color (150, 0, 150));
            } else {
                g.setColor ( defFg ); //XXX?
            }
        }
    }
    
    private boolean configureFrom (WeightedStringImpl ws, int maxChars) {
        return config ( ws, maxChars ) != null;
    }

    Entry[] config (WeightedStringImpl ws, int maxChars) { //Pkg private for unit tests
        currSeg = 0;
//        Arrays.fill ( skips, false );
        if ( maxChars >= ws.length ) {
            return null;
        } else {
            ensureCapacity ( ws );
            return weightAndElide ( ws, maxChars );
        }
    }

    private void ensureCapacity (WeightedStringImpl ws) {
/*        if ( skips.length <= ws.getCount () ) {
            skips = new boolean[ ws.getCount () + ( ws.getCount () / 2 ) ];
        }*/
    }

    private char[] first (WeightedStringImpl ws) {
        currSeg = 0;
        while (currSeg <= ws.getCount () && willAbbreviate && 
               ((AbbrevInfo)abbrevInfos.get(currSeg)).isSkip) {
            currSeg++;
        }
        return current ( ws );
    }

    private char[] current (WeightedStringImpl ws) {
        char[] result;
        if ( currSeg > ws.getCount () ) {
            currSeg = ws.getCount () + 1;
            result = null;
        } else {
            result = willAbbreviate ? seg ( ws ) : ws.getCharsSegment(currSeg);
        }
        return result;
    }

    private char[] next (WeightedStringImpl ws) {
        currSeg++;
        while (currSeg <= ws.getCount () && willAbbreviate && 
                ((AbbrevInfo)abbrevInfos.get(currSeg)).isSkip) {
            currSeg++;
        }
        return current ( ws );
    }
    
    private int index () {
        return currSeg;
    }

    private char[] seg (WeightedStringImpl ws) {
        char[] abbrevSeg = ((AbbrevInfo)abbrevInfos.get(currSeg)).abbrev;
        char[] result = abbrevSeg != null ? abbrevSeg : ws.getCharsSegment(currSeg);
        
        return result;
        
        /*char[] result = (char[]) ws.chars[ currSeg ];
        int resCount = 0;
        for ( int i = 0; i < result.length; i++ ) {
            if ( result[ i ] != 0 ) {
                resCount++;
            }
        }
        if ( resCount == result.length ) {
            return result;
        } else {
            char[] nue = new char[ resCount ];
            int ix = 0;
            for ( int i = 0; i < result.length; i++ ) {
                if ( result[ i ] != 0 ) {
                    nue[ ix ] = result[ i ];
                    ix++;
                }
            }
            return nue;
        }*/
    }

    // THE WORKING VERSION - KEEP:
    private Entry[] weightAndElide (WeightedStringImpl ws, int maxChars) {
        Entry[] entries = new Entry[ ws.getCount () + 1 ];
        abbrevInfos.clear();
        
        int[] lengths = new int[ 11 ];
        int[] lengthsExcludingImportant = new int[ 11 ];
        char[] curSeg;
                
        for ( int i = 0; i < entries.length; i++ ) {
            char[] c = ws.getCharsSegment(i);
            float importance = ws.getImportance ( i );
            entries[ i ] = new Entry ( c, importance, i );
            abbrevInfos.add(new AbbrevInfo());
            
            int[] l = entries[ i ].getLengths ();

            for ( int j = 0; j <= 10; j++ ) {
                lengths[ j ] += l[ j ];
                lengthsExcludingImportant[ j ] += importance == 1.0f ?
                        entries[ i ].fullLength () :
                        l[ j ];
            }
        }

        Arrays.sort ( entries );
        boolean exImp = true;
        int tg = 0;
        for ( tg = 0; tg <= 10; tg++ ) {
            if ( lengthsExcludingImportant[ tg ] <= maxChars ) {
                break;
            }
        }

        if ( tg > 10 ) {
            exImp = false;
            for ( tg = 0; tg <= 10; tg++ ) {
                if ( lengths[ tg ] <= maxChars ) {
                    break;
                }
            }
        }
        if ( tg >= 10 ) {
            tg = 9;
        }

        float weight = ( tg + 1 ) / 10f;

        Entry mostImp = null;

        int len = 0;
        AbbrevInfo curAbbrev;
        for ( int i = 0; i < entries.length; i++ ) {
            if ( !exImp || entries[ i ].getImportance () < 1.0f ) {
                entries[ i ].setTargetWeight ( weight );
                curAbbrev = (AbbrevInfo)abbrevInfos.get(entries[i].getIndex());
                curAbbrev.abbrev = entries[i].elide(-1);
                
                len += curAbbrev.abbrev.length;
            } else {
                mostImp = entries[ i ];
                len += entries[ i ].length ();
            }
        }

        Arrays.sort ( entries, new IndexComparator () );

        if ( len > maxChars ) {
            for ( int i = entries.length - 1; i >= 0; i-- ) {
                if ( entries[ i ].getImportance () >= 1.0f ) {
                    break;
                } else {
                    ((AbbrevInfo)abbrevInfos.get(i)).isSkip = true;
                    entries[ i ].setSkip ( true );
                    len -= entries[ i ].getLengthAtTarget ();
                }
            }
        }
//        if (len > maxChars && mostImp != null && exImp) {
        if ( mostImp != null && len > maxChars ) {
            mostImp.setTargetWeight ( 0.56f );
            curAbbrev = (AbbrevInfo)abbrevInfos.get(mostImp.getIndex());
            curAbbrev.abbrev = mostImp.elide(-1);
        }

        //Return value only used by unit tests
        return entries;
    }

    private static final class IndexComparator implements Comparator {
        public int compare (Object a, Object b) {
            Entry ea = (Entry) a;
            Entry eb = (Entry) b;
            return ea.index - eb.index;
        }
    }
}
