/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/* Java Scanner */

package org.netbeans.lib.java.parser;

import java.util.ArrayList;
import java.util.List;
import java.lang.reflect.Modifier;

%%

%class Scanner
%implements ParserTokens,JScanner

%unicode
%pack
%byaccj
%char
%final
%public

//%debug
//%standalone

%init{
  in = unicodeReader = new UnicodeEscapes(in);
%init}

%{
  private StringBuffer string = new StringBuffer();
  private Token lastToken=null;
  private List padding=new ArrayList();
  private Parser parser;
  private UnicodeEscapes unicodeReader;
  private ArrayList tokens=new ArrayList();
  private int savedOffset;
  private static final Token EMPTY_PADDING[]=new Token[0];
  private static final Integer MOD_ABSTRACT=new Integer(Modifier.ABSTRACT);
  private static final Integer MOD_PUBLIC=new Integer(Modifier.PUBLIC);
  private static final Integer MOD_PROTECTED=new Integer(Modifier.PROTECTED);
  private static final Integer MOD_PRIVATE=new Integer(Modifier.PRIVATE);
  private static final Integer MOD_STATIC=new Integer(Modifier.STATIC);
  private static final Integer MOD_FINAL=new Integer(Modifier.FINAL);
  private static final Integer MOD_SYNCHRONIZED=new Integer(Modifier.SYNCHRONIZED);
  private static final Integer MOD_NATIVE=new Integer(Modifier.NATIVE);
  private static final Integer MOD_STRICTFP=new Integer(Modifier.STRICT);
  private static final Integer MOD_TRANSIENT=new Integer(Modifier.TRANSIENT);
  private static final Integer MOD_VOLATILE=new Integer(Modifier.VOLATILE);
  private static final Boolean LIT_TRUE=new Boolean(true);
  private static final Boolean LIT_FALSE=new Boolean(false);

  private int token(int type) {
    return token(type,null);
  }

  private Token cretateToken(int type, Object value,Token pad[],int index) {
	  int startPos;
	  int endPos;
          int scStartPos;

          if (type==STRING_LIT || type==CHAR_LIT)
              scStartPos=savedOffset;
          else 
              scStartPos=yychar;
	  startPos=unicodeReader.convertPosition(scStartPos);
	  endPos=unicodeReader.convertPosition(yychar+yylength());
	  //System.out.println("Current start position:"+yychar+" Correct start position:"+startPos);
          //System.out.println("Current end position:"+yychar+yylength()+" Correct end position:"+endPos);
	  return new ScannerToken(type, value, startPos, endPos, pad,index);
  }
  
  private int token(int type, Object value) {
    if (parser!=null) {
        int psize=padding.size();
        Token padArr[];

        padArr=(psize==0)?EMPTY_PADDING:(Token[])padding.toArray(new Token[padding.size()]);
        lastToken = cretateToken(type,value,padArr,tokens.size());
            tokens.add(lastToken);
        if (parser!=null)
            parser.yylval = (ASTClass)lastToken;
        padding.clear();
    }
    return type;
  }

  private void addPadding(int type) {
    if (parser!=null) {
        Token pad=cretateToken(type,type==DOC_COMMENT?yytext():null,null,padding.size());
        padding.add(pad);
    }
  }

  protected void setParser(Parser p) {
     parser=p;
  }

  protected Token getToken(int index) {
     if (parser!=null) 
        return (Token)tokens.get(index);
     return null;
  }

  protected Token[] getTokens() {
      if (parser!=null) {
	  Token[] array = new Token[tokens.size()];
	  return (Token[])tokens.toArray(array);
      }
     return null;
  }

  private Integer atoi(int start,int end,int radix) {
    Long number=atol(start,end,radix);

    return new Integer(number.intValue());
  }

  private Long atol(int start,int end,int radix) {
    long result=0;
    int i;

    for(i=start;i<end;i++) {
      int digit=0;
      char ch=yycharat(i);
      
      result*=radix;
      if (ch<='9')
        digit=ch-'0';
      else if (ch>='A' && ch<='F')
        digit=ch-'A'+10;
      else if (ch>='a' && ch<='f')
        digit=ch-'a'+10;
      result+=digit;
    }
    return new Long(result);
  }

  public static Scanner newScanner (java.io.Reader in, String sourceLevel,
				     boolean liteScanning) {
      return new Scanner(in);
  }

  public int yyhash() {
      return yytext().hashCode();
  }

  public boolean isDeprecated() {
      return false;
  }

  public void resetDeprecatedFlag() { }
%}

/* main character classes */
LineTerminator = \r|\n|\r\n
WhiteSpace = [ \t\f]
InputCharacter = [^\n\r]

/* comments */
TraditionalComment = "/*" ~"*/"
EndOfLineComment = "//" {InputCharacter}*
/* DocumentationComment = "/**"[^/] ~"*/" */

/* identifiers */
Identifier = [:jletter:][:jletterdigit:]*

/* integer literals */
DecIntegerLiteral = 0 | [1-9][0-9]*
DecLongLiteral    = {DecIntegerLiteral} [lL]

HexIntegerLiteral = 0 [xX] 0* {HexDigit} {1,8}
HexLongLiteral    = 0 [xX] 0* {HexDigit} {1,16} [lL]
HexDigit          = [0-9a-fA-F]

OctIntegerLiteral = 0+ [1-3]? {OctDigit} {1,15}
OctLongLiteral    = 0+ 1? {OctDigit} {1,21} [lL]
OctDigit          = [0-7]
    
/* floating point literals */  
FractionNumber = (([0-9]+("."[0-9]*)?) | ("."[0-9]+)) ([eE][-+]?[0-9]+)?
FloatLiteral  = {FractionNumber} [fF]
DoubleLiteral = {FractionNumber}

/* string and character literals */
StringCharacter = [^\r\n\"\\]
SingleCharacter = [^\r\n\'\\]

%state STRING, CHARACTER

%%

<YYINITIAL> {

  /* keywords */
  "abstract"                     { return token(ABSTRACT,MOD_ABSTRACT); }
  "assert"                       { return token(ASSERT); }
  "boolean"                      { return token(BOOLEAN); }
  "break"                        { return token(BREAK); }
  "byte"                         { return token(BYTE); }
  "case"                         { return token(CASE); }
  "catch"                        { return token(CATCH); }
  "char"                         { return token(CHAR); }
  "class"                        { return token(CLASS); }
  "const"                        { return token(CONST); }
  "continue"                     { return token(CONTINUE); }
  "do"                           { return token(DO); }
  "double"                       { return token(DOUBLE); }
  "else"                         { return token(ELSE); }
  "extends"                      { return token(EXTENDS); }
  "final"                        { return token(FINAL,MOD_FINAL); }
  "finally"                      { return token(FINALLY); }
  "float"                        { return token(FLOAT); }
  "for"                          { return token(FOR); }
  "default"                      { return token(DEFAULT); }
  "implements"                   { return token(IMPLEMENTS); }
  "import"                       { return token(IMPORT); }
  "instanceof"                   { return token(INSTANCEOF); }
  "int"                          { return token(INT); }
  "interface"                    { return token(INTERFACE); }
  "long"                         { return token(LONG); }
  "native"                       { return token(NATIVE,MOD_NATIVE); }
  "new"                          { return token(NEW); }
  "goto"                         { return token(GOTO); }
  "if"                           { return token(IF); }
  "public"                       { return token(PUBLIC,MOD_PUBLIC); }
  "short"                        { return token(SHORT); }
  "super"                        { return token(SUPER); }
  "switch"                       { return token(SWITCH); }
  "synchronized"                 { return token(SYNCHRONIZED,MOD_SYNCHRONIZED); }
  "package"                      { return token(PACKAGE); }
  "private"                      { return token(PRIVATE,MOD_PRIVATE); }
  "protected"                    { return token(PROTECTED,MOD_PROTECTED); }
  "transient"                    { return token(TRANSIENT,MOD_TRANSIENT); }
  "return"                       { return token(RETURN); }
  "void"                         { return token(VOID); }
  "static"                       { return token(STATIC,MOD_STATIC); }
  "while"                        { return token(WHILE); }
  "this"                         { return token(THIS); }
  "throw"                        { return token(THROW); }
  "throws"                       { return token(THROWS); }
  "try"                          { return token(TRY); }
  "volatile"                     { return token(VOLATILE,MOD_VOLATILE); }
  "strictfp"                     { return token(STRICTFP,MOD_STRICTFP); }
  
  /* boolean literals */
  "true"                         { return token(BOOL_LIT, LIT_TRUE); }
  "false"                        { return token(BOOL_LIT, LIT_FALSE); }
  
  /* null literal */
  "null"                         { return token(NULL_LIT); }
  
  
  /* separators */
  "("                            { return token(L_PAR); }
  ")"                            { return token(R_PAR); }
  "{"                            { return token(L_CURLY); }
  "}"                            { return token(R_CURLY); }
  "["                            { return token(L_BRACKET); }
  "]"                            { return token(R_BRACKET); }
  ";"                            { return token(SEMICOLON); }
  ","                            { return token(COMMA); }
  "."                            { return token(DOT); }
  "..."                          { return token(ELLIPSIS); }

  /* operators */
  "="                            { return token(ASSIGN); }
  ">"                            { return token(GT); }
  "<"                            { return token(LT); }
  "!"                            { return token(NOT); }
  "~"                            { return token(COMP); }
  "?"                            { return token(QUESTION); }
  ":"                            { return token(COLON); }
  "=="                           { return token(EQ); }
  "<="                           { return token(LTE); }
  ">="                           { return token(GTE); }
  "!="                           { return token(NEQ); }
  "&&"                           { return token(BOOL_AND); }
  "||"                           { return token(BOOL_OR); }
  "++"                           { return token(INCREMENT); }
  "--"                           { return token(DECREMENT); }
  "+"                            { return token(PLUS); }
  "-"                            { return token(MINUS); }
  "*"                            { return token(MULTI); }
  "/"                            { return token(DIV); }
  "&"                            { return token(AND); }
  "|"                            { return token(OR); }
  "^"                            { return token(XOR); }
  "%"                            { return token(MOD); }
  "<<"                           { return token(L_SHIFT); }
  ">>"                           { return token(R_SHIFT); }
  ">>>"                          { return token(UR_SHIFT); }
  "+="                           { return token(PLUS_ASSIGN); }
  "-="                           { return token(MINUS_ASSIGN); }
  "*="                           { return token(MULTI_ASSIGN); }
  "/="                           { return token(DIV_ASSIGN); }
  "&="                           { return token(AND_ASSIGN); }
  "|="                           { return token(OR_ASSIGN); }
  "^="                           { return token(XOR_ASSIGN); }
  "%="                           { return token(MOD_ASSIGN); }
  "<<="                          { return token(L_SHIFT_ASSIGN); }
  ">>="                          { return token(R_SHIFT_ASSIGN); }
  ">>>="                         { return token(UR_SHIFT_ASSIGN); }
  
  /* string literal */
  \"                             { yybegin(STRING); string.setLength(0); savedOffset=yychar; }

  /* character literal */
  \'                             { yybegin(CHARACTER); savedOffset=yychar; }

  /* numeric literals */

  {DecIntegerLiteral}            { return token(INT_LIT, parser!=null ? atoi(0,yylength(),10) : null); }
  {DecLongLiteral}               { return token(INT_LIT, parser!=null ? atol(0,yylength()-1,10) : null); }
  
  {HexIntegerLiteral}            { return token(INT_LIT, parser!=null ? atoi(2,yylength(),16) : null); }
  {HexLongLiteral}               { return token(INT_LIT, parser!=null ? atol(2,yylength()-1,16) : null); }
 
  {OctIntegerLiteral}            { return token(INT_LIT, parser!=null ? atoi(0,yylength(),8) : null); }  
  {OctLongLiteral}               { return token(INT_LIT, parser!=null ? atol(0,yylength()-1,8) : null); }
  
  {FloatLiteral}                 { return token(FLOAT_LIT, parser!=null ? new Float(yytext().substring(0,yylength()-1)) : null); }
  {DoubleLiteral}                { return token(FLOAT_LIT, parser!=null ? new Double(yytext()) : null); }
  {DoubleLiteral}[dD]            { return token(FLOAT_LIT, parser!=null ? new Double(yytext().substring(0,yylength()-1)) : null); }
  
  /* comments */
  {TraditionalComment}           { if (yylength()>4 && yycharat(2)=='*')
                                      addPadding(DOC_COMMENT);
                                   else
                                      addPadding(COMMENT);
                                 }
  {EndOfLineComment}             { addPadding(EOL_COMMENT); }

  /* whitespace */
  " "+                           { /* ignore spaces */ }
  {WhiteSpace}+                  { addPadding(WHITESPACE); }
  {LineTerminator}+              { addPadding(EOL); }

  /* identifiers */ 
  {Identifier}                   { return token(IDENTIFIER, parser!=null?yytext():null); }  
}

<STRING> {
  \"                             { yybegin(YYINITIAL); return token(STRING_LIT, parser!=null ? string.toString() : null); }
  
  {StringCharacter}+             { if (parser!=null) string.append( yytext() ); }
  
  /* escape sequences */
  "\\b"                          { string.append( '\b' ); }
  "\\t"                          { string.append( '\t' ); }
  "\\n"                          { string.append( '\n' ); }
  "\\f"                          { string.append( '\f' ); }
  "\\r"                          { string.append( '\r' ); }
  "\\\""                         { string.append( '\"' ); }
  "\\'"                          { string.append( '\'' ); }
  "\\\\"                         { string.append( '\\' ); }
  \\[0-3]?{OctDigit}?{OctDigit}  { char val = (char) Integer.parseInt(yytext().substring(1),8);
                        				   string.append( val ); }
  
  /* error cases */
  \\.                            { throw new RuntimeException("Illegal escape sequence \""+yytext()+"\""); }
  {LineTerminator}               { throw new RuntimeException("Unterminated string at end of line"); }
}

<CHARACTER> {
  {SingleCharacter}\'            { yybegin(YYINITIAL); return token(CHAR_LIT, new Character(yytext().charAt(0))); }
  
  /* escape sequences */
  "\\b"\'                        { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\b'));}
  "\\t"\'                        { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\t'));}
  "\\n"\'                        { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\n'));}
  "\\f"\'                        { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\f'));}
  "\\r"\'                        { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\r'));}
  "\\\""\'                       { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\"'));}
  "\\'"\'                        { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\''));}
  "\\\\"\'                       { yybegin(YYINITIAL); return token(CHAR_LIT, new Character('\\')); }
  \\[0-3]?{OctDigit}?{OctDigit}\' { yybegin(YYINITIAL); 
			            int val = Integer.parseInt(yytext().substring(1,yylength()-1),8);
			            return token(CHAR_LIT, new Character((char)val)); 
                                 }
  /* error cases */
  \\.                            { throw new RuntimeException("Illegal escape sequence \""+yytext()+"\""); }
  {LineTerminator}               { throw new RuntimeException("Unterminated character literal at end of line"); }
}

/* error fallback */
.|\n                             { throw new RuntimeException("Illegal character \""+yytext()+"\" at line "+yyline+", column "+yycolumn); }
