/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyVetoException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.*;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.java.bridge.DefaultLangModel;
import org.netbeans.modules.java.bridge.LangModel;
import org.netbeans.modules.java.bridge.SrcElementImpl;
import org.netbeans.modules.java.parser.JavaParser;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.ResourceClassImpl;
import org.openide.ErrorManager;
import org.openide.cookies.*;
import org.openide.filesystems.*;
import org.openide.loaders.*;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.CookieSet;
import org.openide.nodes.Node;
import org.openide.src.*;
import org.openide.src.nodes.ElementNodeFactory;
import org.openide.src.nodes.FilterFactory;
import org.openide.text.CloneableEditorSupport;
import org.openide.util.*;

/**
 * Data object representing a Java source file.
 * May be subclassed (but please don't).
 */
public class JavaDataObject extends MultiDataObject implements CookieSet.Factory {
    /** generated Serialized Version UID */
    static final long serialVersionUID = -6035788991669336965L;

    /**
     * Holds a reference to editor support class. Lazy initialized from getJavaEditor,
     * or a getter for EditorSupport.class cookie.
     */
    transient private JavaEditor editorSupport;

    transient protected AbstractNode alteranteParent;

    /** WeakListener that intercepts external modification while the file is not
     *  opened in the Editor.
     */
    transient private FileChangeListener fileChangeListener;

    /** Hook to keep the WeakListener alive for the lifetime of the DataObject
     */
    transient private FileChangeListener fileChangeListenerHook;


    /** TEMPORARY: Filename of the primary file before it was changed (DO was
     * renamed or moved.
     */
    transient private String previousFileName;

    /** Lock for parsing and connection support
     */
    transient private Object lock;

    private transient boolean initialized;

    /**
     * Holds a reference to the glue between the parser and the dataobject.
     * Lazy initialized from {@link #initializeParsingSupport}
     */
    private transient JavaParserGlue   parserGlue;

    private transient SourceElement source = null;

    private transient DefaultLangModel model = null;

    /** Create new data object.
    * @param pf primary file object for this data object (i.e. the source file)
    * @param loader the associated loader
    */
    public JavaDataObject(FileObject pf, MultiFileLoader loader) throws DataObjectExistsException {
        super(pf, loader);
        init();
    }

    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException {
        in.defaultReadObject();
        init();
    }

    private void init() {
        MultiDataObject.Entry entry = getPrimaryEntry();
        CookieSet cookies = getCookieSet();
        Class[] cookieClasses = new Class[] {
            JavaEditor.class,
            JavaParser.class,
            SourceCookie.Editor.class,
        };
        cookies.add(cookieClasses, this);

        PrimaryListener l = new PrimaryListener();
	addPropertyChangeListener(l);
        fileChangeListenerHook = l;
	fileChangeListener = FileUtil.weakFileChangeListener(fileChangeListenerHook, entry.getFile());
        lock = new Object();
        addSourceChangeListener(null);
        initialized = true;
        // [TODO] the following code was comented out as it interfered with FileScanner during IDE startup
        // it should be uncommented after we figure out how to avoid the interference
//        try {
//            updateResource();
//        } catch (IOException ex) {
//            ErrorManager.getDefault().notify(ex);
//        }
    }
    
    public synchronized DefaultLangModel getModel () {
        if (model == null) {
            synchronized (lock) {
                if (model == null)
                    model = new DefaultLangModel (this);
            }
        }
        return model;
    }


    void firePropertyChange0(String s, Object v1, Object v2) {
        super.firePropertyChange(s, v1, v2);
    }

    /**
     * Creates a parsing support.
     */
    private JavaParser initializeParsingSupport() {
        if (parserGlue != null)
            return parserGlue.getParser();
        synchronized (lock) {
            if (parserGlue != null) {
                return parserGlue.getParser();
            }

            parserGlue = new JavaParserGlue(getPrimaryEntry());
            if (editorSupport != null) {
                parserGlue.cloneableSupportCreated(editorSupport);
            }
        }
        return parserGlue.getParser();
    }

    public void setValid(boolean v) throws PropertyVetoException {
        super.setValid(v);
        if (!v) {
            synchronized (this) {
                if (parserGlue != null) {
                    parserGlue.suspendDocumentChanges();
                }
            }
        }
    }

    /**
     * Finds a cloneableEditorSupport that holds the displayable/parseable source. The implementation
     * currently uses a hack, that extract package-private member from the EditorSupport's
     * implementation.
     */
    protected CloneableEditorSupport findCloneableEditorSupport() {
        return (JavaEditor)getCookie(JavaEditor.class);
        //return extractCloneableEditor(supp);
    }

    /** Attaches a file change listener to the primary (source) file.
     * Optionally removes the listener from previously used file object.
     * @param previousPrimary if not null, then the method removes change listener from this
     *        file object.
     */
    void addSourceChangeListener(FileObject previousPrimary) {
      if (previousPrimary != null) {
        previousPrimary.removeFileChangeListener(fileChangeListener);
      }
      getPrimaryEntry().getFile().addFileChangeListener(fileChangeListener);
    }

    void addSaveCookie(SaveCookie save) {
        getCookieSet().add(save);
    }
    
    void removeSaveCookie(SaveCookie save) {
        getCookieSet().remove(save);
    }

    public void resumeSupports() {
        parserGlue.resumeDocumentChanges();
    }

    public void suspendSupports() {
        // initialize support objects. If they were requested after this method,
        // they would be created in an active state.
        initializeParsingSupport();
        // suspend everything that is bound to the model and document
        parserGlue.suspendDocumentChanges();
    }

    boolean isJavaFileReadOnly() {
	FileObject primary = getPrimaryFile();
	if (!isValid() || !primary.isValid()) {
	    return true;
	}
	return !primary.canWrite();
    }

    // ==================== Handle methods =======================

    protected DataObject handleCopy(DataFolder df) throws IOException {
        String originalName = getName();
        DataObject dob = super.handleCopy(df);
        if (!Repository.getDefault().getDefaultFileSystem().equals(dob.getPrimaryFile().getFileSystem())) {
            JavaMetamodel.getDefaultRepository().beginTrans(true);
            boolean fail = true;
            try {
                ClassPath cp = ClassPath.getClassPath(dob.getPrimaryFile(),ClassPath.SOURCE);
                if (cp!=null) {
                    JavaModelPackage model = JavaMetamodel.getManager().resolveJavaExtent(cp.findOwnerRoot(dob.getPrimaryFile()));
                    if (model == null) {
                        ErrorManager.getDefault().log(ErrorManager.WARNING, "JavaDataObject: Extent for " + dob.getName() + " not found");
                        fail = false;
                        return dob;
                    }
                    ResourceClassImpl resClass = (ResourceClassImpl) model.getResource();
                    String resourceName = cp.getResourceName(dob.getPrimaryFile());
                    Resource res = resClass.resolveResource(resourceName, true, false);
                
                    res.setPackageName(cp.getResourceName(df.getPrimaryFile()).replace('/','.'));
                    Iterator i = res.getClassifiers().iterator();
                    while (i.hasNext()) {
                        JavaClass clazz = (JavaClass) i.next();
                        if (clazz.getSimpleName().equals(originalName)) {
                            clazz.setSimpleName(dob.getName());
                        }
                    }
                }
                fail = false;
            } finally {
                JavaMetamodel.getDefaultRepository().endTrans(fail);
            }
        }
        return dob;
    }
    
    /* Renames all entries and changes their files to new ones.
    */
    protected FileObject handleRename (String name) throws IOException {
        if (!"package-info".equals(name) && !Utilities.isJavaIdentifier(name)) // NOI18N
            throw new IOException(NbBundle.getMessage(JavaDataObject.class, "FMT_Not_Valid_FileName", name));

        FileObject fo = super.handleRename(name);
        return fo;
    }

    /* Moves primary and secondary files to a new folder.
     * May ask for user confirmation before overwriting.
     * @param df the new folder
     * @return the moved primary file object
     * @throws IOException if there was a problem moving
     * @throws UserCancelException if the user cancelled the move
    */
    protected FileObject handleMove (DataFolder df) throws IOException {
        FileObject f = super.handleMove(df);
        SrcElementImpl src = (SrcElementImpl) getSource().getCookie(SrcElementImpl.class);
        src.invalidateDelegate();
        return f;
    }

    /* Creates new object from template.
    * @exception IOException
    */
    protected DataObject handleCreateFromTemplate (DataFolder df, String name) throws IOException {
        if (name == null) {
            // special case: name is null (unspecified or from one-parameter createFromTemplate)
            name = FileUtil.findFreeFileName(df.getPrimaryFile(),
                getPrimaryFile().getName(), "java"); // NOI18N
        } else if (!"package-info".equals(name) && !Utilities.isJavaIdentifier(name)) // NOI18N
            throw new IOException(NbBundle.getMessage(JavaDataObject.class, "FMT_Not_Valid_FileName", name));

        IndentFileEntry entry = (IndentFileEntry)getPrimaryEntry();
        entry.initializeIndentEngine();
        DataObject obj;

        boolean failed = true;
        JavaMetamodel.getDefaultRepository().beginTrans(true);
        try {
            obj = createDataObject(df, name);
            ClassPath cp = ClassPath.getClassPath(df.getPrimaryFile(),ClassPath.SOURCE);
                if (cp != null) {
                    String packageName = cp.getResourceName (df.getPrimaryFile(),'.',false);
                    assert packageName != null;
                    //if (isValidPackageName(packageName)) {
                      JavaModelPackage model = JavaMetamodel.getManager().resolveJavaExtent(cp.findOwnerRoot (obj.getPrimaryFile()));
                      ResourceClassImpl resClass = (ResourceClassImpl) model.getResource();
                      String resourceName = cp.getResourceName (obj.getPrimaryFile());
                      Resource res = resClass.resolveResource(resourceName, true, false);
                      res.setName(resourceName);
                      Iterator i = res.getClassifiers().iterator();
                      if (i.hasNext()) {
                          JavaClass clazz = (JavaClass) i.next();
                          clazz.setSimpleName(name);
                      }
                      //package name is set only if package name was already set
                      if (!"".equals(res.getPackageName())) {
                          res.setPackageName(packageName);
                      }
                }
                else {
                    //Destnation folder is not in project
                    //Skeep update but log
                    //TODO: Remove logging
                    ErrorManager.getDefault().log ("Can not update source while creating from template, destination folder "+
                        df.getPrimaryFile().getPath()+" is not in any project."); // NOI18N
                }
            failed = false;
        } finally {
            JavaMetamodel.getDefaultRepository().endTrans(failed);
            entry.setIndentEngine(null);
        }
        return obj;
    }

    /**
     * creates new file and its data object from template
     * @param df where to create
     * @param name file name
     * @return new data object
     * @throws IOException
     */ 
    private DataObject createDataObject(DataFolder df, String name) throws IOException {
        DataObject obj = super.handleCreateFromTemplate(df, name);
        
        // <workaround issue=56870>
        try {
            obj.setValid(false);
            obj = DataObject.find(obj.getPrimaryFile());
        } catch (PropertyVetoException e) {
            // do nothing
        }
        // </workaround>
        return obj;
    }

    /** Create the editor support for this data object.
    * By default, creates a <code>JavaEditor</code> with the source file entry;
    * subclasses may override this.
    * @return the editor support
    */
    protected JavaEditor createJavaEditor () {
        JavaEditor je = new JavaEditor (this);
        return je;
    }

    /** Provide node that should represent this data object.
    * This implementation creates and returns a {@link JavaNode}.
    * Subclasses may wish to return a specialized subclass of this node type.
    * You should probably make the default action be {@link OpenAction}.
    * @return the node representation for this data object
    */
    protected Node createNodeDelegate () {
        JavaNode node = new JavaNode (this);
        return node;
    }

    /** Get the parsed representation of this source file.
    * May not be fully parsed yet; the source element itself indicates its status.
    * @return the source element for this Java source file
    */
    public SourceElement getSource() {
        // return ((JavaParser)getCookie(JavaParser.class)).getSource();        
        if (source == null) {
            synchronized (lock) {
                if (source == null)
                    source = new SourceElement (new SrcElementImpl (this));
            }
        }
        return source;
    }

    /** Get the current editor support.
    * Ought not be subclasses; use {@link #createJavaEditor}.
    * @return the editor support
    */
    public JavaEditor getJavaEditor() {
        if (editorSupport == null) {
            synchronized (this) {
                editorSupport = createJavaEditor();
                if (parserGlue != null)
                    parserGlue.cloneableSupportCreated(editorSupport);
                else
                    initializeParsingSupport();
            }
        }
        return editorSupport;
    }

    // =============== The mechanism for regeisteing node factories ==============

    private static NodeFactoryPool explorerFactories;
    private static NodeFactoryPool browserFactories;
    private static ElementNodeFactory basicBrowser;

    /**
     * DO NOT USE THIS METHOD!!! <P>
     * This method is intended to be called only during initialization of java
     * module-provided node factories from the installation layer. It won't
     * be maintained for compatibility reasons.
     */
    synchronized static ElementNodeFactory createBasicExplorerFactory() {
        return JavaElementNodeFactory.DEFAULT;
    }

    /**
     * DO NOT USE THIS METHOD!!! <P>
     * This method is intended to be called only during initialization of java
     * module-provided node factories from the installation layer. It won't
     * be maintained for compatibility reasons.
     */
    synchronized static ElementNodeFactory createBasicBrowserFactory() {
        if (basicBrowser == null) {
            basicBrowser = org.netbeans.modules.java.ui.nodes.SourceNodes.createElementNodeFactory(
                    org.netbeans.modules.java.ui.nodes.SourceNodes.getBrowserFactory());
        }
        return basicBrowser;
    }

    public static ElementNodeFactory getExplorerFactory() {
        NodeFactoryPool pool = createExplorerFactory();
        ElementNodeFactory f = null;

        if (pool != null)
            f = pool.getHead();
        if (f == null)
            f = createBasicExplorerFactory();
        return f;
    }

    /**
     * 
     * @deprecated
     */ 
    public static ElementNodeFactory getBrowserFactory() {
        ErrorManager.getDefault().notify(
                ErrorManager.WARNING,
                new IllegalStateException("JavaDataObject.getBrowserFactory is deprecated. Use SourceNodes.getBrowserFactory() instead.") // NOI18N
        );
        return createBasicBrowserFactory();
    }

    static NodeFactoryPool createFactoryPool(String folderName, ElementNodeFactory def) {
        FileObject f = Repository.getDefault().getDefaultFileSystem().findResource(folderName);
	if (f == null)
	    return null;
        try {
            DataFolder folder = (DataFolder)DataObject.find(f).getCookie(DataFolder.class);
            return new NodeFactoryPool(folder, def);
        } catch (DataObjectNotFoundException ex) {
            return null;
        }
    }

    synchronized static NodeFactoryPool createBrowserFactory() {
        if (browserFactories != null)
            return browserFactories;
        browserFactories = createFactoryPool("/NodeFactories/java/objectbrowser", createBasicBrowserFactory()); // NOI18N
        return browserFactories;
    }

    synchronized static NodeFactoryPool createExplorerFactory() {
        if (explorerFactories != null)
            return explorerFactories;
        explorerFactories = createFactoryPool("/NodeFactories/java/explorer", createBasicExplorerFactory()); // NOI18N
        return explorerFactories;
    }

    /**
     * @deprecated use installation layer for registering a factory for the the whole
     * time a module is installed. Note: This feature will be dropped in the next
     * release.
     */
    public static void addExplorerFilterFactory( FilterFactory factory ) {
        NodeFactoryPool p = createExplorerFactory();
        if (p != null)
            p.addFactory(factory);
    }

    /**
     * @deprecated use installation layer for registering a factory for the the whole
     * time a module is installed. Note: This feature will be dropped in the next
     * release.
     */
    public static void removeExplorerFilterFactory( FilterFactory factory ) {
        NodeFactoryPool p = createExplorerFactory();
        if (p != null)
            p.removeFactory(factory);
    }

    /**
     * @deprecated use installation layer for registering a factory for the the whole
     * time a module is installed. Note: This feature will be dropped in the next
     * release.
     */
    public static void addBrowserFilterFactory(FilterFactory factory) {
        NodeFactoryPool p = createBrowserFactory();
        if (p != null)
            p.addFactory(factory);
    }

    /**
     * @deprecated use installation layer for registering a factory for the the whole
     * time a module is installed. Note: This feature will be dropped in the next
     * release.
     */
    public static void removeBrowserFilterFactory( FilterFactory factory ) {
        NodeFactoryPool p = createBrowserFactory();
        if (p != null)
            p.removeFactory(factory);
    }

    /* ===================== File -> model dependency handling ===================== */
    protected static boolean isValidPackageName(String str) {
        StringTokenizer tok = new StringTokenizer(str, "."); // NOI18N
        while (tok.hasMoreTokens()) {
            String part = tok.nextToken();
            if (!org.openide.util.Utilities.isJavaIdentifier(part))
                return false;
        }
        return true;
    }

    protected void primaryFileMoved(FileObject oldFile, FileObject newFile) {
        addSourceChangeListener(oldFile);
    }

    protected void primaryFileChanged() {
//        if (!getJavaEditor().isDocumentLoaded()) {
//            reparseResource(false);
//        }
    }

    // =============== Primary file monitoring ============
    private class PrimaryListener extends FileChangeAdapter implements PropertyChangeListener{
        public void propertyChange(final PropertyChangeEvent evt) {
            if (!initialized)
                return;
            String propName = evt.getPropertyName();
            if (PROP_PRIMARY_FILE.equals(propName)) {
                primaryFileMoved((FileObject)evt.getOldValue(), (FileObject)evt.getNewValue());
            } else if (PROP_NAME.equals(propName)) {
                previousFileName = (String)evt.getOldValue();
                primaryFileMoved(getPrimaryFile(), getPrimaryFile());
            }
        }

        public void fileChanged(FileEvent e) {
            if (!initialized)
                return;
            primaryFileChanged();
        }
    }


    /**
     * Extract the date of last modification of the source file. If the file is opened
     * in the editor AND marked as modified (in the editor), it returns true.
     * If the DataObject reports that it was modified, but the editor support knows
     * nothing about it, just ignore -- other parts of the DataObject are changed.
     */
    Date getLastModified() {
        SaveCookie c = (SaveCookie)getCookie(SaveCookie.class);
        if (c != null) {
            EditorCookie ck = (EditorCookie)getCookie(EditorCookie.class);
            if (ck != null && ck.isModified())
                return new Date();
        }
        return getPrimaryFile().lastModified();
    }

    /** Creates a Node.Cookie of given class. The method
     * may be called more than once.
     */
    public Node.Cookie createCookie(Class klass) {
        // all execution-related services -> getExecSupport
        if (klass.isAssignableFrom(JavaEditor.class)) {
            return getJavaEditor();
        }
        if (SourceCookie.class.isAssignableFrom(klass) ||
            JavaParser.class.isAssignableFrom(klass)) {
            if (initializeParsingSupport() == null)
                return null;
            if (klass.isAssignableFrom(parserGlue.getClass()))
                return parserGlue;
            else
                return parserGlue.getParser();
        }
        return null;
    }


    /**
     * @deprecated do not use this method, it will be removed in next release
     * @return empty Collection. JavaDataObject does not have any secondary entries
     */
    public Collection getCompiledClasses() {
        Thread.dumpStack();
        return Collections.EMPTY_LIST;
    }

    /*
    public String toString() {
        return "[JDO for " + getPrimaryFile() + "]"; // NOI18N
    }
     */
}
