/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java;

import java.beans.*;
import java.util.Collection;
import java.util.LinkedList;

import org.openide.actions.*;
import org.openide.cookies.OpenCookie;
import org.openide.cookies.FilterCookie;
import org.openide.nodes.*;
import org.openide.loaders.DataObject;
import org.openide.src.*;
import org.openide.src.nodes.*;
import org.openide.util.NbBundle;
import org.openide.util.actions.SystemAction;
import org.openide.*;

import org.netbeans.modules.java.tools.*;
import org.netbeans.modules.java.parser.JavaParser;

/** The implementation of hierarchy nodes factory for the java loader.
*
* @author Petr Hamernik
*/
class JavaElementNodeFactory extends DefaultFactory {
    /** Default instance of this factory. */
    public static final DefaultFactory DEFAULT = new JavaElementNodeFactory();

    /** Array of the actions of the java methods, constructors and fields. */
    private static final SystemAction[] DEFAULT_ACTIONS = new SystemAction[] {
                SystemAction.get(OpenAction.class),
                null,
                SystemAction.get(CutAction.class),
                SystemAction.get(CopyAction.class),
                null,
                SystemAction.get(DeleteAction.class),
                SystemAction.get(RenameAction.class),
                null,
                SystemAction.get(ToolsAction.class),
                SystemAction.get(PropertiesAction.class)
            };

    /** Array of the actions of the java intializers. */
    private static final SystemAction[] INITIALIZER_ACTIONS = new SystemAction[] {
                SystemAction.get(OpenAction.class),
                null,
                SystemAction.get(CutAction.class),
                SystemAction.get(CopyAction.class),
                null,
                SystemAction.get(DeleteAction.class),
                null,
                SystemAction.get(ToolsAction.class),
                SystemAction.get(PropertiesAction.class)
            };

    /** Array of the actions of the java classes. */
    private static final SystemAction[] CLASS_ACTIONS = new SystemAction[] {
                SystemAction.get(OpenAction.class),
                null,
                SystemAction.get(CutAction.class),
                SystemAction.get(CopyAction.class),
                SystemAction.get(PasteAction.class),
                null,
                SystemAction.get(DeleteAction.class),
                SystemAction.get(RenameAction.class),
                null,
                SystemAction.get(NewAction.class),
                null,
                SystemAction.get(ToolsAction.class),
                SystemAction.get(PropertiesAction.class)
            };

    /** This node can return current element factory as cookie */
    private final Node FACTORY_GETTER_NODE = new FactoryGetterNode();


    /** Create nodes for tree */
    private boolean tree = false;

    /** Creates new factory. */
    public JavaElementNodeFactory() {
        super(true);
    }

    /** If true generate nodes for tree.
    */
    public void setGenerateForTree (boolean tree) {
        this.tree = tree;
    }

    /** Returns true if generate nodes for tree.
    * @returns true if generate nodes for tree.
    */
    public boolean getGenerateForTree () {
        return tree;
    }

    /** Returns the node asociated with specified element.
    * @return ElementNode
    */
    public Node createMethodNode(MethodElement element) {
        MethodElementNode n = new MethodElementNode(element, true);
        n.setDefaultAction(SystemAction.get(OpenAction.class));
        n.setActions(DEFAULT_ACTIONS);
        return n;
    }

    /** Returns the node asociated with specified element.
    * @return ElementNode
    */
    public Node createConstructorNode(ConstructorElement element) {
        ConstructorElementNode n = new ConstructorElementNode(element, true);
        n.setDefaultAction(SystemAction.get(OpenAction.class));
        n.setActions(DEFAULT_ACTIONS);
        return n;
    }

    /** Returns the node asociated with specified element.
    * @return ElementNode
    */
    public Node createFieldNode(FieldElement element) {
        FieldElementNode n = new FieldElementNode(element, true);
        n.setDefaultAction(SystemAction.get(OpenAction.class));
        n.setActions(DEFAULT_ACTIONS);
        return n;
    }

    /** Returns the node asociated with specified element.
    * @return ElementNode
    */
    public Node createInitializerNode(InitializerElement element) {
        InitializerElementNode n = new InitializerElementNode(element, true);
        n.setDefaultAction(SystemAction.get(OpenAction.class));
        n.setActions(INITIALIZER_ACTIONS);
        return n;
    }

    /** Returns the node asociated with specified element.
    * @return ElementNode
    */
    public Node createClassNode(final ClassElement element) {

        if ( element == null ) {
            return FACTORY_GETTER_NODE;
        }


        ClassElementNode n;
        if (tree) {
            ClassChildren children = new ClassChildren( JavaDataObject.getBrowserFactory(), element);
            ClassElementFilter filter = new ClassElementFilter();
            n = new ClassElementNode(element, children ,true) {
                {
                    getCookieSet().add((FilterCookie) getChildren ());
                }
            };

            n.setElementFormat(new ElementFormat (
                                   NbBundle.getBundle (JavaElementNodeFactory.class).getString("CTL_Class_name_format")
                               ));

            // filter out inner classes
            filter.setOrder (new int[] {
                                 ClassElementFilter.FIELD,
                                 ClassElementFilter.CONSTRUCTOR,
                                 ClassElementFilter.METHOD,
                             });
            children.setFilter (filter);
        }
        else {
            n = (ClassElementNode) super.createClassNode(element);
        }
        n.setDefaultAction(SystemAction.get(OpenAction.class));
        n.setActions(CLASS_ACTIONS);
        return n;
    }

    protected Children createClassChildren( ClassElement element ) {
        return createClassChildren( element, JavaDataObject.getExplorerFactory() );
    }

    /**
     * This method will try to extract more information than the ordinary Error message.
     */
    public Node createErrorNode() {
        final Node n = super.createErrorNode();

        n.addNodeListener(new NodeListener() {
            public void propertyChange(PropertyChangeEvent e) {
                Node parent = n.getParentNode();
                DataObject d = null;
               
                if (parent == null)
                    return;
                n.removeNodeListener(this);
                do {
                    d = (DataObject)parent.getCookie(DataObject.class);
                    parent = parent.getParentNode();
                } while (parent != null && d == null);
                if (d == null)
                    return;
                setErrorDescription(n, (JavaParser)d.getCookie(JavaParser.class));
            }
            
            public void childrenAdded(NodeMemberEvent e) {}
            public void childrenRemoved(NodeMemberEvent e) {}
            public void childrenReordered(NodeReorderEvent e) {}
            public void nodeDestroyed(NodeEvent e) {}
        });
        return n;
    }

    private void setErrorDescription(Node n, JavaParser p) {
        if (p == null)
            return;
        SourceException e = p.getErrorCause();
        String msg = findErrorMessage(e);
        if (msg == null)
            return;
        
        if (e instanceof SourceException.IO) {
            n.setDisplayName(Util.getString("MSG_PARSE_ERROR_IO"));
        }
        n.setShortDescription(msg);
    }

    private String findErrorMessage(Throwable t) {
        if (t == null) {
            return null;
        }
        
        ErrorManager.Annotation[] ann = ErrorManager.getDefault().findAnnotations(t);
        if (ann == null)
            return t.getLocalizedMessage();
        for (int i = 0; i < ann.length; i++) {
            String normal = ann[i].getMessage();
            String localized = ann[i].getLocalizedMessage();
            if (localized != null)
                return localized;
            Throwable t2 = ann[i].getStackTrace();
            if (t2 == null)
                continue;
            localized = t2.getLocalizedMessage();
            if (localized != null) {
                if (!localized.equals(normal))
                    return localized;
            }
        }
        return null;
    }
    
    /** This is an unusuall use of Node and FilterCookie */

    private class FactoryGetterNode extends AbstractNode implements FilterCookie {

        FactoryGetterNode( ) {
            super ( Children.LEAF );
        }

        public synchronized Node.Cookie getCookie( Class clazz ) {
            if ( clazz == FilterFactory.class )
                return this;
            else
                return super.getCookie( clazz );
        }

        public Class getFilterClass() {
            return null;
        }

        public void setFilter( Object filter ) {}

        public Object getFilter( ) {
            if ( tree )
                return JavaDataObject.getBrowserFactory();
            else
                return JavaDataObject.getExplorerFactory();
        }

    }

}

