/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java;

import java.awt.Image;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.lang.ref.WeakReference;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import javax.jmi.reflect.InvalidObjectException;
import javax.jmi.reflect.JmiException;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.api.mdr.events.ExtentEvent;
import org.netbeans.api.mdr.events.MDRChangeEvent;
import org.netbeans.api.mdr.events.MDRChangeListener;
import org.netbeans.api.mdr.events.MDRChangeSource;
import org.netbeans.api.queries.FileBuiltQuery;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.java.settings.JavaSettings;
import org.netbeans.modules.java.tools.BadgeCache;
import org.netbeans.modules.java.ui.nodes.SourceNodes;
import org.netbeans.modules.java.ui.nodes.elements.SourceChildren;
import org.netbeans.modules.java.ui.nodes.elements.SourceEditSupport;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.ParsingListener;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataNode;
import org.openide.loaders.DataObject;
import org.openide.nodes.Children;
import org.openide.nodes.Node;
import org.openide.nodes.PropertySupport;
import org.openide.nodes.PropertySupport.ReadOnly;
import org.openide.nodes.Sheet;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;
import org.openide.util.Utilities;
import org.openide.util.WeakListeners;
import org.openide.util.datatransfer.NewType;

/**
 * The node representation of Java source files.
 */
public class JavaNode extends DataNode {
    
    private static final boolean DONT_RESOLVE_JAVA_BADGES = Boolean.getBoolean("perf.dont.resolve.java.badges");

    private static final String SHEETNAME_TEXT_PROPERTIES = "textProperties"; // NOI18N
    private static final String PROP_ENCODING = "encoding"; // NOI18N

    /** generated Serialized Version UID */
    private static final long serialVersionUID = -7396485743899766258L;

    private static final String ICON_BASE = "org/netbeans/modules/java/resources/"; // NOI18N
//    private static final String BARE_ICON_BASE = "org/netbeans/modules/java/resources/class"; // NOI18N

    private static final String BADGE_MAIN = ICON_BASE + "executable-badge"; // NOI18N
    private static final String BADGE_ERROR = ICON_BASE + "error-badge"; // NOI18N
    private static final String BADGE_NEEDS_COMPILE = ICON_BASE + "needs-compile"; // NOI18N
    
    private static final String[] ICON_ATTRS_NONE = {};
    
    protected static final String ICON_ATTR_NEEDS_COMPILE = "needsCompile"; // NOI18N
    /**
     * Icon attribute -- main class.
     */
    protected static final String   ICON_ATTR_MAIN = "mainClass"; // NOI18N
    
    /**
     * Icon attribute -- errors in the source.
     */
    protected static final String   ICON_ATTR_ERROR = "error"; // NOI18N

    private static final int ICON_REFRESH_DELAY_MSECS = 1000;

    private BadgeCache  iconCache;
    
    private HashSet     currentBadges;

    private StatePropagator statePropagator;
    
    /** Create a node for the Java data object using the default children.
    * @param jdo the data object to represent
    */
    public JavaNode (JavaDataObject jdo) {
	this(jdo, jdo.isTemplate() ? Children.LEAF : new JavaSourceChildren(jdo));
        currentBadges = new HashSet();
    }

    // T.B.: Workaround for issue #28623 - aggresively creating source element, classloading
    private static final class JavaSourceChildren extends SourceChildren {
        JavaDataObject jdo;
        private ExtentListener wExtentL;
        
        public JavaSourceChildren (JavaDataObject jdo) {
            super(SourceNodes.getExplorerFactory());
            this.jdo = jdo;
        }
	
        protected void removeNotify() {
            super.removeNotify();
            setElement (null);
        }
        
        protected Resource createResource() {
            Resource r = null;
            JavaModel.getJavaRepository().beginTrans(false);
            try {
                r = JavaModel.getResource(jdo.getPrimaryFile());
            } catch (JmiException je) {
                //do nothing
                //error node will be displayed
            }
            finally {
                JavaModel.getJavaRepository().endTrans();
            }
            return r;
        }
        
        public void setElement(final Resource element) {
            MDRepository repository = JavaModel.getJavaRepository();
            if (this.element == null && element != null) {
                if (wExtentL == null) {
                    wExtentL = new ExtentListener(this, (MDRChangeSource) repository);
                }
                ((MDRChangeSource) repository).addListener(wExtentL);
            } else if (element == null) {
                ((MDRChangeSource) repository).removeListener(wExtentL);
            }
            super.setElement(element);
        }
        
        // innerclass ...........................................................
        
        /** The listener for listening on extent unmount */
        private static final class ExtentListener extends WeakReference implements MDRChangeListener, Runnable {

            private final MDRChangeSource source;

            public ExtentListener(JavaSourceChildren referent, MDRChangeSource source) {
                super(referent, Utilities.activeReferenceQueue());
                this.source = source;
            }

            public void change(MDRChangeEvent e) {
                JavaSourceChildren sc = (JavaSourceChildren) get();
                if (sc == null) return;
                // if (sc.element == null || !sc.element.isValid()) return;
                if ((e instanceof ExtentEvent) && ((ExtentEvent) e).getType() == ExtentEvent.EVENT_EXTENT_DELETE) {
                    sc.setElement(JavaModel.getResource(sc.jdo.getPrimaryFile()));
                }
            }

            public void run() {
                source.removeListener(this);
            }

        }
        
    } // JavaSourceChildren ................

    /**
     * This method is called during node initialization and whenever the base icon is
     * replaced to refill the badge cache with custom badges. The JavaNode's implementation
     * adds executable and error badges. Override to add your own badges, or replace the
     * default ones.
     */
    protected void initializeBadges(BadgeCache cache) {
        cache.registerBadge(ICON_ATTR_MAIN, BADGE_MAIN, 9, 8);
        cache.registerBadge(ICON_ATTR_ERROR, BADGE_ERROR, 8, 8);
        cache.registerBadge(ICON_ATTR_NEEDS_COMPILE, BADGE_NEEDS_COMPILE, 16, 0);
    }
    
    /** Create a node for the Java data object with configurable children.
    * Subclasses should use this constructor if they wish to provide special display for the child nodes.
    * Typically this would involve creating a subclass of {@link SourceChildren} based
    * on the {@link JavaDataObject#getSource provided source element}; the children list
    * may have extra nodes {@link Children#add(Node[]) added}, either at the {@link Children.Keys#setBefore beginning or end}.
    * @param jdo the data object to represent
    * @param children the children for this node
    */
    public JavaNode (JavaDataObject jdo, Children children) {
        super (jdo, children);
        initialize();
    }
    
    public void setIconBase(String base) {
        super.setIconBase(base);
        synchronized (this) {
            iconCache = BadgeCache.getCache(base);
            initializeBadges(iconCache);
        }
    }

    private void initialize () {
        setIconBase(getBareIconBase());
        StateUpdater.registerNode(this);
    }

    /**
     * Schedules a refresh of node's icon, after ICON_REFRESH_DELAY_MSEC.
     * Requests is not scheduled if another one is still pending.
     * @deprecated
     */
    protected void requestResolveIcons() {
        StateUpdater.registerNode(this);
    }

    private void readObject(ObjectInputStream is) throws IOException, ClassNotFoundException {
        is.defaultReadObject();
        initialize();
    }
    
    static final void wrapThrowable(Throwable outer, Throwable inner, String message) {
        ErrorManager.getDefault().annotate(
            outer, ErrorManager.USER, null, message, inner, null);
    }

    /** Create the property sheet.
    * Subclasses may want to override this and add additional properties.
    * @return the sheet
    */
    protected Sheet createSheet () {
        Sheet sheet = super.createSheet();
        
        //if there is any rename handler installed
        //push under our own property 
        if (getRenameHandler() != null)
            sheet.get(Sheet.PROPERTIES).put(createNameProperty());

        Sheet.Set ps = new Sheet.Set();
        ps.setName(SHEETNAME_TEXT_PROPERTIES);
        ps.setDisplayName(Util.getString("PROP_textfileSetName")); // NOI18N
        ps.setShortDescription(Util.getString("HINT_textfileSetName")); // NOI18N
        ps.put(new PropertySupport.ReadWrite(PROP_ENCODING, 
            String.class, Util.getString("PROP_fileEncoding"), Util.getString("HINT_fileEncoding")) { // NOI18N
            public Object getValue() {
                String enc = Util.getFileEncoding0(getDataObject().getPrimaryFile());
                if (enc == null)
                    return "";
                else
                    return enc;
            }
            
            public void setValue(Object enc) throws InvocationTargetException {
                String encoding = (String)enc;
                if (encoding != null) {
                    if (!"".equals(encoding)) {
                        try {
                            Charset.forName(encoding);
                        } catch (IllegalArgumentException ex) {
                            // IllegalCharsetNameException or UnsupportedCharsetException
                            InvocationTargetException t =  new InvocationTargetException(ex);
                            wrapThrowable(t, ex,
                                MessageFormat.format(Util.getString("FMT_UnsupportedEncoding"), // NOI18N
                                    new Object[] {
                                        encoding
                                    }
                                ));
                            throw t;
                        }
                    } else
                        encoding = null;
                }
                try {
                    Util.setFileEncoding(getDataObject().getPrimaryFile(), encoding);
                    ((JavaDataObject)getDataObject()).firePropertyChange0(PROP_ENCODING, null, null);
                } catch (IOException ex) {
                    throw new InvocationTargetException(ex);
                }
            }
        });
        sheet.put(ps);
        // Add classpath-related properties.
        ps = new Sheet.Set();
        ps.setName("classpaths"); // NOI18N
        ps.setDisplayName(NbBundle.getMessage(JavaNode.class, "LBL_JavaNode_sheet_classpaths"));
        ps.setShortDescription(NbBundle.getMessage(JavaNode.class, "HINT_JavaNode_sheet_classpaths"));
        ps.put(new Node.Property[] {
            new ClasspathProperty(ClassPath.COMPILE,
                NbBundle.getMessage(JavaNode.class, "PROP_JavaNode_compile_classpath"),
                NbBundle.getMessage(JavaNode.class, "HINT_JavaNode_compile_classpath")),
            new ClasspathProperty(ClassPath.EXECUTE,
                NbBundle.getMessage(JavaNode.class, "PROP_JavaNode_execute_classpath"),
                NbBundle.getMessage(JavaNode.class, "HINT_JavaNode_execute_classpath")),
            new ClasspathProperty(ClassPath.BOOT,
                NbBundle.getMessage(JavaNode.class, "PROP_JavaNode_boot_classpath"),
                NbBundle.getMessage(JavaNode.class, "HINT_JavaNode_boot_classpath")),
        });
        sheet.put(ps);
        return sheet;
    }
    
    private Node.Property createNameProperty () {
        Node.Property p = new PropertySupport.ReadWrite (
                              DataObject.PROP_NAME,
                              String.class,
                              NbBundle.getMessage (DataObject.class, "PROP_name"),
                              NbBundle.getMessage (DataObject.class, "HINT_name")
                          ) {
                              public Object getValue () {
                                  return JavaNode.this.getName();
                              }

                              public Object getValue(String key) {
                                  if ("suppressCustomEditor".equals (key)) { //NOI18N
                                      return Boolean.TRUE;
                                  } else {
                                      return super.getValue (key);
                                  }
                              }
                              public void setValue(Object val) throws IllegalAccessException,
                                      IllegalArgumentException, InvocationTargetException {
                                  if (!canWrite())
                                      throw new IllegalAccessException();
                                  if (!(val instanceof String))
                                      throw new IllegalArgumentException();
                                  
                                  JavaNode.this.setName((String)val);
                              }
                              
                              public boolean canWrite() {
                                  return JavaNode.this.canRename();
                              }

                          };

        return p;
    }    
    
    /**
     * Displays one kind of classpath for this Java source.
     * Tries to use the normal format (directory or JAR names), falling back to URLs if necessary.
     * Displays corresponding sources instead of binaries where this is possible.
     */
    private final class ClasspathProperty extends PropertySupport.ReadOnly {
        
        private final String id;
        
        public ClasspathProperty(String id, String displayName, String shortDescription) {
            super(id, /*XXX NbClassPath would be preferable, but needs org.openide.execution*/String.class, displayName, shortDescription);
            this.id = id;
            // XXX the following does not always work... why?
            setValue("oneline", Boolean.FALSE); // NOI18N
        }

        public Object getValue() {
            ClassPath cp = ClassPath.getClassPath(getDataObject().getPrimaryFile(), id);
            if (cp != null) {
                StringBuffer sb = new StringBuffer();
                Iterator/*<ClassPath.Entry>*/ entries = cp.entries().iterator();
                while (entries.hasNext()) {
                    ClassPath.Entry entry = (ClassPath.Entry) entries.next();
                    URL u = entry.getURL();
                    append(sb, u);
                }
                return sb.toString();
            } else {
                return NbBundle.getMessage(JavaNode.class, "LBL_JavaNode_classpath_unknown");
            }
        }
        
        private void append(StringBuffer sb, URL u) {
            String item = u.toExternalForm(); // fallback
            if (u.getProtocol().equals("file")) { // NOI18N
                item = new File(URI.create(item)).getAbsolutePath();
            } else if (u.getProtocol().equals("jar") && item.endsWith("!/")) { // NOI18N
                URL embedded = FileUtil.getArchiveFile(u);
                assert embedded != null : u;
                if (embedded.getProtocol().equals("file")) { // NOI18N
                    item = new File(URI.create(embedded.toExternalForm())).getAbsolutePath();
                }
            }
            if (sb.length() > 0) {
                sb.append(File.pathSeparatorChar);
            }
            sb.append(item);
        }
    }

    /** Get the associated Java data object.
    * ({@link #getDataObject} is protected; this provides access to it.)
    * @return the data object
    */
    protected JavaDataObject getJavaDataObject() {
        return (JavaDataObject) getDataObject();
    }
    
    public Image getIcon(int type) {
        return iconCache.getIcon(super.getIcon(type), getBadges());
    }
    
    public Image getOpenedIcon(int type) {
        return iconCache.getIcon(super.getOpenedIcon(type), getBadges());
    }
    
    public void setName(String name) {
        RenameHandler handler = getRenameHandler();
        if (handler == null) {
            super.setName(name);
        } else {
            try {
                handler.handleRename(JavaNode.this, name);
            } catch (IllegalArgumentException ioe) {
                super.setName(name);
            }
        }
    }
    
    private static synchronized RenameHandler getRenameHandler() {
        Lookup.Result renameImplementations = Lookup.getDefault().lookup(new Lookup.Template(RenameHandler.class));
        List handlers = (List) renameImplementations.allInstances();
        if (handlers.size()==0)
            return null;
        if (handlers.size()>1)
            ErrorManager.getDefault().log(ErrorManager.WARNING, "Multiple instances of RenameHandler found in Lookup; only using first one: " + handlers); //NOI18N
        return (RenameHandler) handlers.get(0);
    }

    /** Get the icon base.
    * This should be a resource path, e.g. <code>/some/path/</code>,
    * where icons are held. Subclasses may override this.
    * @return the icon base
    * @see #getIcons
    * @deprecated the function is not consistent with openide's setIconBase() behaviour.
    * Given the icon badging, there's no need for composing icon names - see
    * {@link #setBadges}, {@link #getBadges} or {@link #getBareIconBase()}
    */
    protected String getIconBase() {
        return ICON_BASE;
    }
    
    /**
     * Returns the base resource name for the basic icon that identifies object type
     * to the user. Additional images may be superimposed over that icon using icon
     * badging specification. Override in subclasses to provide a different icon for
     * the node.
     * @return the icon base 
     */
    protected String getBareIconBase() {
        return getIconBase() + getIcons()[0];
    }

    /** Get the icons.
    * This should be a list of bare icon names (i.e. no extension or path) in the icon base.
    * It should contain five icons in order for:
    * <ul>
    * <li>a regular class
    * <li>a class with a main method
    * <li>a class with a parse error
    * <li>a JavaBean class
    * <li>a JavaBean class with a main method
    * </ul>
    * Subclasses may override this.
    * @return the icons
    * @see #getIconBase
    * @deprecated State icons are now handled using icon badging mechanism. If you need
    * to add or modify the badges attached by the java module, override {@link #getBadges}
    */
    protected String[] getIcons() {
        return new String[] { "class" }; // NOI18N
    }

    /**
     * Returns badges that should be combined with the base icon.
     * @return array of badges, can be null if no badges should be used.
     */
    protected String[] getBadges() {
        if (currentBadges.isEmpty())
            return null;
        return (String[])currentBadges.toArray(new String[currentBadges.size()]);
    }
    
    /**
     * Replaces the set of badges applied to the base icon. If the new set differs
     * from the current one, the new value is recorded and icon property changes are
     * fired.
     * @param badges list of badge identifiers; null to clear.
     */
    protected void setBadges(String[] badges) {
        if (badges == null)
            badges = new String[0];
        if (currentBadges.size() == badges.length) {
            boolean match = true;
            
            for (int i = 0; i < badges.length; i++) {
                if (!currentBadges.contains(badges[i])) {
                    match = false;
                    break;
                }
            }
            if (match)
                return;
        }
        currentBadges = new HashSet(Arrays.asList(badges));
        fireIconChange();
        fireOpenedIconChange();
    }
    
    /** Update the icon for this node based on parse status.
    * Called automatically at the proper times.
    * @see #getIconBase
    * @see #getBadges
    * @see #setBadges
    */
    protected void resolveIcons() {
        if (DONT_RESOLVE_JAVA_BADGES) 
            return ;
        if (this.statePropagator == null) {
            this.statePropagator = StatePropagator.getDefault(this);
        }
        JavaDataObject jdo = getJavaDataObject();
        FileObject fo = jdo.getPrimaryFile();
        boolean isTemplate;
        try {
            isTemplate = fo.getFileSystem().isDefault() && jdo.isTemplate();
        } catch (FileStateInvalidException fse) {
            isTemplate = false;
        }

        //TDB - avoid classloading by turning off parsing for templates, bug 28623
        if (isTemplate) 
            return;
        //Ignore virtual files #46348
        if (fo.isVirtual())
            return;

        final java.util.Collection badges = new java.util.ArrayList(3);
        final String desc;
        
        String pack2 = ""; // NOI18N
        boolean isValidResource = false;
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            JavaModel.setClassPath(fo);
            Resource resource = JavaModel.getResource(fo);
            isValidResource = resource != null && resource.isValid();
            if (isValidResource) {
                pack2 = resource.getPackageName();
            }
        } finally {
            JavaModel.getJavaRepository().endTrans();
        }
        if (isValidResource) {
//            if (!resource.getErrors().isEmpty()) {
//                desc = Util.getString("HINT_ParsingErrors");
//                badges.add(ICON_ATTR_ERROR);
            
            FileObject parent = fo.getParent();
            ClassPath cp = ClassPath.getClassPath(parent, ClassPath.SOURCE);
            String pack = (cp == null) ? null : cp.getResourceName(parent, '.',false); // NOI18N
            // check the package
            if (pack == null || !pack.equals(pack2)) {
                desc = new MessageFormat(Util.getString("FMT_Bad_Package")).format(new Object[] { pack2 });
                //badges.add(ICON_ATTR_ERROR);
            } else {
                desc = null;
            }
            
            if (hasMain()) {
                badges.add(ICON_ATTR_MAIN);
            }
        } else {
            if (currentBadges.contains(ICON_ATTR_MAIN)) {
                badges.add(ICON_ATTR_MAIN);
            }
            desc=getShortDescription();
        }
    

        // next: check whether the dataobject is up-to-date (if the settings permit that).
        // don't check it if the node corresponds to a template.
        if (!isTemplate) {
            String compBadge = resolveCompileBadge();
            if (compBadge != null)
                badges.add(compBadge);
        }

        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                setShortDescription(desc);
                if (badges.isEmpty()) {
                    setBadges(ICON_ATTRS_NONE);
                } else {
                    setBadges((String[])badges.toArray(new String[badges.size()]));
                }
            }
        });
    }
    
    private boolean hasMain() {
        FileObject fo = getJavaDataObject().getPrimaryFile();
        if(!fo.isValid())
            return false;
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            JavaModel.setClassPath(fo);
            Resource r = JavaModel.getResource(fo);

            return r!=null && !r.getMain().isEmpty(); 
        } finally {
            JavaModel.getJavaRepository().endTrans();
        }
    }
    
    protected String resolveCompileBadge() {
        if (!JavaSettings.getDefault().isCompileStatusEnabled()) {
            return null;
        }
        FileBuiltQuery.Status upToDate = FileBuiltQuery.getStatus(getDataObject().getPrimaryFile());
        return (upToDate != null && !upToDate.isBuilt()) ? ICON_ATTR_NEEDS_COMPILE : null;

    }

    public NewType[] getNewTypes() {
        if (getJavaDataObject().isJavaFileReadOnly()) {
            return super.getNewTypes();
        }
        
        return SourceEditSupport.createJavaNodeNewTypes(this);
    }

    private static final class StateUpdater implements Runnable {
        
        private static StateUpdater INSTANCE;
        /** nodes that will be processed in a scheduled {@link #task} */
        private Set registeredNodes;
        private final RequestProcessor QUEUE;
        private RequestProcessor.Task task;


        private StateUpdater() {
            registeredNodes = new HashSet(37);
            QUEUE = new RequestProcessor("Java Node State Updater"); // NOI18N
        }
        
        /**
         * registeres nodes to process their icons in async way. {@link JavaNode#resolveIcons()} is the callback used
         * by the updater that should compute icon.
         * @param node
         */ 
        public static void registerNode(JavaNode node) {
            init();
            INSTANCE.scheduleNode(node);
        }

        private static void init() {
            if (INSTANCE == null) {
                INSTANCE = new StateUpdater();
            }
        }
        
        private synchronized void scheduleNode(JavaNode node) {
            if (!registeredNodes.contains(node)) {
                registeredNodes.add(node);
                if (task == null) {
                    task = QUEUE.post(this, JavaNode.ICON_REFRESH_DELAY_MSECS);
                } else {
                    task.schedule(JavaNode.ICON_REFRESH_DELAY_MSECS);
                }
            }
        }

        public void run() {
            List nodes;
            synchronized(this) {
                nodes = new ArrayList(this.registeredNodes);
                this.registeredNodes.clear();
            }
            updateNodes(nodes);
        }

        private void updateNodes(List nodes) {
            for (Iterator it = nodes.iterator(); it.hasNext();) {
                JavaNode node = (JavaNode) it.next();
                if (node != null) {
                    node.resolveIcons();
                }
            }
        }

    }
    
    /**
     * Listens to all sources that may influence the state of java source presented by the {@link JavaNode}.
     * Changes of the state are propagated to the {@link StateUpdater}. Each JavaNode should keep own instance of the
     * propagator (see {@link #getDefault(JavaNode)})
     */ 
    private static final class StatePropagator
            extends WeakReference
            implements PropertyChangeListener, ChangeListener, ParsingListener, Runnable {

        private FileBuiltQuery.Status upToDate;
        private final DataObject dobj;
        private final PropertyChangeListener weakPropLsnr;
        private final ParsingListener weakParsingLsnr;

        private StatePropagator(JavaNode node) {
            super(node, Utilities.activeReferenceQueue());
            this.dobj = node.getDataObject();

            dobj.addPropertyChangeListener(this);
            registerBuildStatusListener();

            // #67841 - weak listening
            this.weakPropLsnr = WeakListeners.propertyChange(this, JavaSettings.getDefault());
            JavaSettings.getDefault().addPropertyChangeListener(weakPropLsnr);
            this.weakParsingLsnr = new WeakParsingListener(this);
            JavaMetamodel.addParsingListener(weakParsingLsnr);
        }

        /**
         * factory to create a propagator for particular node
         * @param node java node
         * @return the propagator
         */ 
        public static StatePropagator getDefault(JavaNode node) {
            return new StatePropagator(node);
        }
        
        public void propertyChange(PropertyChangeEvent evt) {
            Object src = evt.getSource();
            String name = evt.getPropertyName();
            JavaNode node = getJavaNode();
            if ((JavaSettings.PROP_SHOW_COMPILE_STATUS.equals(name) || name == null) && src == JavaSettings.getDefault()) {
                registerBuildStatusListener();
                StateUpdater.registerNode(node);
            } else if ((DataObject.PROP_PRIMARY_FILE.equals(name) || name == null) && src == node.getDataObject()) {
                // XXX #47035: workaround that allows to keep fresh FileBuiltQuery.Status even if the file is moved to another package
                registerBuildStatusListener();
                StateUpdater.registerNode(node);
            }
        }

        public void stateChanged(ChangeEvent e) {
            JavaNode node = getJavaNode();
            if (node != null && JavaSettings.getDefault().isCompileStatusEnabled()) {
                StateUpdater.registerNode(node);
            }
        }

        public void resourceParsed(Resource rsc) {
            JavaNode node = getJavaNode();
            if (node != null) {
                FileObject fo = node.getDataObject().getPrimaryFile();
                try {
                    if (fo.getPath().endsWith(rsc.getName()) && fo.equals(JavaModel.getFileObject(rsc))) {
                        StateUpdater.registerNode(node);
                    }
                } catch (InvalidObjectException ioe) {
                    //do nothing
                    //resource is no more valid
                }
            }
        }

        public void run() {
            // here goes GC
            JavaSettings.getDefault().removePropertyChangeListener(this.weakPropLsnr);
            this.dobj.removePropertyChangeListener(this);
            if (this.upToDate != null) {
                this.upToDate.removeChangeListener(this);
            }
        }
        
        private JavaNode getJavaNode() {
            return (JavaNode) get();
        }
        
        private void registerBuildStatusListener() {
            JavaNode node = getJavaNode();
            if (node != null) {
                FileBuiltQuery.Status status = this.upToDate;
                if (status != null) {
                    status.removeChangeListener(this);
                }
                if (JavaSettings.getDefault().isCompileStatusEnabled()) {
                    status = FileBuiltQuery.getStatus(node.getDataObject().getPrimaryFile());
                    if (status != null) {
                        status.addChangeListener(this);
                    }
                } else {
                    status = null;
                }
                this.upToDate = status;
            }
        }
    }

    private static final class WeakParsingListener extends WeakReference
            implements Runnable, ParsingListener {

        public WeakParsingListener(ParsingListener delegate) {
            super(delegate, Utilities.activeReferenceQueue());
        }

        public void run() {
            JavaMetamodel.removeParsingListener(this);
        }

        public void resourceParsed(Resource rsc) {
            ParsingListener delegate = (ParsingListener) get();
            if (delegate != null) {
                delegate.resourceParsed(rsc);
            }
        }
    }
}
