/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.bridge;

import java.util.*;

import org.openide.src.*;

/**
 * The Binding (sub-)classes give individual model elements the I/O capabilities (e.g.
 * a Document or a Stream). Each element is, in fact, a composite of the element itself
 * and the I/O binding of that element. All model change operations are transformed into
 * simpler calls on the binding objects.
 *
 * @author sdedic
 * @version 0.1
 */
public interface Binding {
    
    /** Binds any MemberElement to the underlying text.
     */
    public interface Member extends Binding {
        /**
         * Requests change of member's modifiers.
         */
        public void changeModifiers(int newMods) throws SourceException;
        
        /**
         * Requests a change of member's name.
         */
        public void changeName(Identifier name) throws SourceException;
    }
    
    public interface Body {
        public String getBodyContent() throws SourceException;
        public void changeBody(String bodyString) throws SourceException;
        public void copyBody(String bodyString);
    }

    /** Binds an initializer to the source.
     */
    public interface Initializer extends Binding, Body {
        /**
         * Changes static <-> nonstatic
         */
        public void changeStatic(boolean enable) throws SourceException;
    }
    
    public interface Field extends Member {
        /** Changes the declared type of the field.
         */
        public void changeType(Type newType) throws SourceException;
        
        /** Changes contents of the initializer of the field.
         */
        public void changeInitializer(String newInitializer) throws SourceException;
    }

    /** Binds a method, or a constructor to the source.
     */
    public interface Method extends Member, Body {
        /** Retrieves the text of the body.
         */
        public String getBodyContent() throws SourceException;

        /** Makes the method abstract. The implementation should erase the current
         * body definition and replace it with an abstract tag (; in the source representation)
         */
        public void makeAbstract() throws SourceException;
        
        /** Create a nonabstract body containing the passed text.
         */
         public void createBody(String bodyText) throws SourceException;
         
        /** Changes the return type declaration.
         */
        public void changeReturnType(Type type) throws SourceException;
        
        /** Changes parameter list for the method.
         */
        public void changeParameters(MethodParameter[] params) throws SourceException;
        
        /** Changes exception list for the method.
         */
        public void changeExceptions(Identifier[] exceptions) throws SourceException;
    }
    
    /** Container interface that manages contained bindings. Currently only reorder operation
     * is supported.
     */
    public interface Container {
        /**
         * Initializes a new binding for the element so the element is stored after the
         * `previous' binding, if that matters to the binding implementation.
         * @param toInitialize the binding that is being initialized & bound to the storage.
         * @param previous marker spot, the binding that should precede the new one. 
         */
        public void insert(Binding toInitialize, Binding previous) throws SourceException;
        
        /** Replaces the slot contents with another element (different type permitted ?)
         */
        public void replace(Binding oldBinding, Binding newBinding) throws SourceException;
        
        /** The map contains mapping from target places to their new contents.
         */
        public void reorder(Map fromToMap) throws SourceException;
        
        /** Determines, if the executing code is allowed to insert after the specified
         * binding.
         */
        public boolean canInsertAfter(Binding b);

        /**
         * Changes container's contents as one operation, given the information in 
         * the event object.
         */
        public void changeMembers(MultiPropertyChangeEvent evt) throws SourceException;
    }
    
    public interface Class extends Member, Container {
        /**
         * Changes the superclass' name.
         */
        public void changeSuperclass(Identifier id) throws SourceException;
     
        /** Rewrites some interfaces from class' implements property.
         */
        public void changeInterfaces(Identifier[] replaceWith) throws SourceException;

        /** Changes class into an interface and vice-versa. Although classes and interfaces
         * are clearly separated in the language specs, they represent the same concept with
         * some minor differencies.
         */
        public void changeClassType(boolean properClass) throws SourceException;
    }
    
    public interface Import extends Binding {
        public void changeImport(org.openide.src.Import i) throws SourceException;
    }
    
    public interface Source extends Binding {
        public void changePackage(Identifier id) throws SourceException;
        public Binding.Container    getImportsSection();
        public Binding.Container    getClassSection();
    }
    
    public void changeJavaDoc(JavaDoc content) throws SourceException;
}

