/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.bridge;

import java.util.StringTokenizer;

import org.openide.src.*;

/** Implementation of JavaDoc comment blocks in Java loader.
*
* @author Petr Hamernik, Petr Hrebejk
*/
class JavaDocImpl extends Object implements JavaDoc, java.io.Serializable {

    /** Holds the memory implementation of JavaDoc */
    JavaDoc javaDoc;

    /** The element which this javadoc belongs to.
    * It is called when javadoc changed and should be regenerated
    */
    ElementImpl impl;

    private static final long serialVersionUID = -4007873715598624939L;
    
    /** Flag for the empty javadoc (element has NO javadoc
    * in the source code)
    */
    //boolean empty;


    JavaDocImpl( ElementImpl impl ) {
        this.impl = impl;
    }

    /** Creates new JavaDoc
    * @param rawText the pre-parsed text of the javadoc comment
    *         provided by the JavaParser
    * @param impl The implementation
    */
    JavaDocImpl(String rawText, ElementImpl impl) {
        javaDoc = JavaDocSupport.createJavaDoc( rawText );
        this.impl = impl;
    }

    public void clearJavaDoc() throws SourceException {
        javaDoc.clearJavaDoc();
        impl.setJavaDocText (null, true);
    }

    public boolean isEmpty() {
        return javaDoc.isEmpty();
    }

    /** Get the entire text of the comment.
    * @return the whole text
    */
    public String getRawText () {
        return javaDoc.getRawText();
    }


    /** Set the raw text of the comment.
    * @param s the whole text to set
    * @exception SourceException if the modification cannot be performed
    */
    public void setRawText (String s) throws SourceException {
        impl.setJavaDocText(s, true);
    }

    /** Updates internal structures holding the tags... */
    private void updateTags() {
        //PENDING
    }

    /** Get the actual text, cleared of all (non-inline) tags.
    * @return the plain text
    */
    public String getText () {
        return javaDoc.getText();
    }


    /** Set the actual text.
    * @param s the actual text, without any (non-inline) tags
    * @exception SourceException if the modification cannot be performed
    */
    public void setText (String s) throws SourceException {
        impl.setJavaDocText(s, false);
        /*
        String oldText = getText();
        try {
            javaDoc.setText( s );
        } catch (SourceException ex) {
            // something went wrong.
            javaDoc.setText(oldText);
            throw ex;
        }
         */
    }

    protected void changeJavaDocText(String s, boolean raw) throws SourceException {
        if (raw) {
            javaDoc.setRawText(s);
        } else {
            javaDoc.setText(s);
        }
    }
    
    /** Gets all tags from comment.
     */
    public JavaDocTag[] getTags() {
        return javaDoc.getTags();
    }

    /** Gets all tags of given name
     */
    public JavaDocTag[] getTags(String name) {
        return javaDoc.getTags( name );
    }

    /** Adds removes or sets tags used in this comment
     * @param elems the new initializers
     * @param action {@link #ADD}, {@link #REMOVE}, or {@link #SET}
     * @exception SourceException if impossible
     */
    public void changeTags(JavaDocTag[] tags, int action) throws SourceException {
        impl.changeJavaDocTags(tags, action);
    }
    
    protected void changeJavaDocTags(JavaDocTag[] tags, int action) 
        throws SourceException {        
        javaDoc.changeTags(tags, action);
    }

    /** Gets all @see tags
     */
    public JavaDocTag.See[] getSeeTags() {
        return javaDoc.getSeeTags();
    }

    /** Regenerates the text in the element.
    * @exception SourceException if the modification cannot be performed
    public void regenerateSource() throws SourceException {
        impl.javaDocChanged();
    }
    */

    /** The JavaDoc of a class.
    * Class javadoc adds no special tags.
    */
    static class Class extends JavaDocImpl implements JavaDoc.Class {
        
        private static final long serialVersionUID = 5651952150233392489L;
        
        public Class(String rawText, ElementImpl impl) {
            super( impl );
            javaDoc = JavaDocSupport.createClassJavaDoc( rawText );
        }
    }

    /** The JavaDoc of a field.
    * <p>Currently adds special @SerialField tag
    */
    static class Field extends JavaDocImpl implements JavaDoc.Field {

        private static final long serialVersionUID = -7737074261842273983L;
        
        public Field(String rawText, ElementImpl impl) {
            super( impl );
            javaDoc = JavaDocSupport.createFieldJavaDoc( rawText );
        }

        /** Gets SerialField tags.
        */
        public JavaDocTag.SerialField[] getSerialFieldTags() {
            return  ((JavaDoc.Field)javaDoc).getSerialFieldTags();
        }

    }

    /** The JavaDoc of a method. Adds two special tags: @param tag and @throws tag.
    */
    static class Method extends JavaDocImpl implements JavaDoc.Method {

        private static final long serialVersionUID = 6902210486353445027L;
        
        public Method(String rawText, CallableImpl impl) {
            super( impl );
            javaDoc = JavaDocSupport.createMethodJavaDoc( rawText );
        }

        /** Gets param tags.
        */
        public JavaDocTag.Param[] getParamTags() {
            return  ((JavaDoc.Method)javaDoc).getParamTags();
        }

        /** Gets throws tags.
        */
        public JavaDocTag.Throws[] getThrowsTags() {
            return  ((JavaDoc.Method)javaDoc).getThrowsTags();
        }
    }
}
