/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.codegen;

import java.util.*;
import java.lang.reflect.Modifier;

import javax.swing.text.Position;
import javax.swing.text.StyledDocument;

import org.openide.src.*;
import org.openide.text.CloneableEditorSupport;
import org.openide.text.PositionBounds;
import org.openide.text.PositionRef;

import org.netbeans.modules.java.bridge.Binding;

/**
 *
 * @author  svata
 * @version
 */
class Method extends Member implements Binding.Method {
    boolean constructor;
    String preexistingBody;
    
    public Method(ConstructorElement el, SourceText s) {
        super(el, s);
        constructor = (!(el instanceof MethodElement));
    }
    
    public void create(PositionBounds b) throws SourceException {
        super.create(b);
        preexistingBody = null;
    }
    
    protected int classifyProperty(String name) {
        if (name == PROP_BODY)
            return CLASS_BODY;
        else
            return CLASS_HEADER;
    }
    
    private MethodElement cloneMethod() {
        MethodElement el = new MethodElement();
        copyProperties(el);
        try {
            el.setReturn(((MethodElement)getElement()).getReturn());
        } catch (SourceException ex) {
            // should NOT happen
        }
        return el;
    }
        
    protected Element cloneElement() {
        Element orig = getElement();
        ConstructorElement el = orig instanceof MethodElement ?
            new MethodElement() : new ConstructorElement();
        copyProperties(el);
        return el;
    }
    
    protected void copyProperties(ConstructorElement target) {
        ConstructorElement my = (ConstructorElement)getElement();
        try {
            target.setName(my.getName());
            target.setParameters(my.getParameters());
            target.setModifiers(my.getModifiers());
            target.setExceptions(my.getExceptions());
            if (my instanceof MethodElement) 
                ((MethodElement)target).setReturn(((MethodElement)my).getReturn());
        } catch (SourceException ex) {
            // should NOT happen
        }
    }
    
    
    /** Changes exception list for the method.
     */
    public void changeExceptions(Identifier[] exceptions) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        ConstructorElement el = (ConstructorElement)cloneElement();
        el.setExceptions(exceptions);
        regenerateHeader(el);
    }
    
    /** Changes parameter list for the method.
     */
    public void changeParameters(MethodParameter[] params) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        ConstructorElement el = (ConstructorElement)cloneElement();
        el.setParameters(params);
        regenerateHeader(el);
    }
    
    /** Changes the return type declaration.
     */
    public void changeReturnType(Type type) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        MethodElement el = cloneMethod();
        el.setReturn(type);
        regenerateHeader(el);
    }

    /**
     * Requests change of member's modifiers.
     */
    public void changeModifiers(int newMods) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        
        ConstructorElement el = (ConstructorElement)cloneElement();
        boolean isAbstract = Modifier.isAbstract(el.getModifiers());
        boolean newAbstract = Modifier.isAbstract(newMods);
        el.setModifiers(newMods);
        regenerateHeader(el);
        if (isAbstract!=newAbstract) {
            if (newAbstract)
                el.setBody(null);
            else {
                if (((ConstructorElement)getElement()).getBody()!=null)
                    return;
                el.setBody("");
            }
            regenerateBody(el);
        }
    }    

    protected void regenerateBody(Element bean) throws SourceException {
        ConstructorElement now = (ConstructorElement)bean;
        ConstructorElement el = (ConstructorElement)getElement();
        boolean existed = el.getBody() != null;
        String newBody = now.getBody();
        
        if (existed && newBody == null) {
            makeAbstract();
        } else if (!existed && newBody != null) {
            createBody(newBody);
        } else if (existed) {
            changeBody(newBody);
        }
    }

    /** Create a nonabstract body containing the passed text.
     */
    public void createBody(String bodyText) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        final StyledDocument doc = findDocument();
        final String s = CodeGenerator.normalizeBody(
            ElementBinding.convertNewlines(bodyText), true);
        
        source.runAtomic(getElement(), new ExceptionRunnable() {
            public void run() throws Exception {
                int afterHeader = headerBounds.getEnd().getOffset();
                String text = CodeGenerator.formatText(doc, afterHeader,
                    " " + s); // NOI18N
                int end = afterHeader + text.length();

                CloneableEditorSupport supp = source.getEditorSupport();
                doc.insertString(afterHeader, text, null);
                bodyBounds = new PositionBounds(
                    supp.createPositionRef(afterHeader + 1, Position.Bias.Forward),
                    supp.createPositionRef(end, Position.Bias.Backward)
                );
                doc.remove(end, wholeBounds.getEnd().getOffset() - end);
            }
        });
    }
    
    /** Retrieves the text of the body.
     */
    public String getBodyContent() throws SourceException {
        if (preexistingBody != null)
            return preexistingBody;
        
        if (bodyBounds != null) {
            return CodeGenerator.readTextBounds(bodyBounds);
        } else {
            return null;
        }
    }

    /** Make abstract means removing the body part and substituting
     * a semicolon.
     */
    public void makeAbstract() throws SourceException {
        final StyledDocument doc = findDocument();

        if (!source.isGeneratorEnabled())
            return;
        source.runAtomic(getElement(), new ExceptionRunnable() {
            public void run() throws Exception {
                int afterHeader = headerBounds.getEnd().getOffset();
                int end = wholeBounds.getEnd().getOffset();

                doc.insertString(afterHeader, ";", null); // NOI18N
                doc.remove(afterHeader + 1, end - afterHeader);
                bodyBounds = null;

                // PENDING: register UNDO event with the document's undo manager,
                // so that the bodyBounds are restored!!!
            }
        });
    }
    
    /**
     * Changes the body contents. The method assumes that there is already some body.
     */
    public void changeBody(String bodyString) throws SourceException {
        if (!source.isGeneratorEnabled())
            return;
        final String normalized = CodeGenerator.normalizeBody(
            ElementBinding.convertNewlines(bodyString), false)+"}"; // NOI18N
        final ElementBinding b = this;
        
        source.runAtomic(getElement(), new ExceptionRunnable() {
            public void run() throws Exception {
                String formatted = CodeGenerator.formatText(
                    CodeGenerator.getDocument(b),
                    bodyBounds.getBegin(),
                    normalized);
                String formattedBody = formatted.substring(0,formatted.lastIndexOf('}'));
                CodeGenerator.fillTextBounds(bodyBounds,formattedBody);
            }
        });
    }
    
    /**
     * Updates the storage binding object from an external SourceText.
     */
    public void updateFrom(Binding other) {
    }
    
    public void copyBody(String bodyString) {
        preexistingBody = bodyString;
    }
}
