/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.parser;

import java.beans.PropertyChangeListener;
import java.io.Reader;
import java.io.InputStream;
import java.io.IOException;
import javax.swing.event.ChangeListener;

import org.openide.cookies.SourceCookie;
import org.openide.nodes.Node;
import org.openide.src.*;
import org.openide.util.Task;

import org.netbeans.modules.java.bridge.LangModel;

/**
 *
 * @author  sdedic
 * @version 
 */
public interface JavaParser extends SourceCookie {
    /**
     * This priority level is recommended for background operations, that the user
     * does not need to wait for.
     */
    public static final int    PRIORITY_BACKGROUND = Thread.MIN_PRIORITY;
    /**
     * Normal priority is recommended for parse operations that needs to be completed soon,
     * but should not block the IDE. This priority is internally used in prepare() call.
     */
    public static final int    PRIORITY_NORMAL = Thread.MIN_PRIORITY + 1;
    /**
     * Recommended priority for carrying out blocking calls. This priority is used
     * by blocking calls on SourceElement.
     */
    public static final int    PRIORITY_DEMAND = Thread.MAX_PRIORITY - 1;
    /**
     * Recommended priority for hierarchy refresh operations. This priority is used
     * in Java module in refresh operation after the user stops editing for a period of
     * time.
     */
    public static final int    PRIORITY_REFRESH = PRIORITY_NORMAL;

    /**
     * Shallow parser request. 
     */
    public static final String SHALLOW_PARSER = "shallow"; // NOI18N
    /**
     * Deep parser request.
     */
    public static final String DEEP_PARSER = "deep"; // NOI18N
    /**
     * MDR parser request.
     */
    public static final String MDR_PARSER = "mdr"; // NOI18N
    /**
     * Adds a Listener on the parser.
     */
    public void addPropertyChangeListener(PropertyChangeListener l);
    
    /**
     * Removes the listener from the parser.
     */
    public void removePropertyChangeListener(PropertyChangeListener l);
    
    /**
     * Listens on parser state changes (more events than just PropertyChanges!)
     */
    public void addChangeListener(ChangeListener l);
    
    /**
     * Removes the change listener from the parser.
     */
    public void removeChangeListener(ChangeListener l);
    
    /**
     * Returns the Element representing the source.
     */
    public SourceElement    getSource();
    
    /**
     * Prepares the source hierarchy; after the task is finished, there are some
     * parsed data. If the model already exists, the task returned is already finished.
     * The returned task, after it completes, also guarantees that the model will not
     * be garbage collected.
     */
    public Task             prepare();
    
    /**
     * Schedules a parse task with various parameters.
     * @param force schedules a parse even if the underlying document contains no
     * uparsed changes. The method returns a task that you can wait for, or attach
     * TaskListeners to. See {@link #prepare} for more information.
     * @param acceptErrors force model update even if the parsed data indicate syntax
     * errors.
     * @param priority priority of the task.
     * @return task for synchronization purposes. The task is marked finished after the
     * model is created or refreshed.
     */
    public Task             parse(int priority, boolean force, boolean acceptErrors);

    /**
     * Schedules a parse task with various parameters.
     * @param force schedules a parse even if the underlying document contains no
     * uparsed changes. The method returns a task that you can wait for, or attach
     * TaskListeners to. See {@link #prepare} for more information.
     * @param acceptErrors force model update even if the parsed data indicate syntax
     * errors.
     * @param priority priority of the task.
     * @return task for synchronization purposes. The task is marked finished after the
     * model is created or refreshed.
     */
    public Task             parse(int priority, boolean force, boolean acceptErrors, ParsableObjectRequest request);
    
    /**
     * Returns current running parser task, if there is no task running returns FinishedTask
     * @return task for synchronization purposes.
     */
    public Task getCurrentTask();
    
    /**
     * Returns the overall current parser's status. The returned value is one of
     * the manifested constants SourceElement.STATUS_*.
     * @return status information
     */
    public int              getStatus();
    
    /**
     * Retrieves the exception that describes the last failure to parse the
     * input. This will be mostly SourceException.IO.
     */
    public SourceException  getErrorCause();
    
    /**
     * Returns the <B>real</B> implementation of the source provided by the data model.
     */
    public SourceElement.Impl   getSourceImpl();
    
    /**
     * Returns the model the parser operatges on.
     */
    public LangModel            getModel();

    /**
     * Interface that the support uses to talk to the outside world.
     */
    public interface Env {
        /**
         * Returns the FileObject corresponding to the source, if there's one.
         * The method may return null if there is no FileObject == the source
         * is only in memory.
         * @return FileObject or <code>null</code>
         */
        public org.openide.filesystems.FileObject   getSourceFile();

        /**
         * Returns the name of the source, if one was assigned. If {@link #getSourceFile}
         * returns non-null, then this method must return getSourceFile().getNameExt().
         * The returned value may used for error reporting when the passed source is
         * not file-based.
         *
         * @return name of the source or <code>null</code>
         */
        public String       getSourceName();

        /** Retrieves the source text for parsing. Can throw IOException, if the
         * source cannot be retrieved.
         */
        public Reader   getSourceText() throws IOException;

        /**
         * Environment can find precompiled classes to speed up name resoltion
         */
        public InputStream  findCompiledClass(String classFQN);
        
        /**
         * Finds appropriate cookie for the element.
         */
        public Node.Cookie  findCookie(SourceElement el, Class cls);
        
        /**
         * Annotate the throwable with a localized message. The flag controls whether
         * the error is a user-level one, or it is an internal failure.
         */
        public void annotateThrowable(Throwable t, String locMessage, boolean user);
        
        /**
         * Annotates wrapping throwable with the nested one - whatever that means to the
         * environment.
         */
        public void annotateThrowable(Throwable wrapper, Throwable nested);
    }
}
