/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.java.parser;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.*;

import javax.swing.event.ChangeListener;
import javax.swing.event.ChangeEvent;
import org.netbeans.api.java.classpath.ClassPath;

import org.openide.filesystems.FileObject;
import org.openide.text.CloneableEditorSupport;

import org.netbeans.modules.java.ElementFactory;
import org.netbeans.modules.java.ErrConsumer;
import org.netbeans.modules.java.JavaDataObject;
import org.netbeans.modules.java.Util;
import org.netbeans.modules.java.settings.JavaSettings;

/**
 *
 * @author  sdedic
 * @version
 */
public class ParseSourceRequest implements ParsableObjectRequest, ErrConsumer {
    public static final int     STATE_WAITING = 0;
    public static final int     STATE_READING = 1;
    public static final int     STATE_CANCELLED = 2;
    public static final int     STATE_ANALYSIS = 3;
    public static final int     STATE_UPDATING = 4;
    public static final int     STATE_COMPLETED = 10;

    private static final int READ_THRESHOLD = 2048;

    JavaParser.Env          environment;
    ChangeListener          listener;
    int                     state;
    boolean                 valid;
    int                     syntaxErrors;
    ElementFactory          builder;
    CloneableEditorSupport  editSupp;

    private List            errors = null;
    private Object          parserType;

    /**
     * For compatibility -- remove when issue #31581 is fixed!
     */
    private String          sourceName;

    public ParseSourceRequest() {
        this((Object)JavaParser.SHALLOW_PARSER);
    }

    /**
     * @deprecated The filename should be provided by parser's environment, rather than
     * by parser request. Just use {@link ParseSourceRequest(java.lang.Object)}.
     *
     * @param filename
     */
    public ParseSourceRequest(String filename) {
        this();
        this.sourceName = filename;
    }

    public ParseSourceRequest(Object parserType) {
        JavaSettings settings=JavaSettings.getDefault();

        this.parserType = parserType;
        state = STATE_WAITING;
        valid = true;
    }

    ParseSourceRequest(JavaParser.Env env, CloneableEditorSupport editSupp) {
        this();
        this.editSupp = editSupp;
        this.environment = env;
    }

    public synchronized void addChangeListener(ChangeListener l) throws TooManyListenersException {
        if (listener != null)
            throw new TooManyListenersException();
        listener = l;
    }

    public synchronized void removeChangeListener(ChangeListener l) {
        if (listener == l)
            listener = null;
    }

    public void setEnvironment(JavaParser.Env env) {
        environment = env;
    }

    public void setEditorSupport(CloneableEditorSupport editor) {
        editSupp = editor;
    }

    /**
     * Notifies the request that the source text has been changed. This causes
     * cancellation of the request in some cases.
     */
    public void sourceChanged() {
        if (state == STATE_READING) {
            // cancel the request only in reading state; if the text is already
            // read, the request can still be cariied out.
            valid = false;
        }
    }

    public void modelChanged() {
        if (state != STATE_WAITING && state != STATE_COMPLETED) {
            valid = false;
        }
    }

    public void setSyntaxErrors(int errors) {
        this.syntaxErrors = errors;
    }

    public int getSyntaxErrors() {
        return syntaxErrors;
    }

    public void setSemanticErrors(int errors) {
        // ignore - for now.
    }

    /**
     * DocumentModelBuilder actually carries out tasks associated with the Factory.
     */
    public ElementFactory getFactory() {
        if (builder == null)
            builder = createBuilder(editSupp);
        return builder;
    }

    protected ElementFactory createBuilder(CloneableEditorSupport supp) {
        return new DocumentModelBuilder(supp);
    }

    public void notifyReschedule() {
        // clear old data.
        builder = null;
        enterState(STATE_WAITING);
    }

    protected void enterState(int state) {
        this.state = state;
        if (listener != null)
            listener.stateChanged(new ChangeEvent(this));
    }

    /**
     * The method is implemented so that it reads the whole contents from the environment's
     * Reader and returns the resulting char array.
     */
    public char[] getSource() throws IOException {
        // unsynchronized; the exact sequence really does not matter here too much;
        // the real problem arises that AFTER the contents is read.
        valid = true;
        enterState(STATE_READING);

        Reader r = environment.getSourceText();
        char[] buf=Util.readContents(r);
        ElementFactory builder = getFactory();
        if (builder instanceof DocumentModelBuilder) {
            ((DocumentModelBuilder)builder).setContent(buf, editSupp.isDocumentLoaded());
        }
        return buf;
    }

    /** This should locate appropriate .class containing the class 'className'.
     * The current implementation will cowardly return false.
     */
    public InputStream findCompiledClass(String className) {
        return environment.findCompiledClass(className);
    }

    /**
     * Returns true, IF the request was not invalidated by the ParserSupport,
     * or because of change in the document while parsing.
     */
    public boolean isValid() {
        return this.valid;
    }

    public boolean needsProcessing() {
        // always needs to process (?)
        return this.valid;
    }

    public void notifyStart() {
    }

    public void notifyComplete() {
        // safepoint: everything was processed.
        enterState(STATE_COMPLETED);
        if (errors==null && JavaParser.DEEP_PARSER.equals(getParserType()))
            errors=new ArrayList();
        // errors == null IS intentional for the shallow parser. Before you change it
        // be aware of JavaEditor.processAnnotations(ParserMessage[] errors) depends on it
    }

    public String getSourceName() {
        if (sourceName != null)
            return sourceName;
        return environment.getSourceName();
    }

    public Object getParserType() {
        return parserType;
    }

    public ErrConsumer getErrConsumer() {
        if (JavaParser.DEEP_PARSER.equals(getParserType())) {
            return this;
        }
        // null IS intentional for the shallow parser. Before you change it
        // be aware of JavaEditor.processAnnotations(ParserMessage[] errors) depends on it
        return null;
    }

    public void pushError(Object severity, FileObject errorFile, int line, int column, String message, String referenceText) {
        Integer lineInt;

        if (errorFile != null && !errorFile.getNameExt().equals(getSourceName()))
            return;             // error in different file

        ParserMessageImpl newErr =
            new ParserMessageImpl(severity,line,column,message);
        if (errors == null) {
            errors = new ArrayList(50);
        }
        errors.add(newErr);
    }

    /** @return the set of errors encountered while processing this
     * request */
    public Collection getMessages() {
        return errors;
    }

    public ClassPath getSourcePath() {
        assert environment != null;
        
        // XXX
        // if following method is really expected to be null then JavaParser.Env
        // must be extended with something what will allow us to find project
        FileObject fo = environment.getSourceFile();
        assert fo != null;
        
        return ClassPath.getClassPath(fo, ClassPath.SOURCE);
    }

    public ClassPath getLibraryPath() {
        assert environment != null;
        
        // XXX
        // if following method is really expected to be null then JavaParser.Env
        // must be extended with something what will allow us to find project
        FileObject fo = environment.getSourceFile();
        assert fo != null;
        
        return ClassPath.getClassPath(fo, ClassPath.COMPILE);
    }

    public ClassPath getBootClassPath() {
        assert environment != null;
        
        // XXX
        // if following method is really expected to be null then JavaParser.Env
        // must be extended with something what will allow us to find project
        FileObject fo = environment.getSourceFile();
        assert fo != null;
        
        return ClassPath.getClassPath(fo, ClassPath.BOOT);
    }
}

