/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.tools;

import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.util.HashMap;
import java.util.Map;

import java.awt.Image;

import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;
import org.openide.util.Utilities;

/**
 * This class used to cache composite images of the node + badges attached to 
 * it. Now, image cache moved to the openide, so this object only registers
 * different <resourceID, x, y> under some unique identifiers and composes
 * them - it's just an utility class that allows you to register first, then
 * use only the identifier instead of passing all parameters each time.
 *
 * @author  sdedic
 * @version 
 */
public class BadgeCache extends AbstractNode {
    /**
     * Holds a map of all available badges.
     */
    private Map badges;
    
    /**
     * Static register of all caches - keyed by icon bases.
     */
    private static Map cacheRegister;
    
    /**
     * Creates a badge cache for a base icon identified by the parameter.
     */
    public BadgeCache(String underlyingIconBase) {
        super(Children.LEAF);
        setIconBase(underlyingIconBase);
    }

    public static BadgeCache createCache(String baseIconID) {
        if (cacheRegister == null) {
            synchronized (BadgeCache.class) {
                if (cacheRegister == null)
                    cacheRegister = new HashMap(7);
            }
        }
        BadgeCache c = new BadgeCache(baseIconID);
        cacheRegister.put(baseIconID, new SoftReference(c));
        return c;
    }
    
    public static BadgeCache getCache(String baseIconID) {
        if (cacheRegister == null) {
            return createCache(baseIconID);
        }
        synchronized (BadgeCache.class) {
            Reference r = (Reference)cacheRegister.get(baseIconID);
            BadgeCache bc;
            
            if (r == null)
                bc = null;
            else
                bc = (BadgeCache)r.get();
            if (bc == null) {
                return createCache(baseIconID);
            } else
                return bc;
        }
    }
    
    /**
     * Registers a badge for this cache. The iconBase will uniquely identify the badge.
     * The ID string should be unique; it may be the resource name itself, as long as
     * the badge is always placed at the specified location. It is advised, that the same
     * of String as passed to the `id' parameter is used in icon badge lists to speed
     * up composite icon lookups.
     *
     * @param iconBase classpath resource ID for the badge icon
     * @param x 
     * @param y 
     */
    public synchronized void registerBadge(String id, String iconBase, int x, int y) {
        if (badges == null)
            badges = new HashMap(7);
        Object b = badges.put(id, new BadgeNode(iconBase, x, y));
    }
    
    /**
     * Creates a icon with the specified badges. If the requested badge is not found
     * in the cache, it is ignored.
     * @param type type of the result icon, see {@link java.beans.BeanInfo} for symbolic
     * constatnts.
     * @param badgeIds ordered collection (array) of badges that should be applied to the
     * base icon.
     * @deprecated use {@link #createCompositeImage(Image, String[]) instead.
     */
    public Image getIcon(int type, String[] badgeIds) {
        return createCompositeImage(getIcon(type), badgeIds);
    }

    /*
     * Creates a icon with the specified badges. If the requested badge is not found
     * in the cache, it is ignored.
     * @param base base image that should be badged
     * @param badgeIds ordered collection (array) of badges that should be applied to the
     * base icon.
     */
    public Image getIcon(Image base, String[] badgeIds) {
        if (badgeIds == null ||
            badgeIds.length == 0)
            return base;
        return createCompositeImage(base, badgeIds);
    }

    protected final Image createCompositeImage(Image baseImage, String[] badgeIds) {
        if (badgeIds == null || badgeIds.length == 0)
            return baseImage;
        for (int i = 0; i < badgeIds.length; i++) {
            BadgeNode bn = (BadgeNode)badges.get(badgeIds[i]);
            if (bn == null)
                continue;
            Image image = Utilities.loadImage(bn.getIconBase() + ".gif"); // NOI18N
            if (image == null) {
                System.err.println("[Badging] Badge " + bn.getIconBase() + " cannot be loaded"); // NOI18N
            } else {
                baseImage = Utilities.mergeImages(baseImage, image,
                    bn.getX(), bn.getY());
            }
        }
        return baseImage;
    }
    
    /**
     * @deprecated use {@link #createCompositeImage(Image, String[]) instead.
     */
    protected final Image createCompositeImage(int type, String[] badgeIds) {
        Image baseImage = getIcon(type);
        return createCompositeImage(baseImage, badgeIds);
    }
    
    private static class BadgeNode {
        int x;
        int y;
        String iconBase;
        
        public BadgeNode(String iconBase, int x, int y) {
            this.x = x;
            this.y = y;
            this.iconBase = iconBase;
        }
        
        public String getIconBase() {
            return iconBase;
        }
        
        public int getX() {
            return x;
        }
        
        public int getY() {
            return y;
        }
    }
}
