/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes;

import org.netbeans.modules.javacore.JMManager;
import org.openide.nodes.Node;
import org.openide.nodes.Children;
import org.openide.nodes.FilterNode;
import org.openide.nodes.AbstractNode;
import org.openide.util.actions.SystemAction;
import org.openide.util.NbBundle;
import org.openide.util.Lookup;
import org.openide.util.lookup.ProxyLookup;
import org.openide.actions.*;
import org.openide.loaders.DataObject;
import org.openide.cookies.OpenCookie;
import org.openide.cookies.SaveCookie;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.java.ui.nodes.elements.*;
import org.netbeans.modules.java.JavaDataObject;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;

/**
 * Default implementation of {@link SourceNodeFactory}.
 * <p>In case the {@link #tree} propety is true the factory supplies read-only nodes
 * for quick view navigator. These nodes are not backward compatible.
 * XXX clipboard and new actions are disabled yet
 * @author Jan Pokorsky
 */
public final class JavaSourceNodeFactory extends FilterSourceNodeFactory {
    
    private static final JavaSourceNodeFactory DEFAULT = new JavaSourceNodeFactory();
    
    /** Create nodes for tree
     * see org.netbeans.modules.java.JavaElementNodeFactory
     */ 
    private boolean tree = false;
    
    public static SourceNodeFactory getDefault() {
        return DEFAULT;
    }
    
    public Node createFieldNode(Field element) {
        FieldNode n = new FieldNode(element, !JavaMetamodel.getManager().isElementGuarded(element));
        n.setActions(DEFAULT_ACTIONS, SystemAction.get(OpenAction.class));
        if (tree) {
            return n;
        } else {
            Node oldNode = getCompatibleFactory().createFieldNode(element);
            return new CompatibleNode(n, oldNode);
        }
    }

    public Node createConstructorNode(Constructor element) {
        ConstructorNode n = new ConstructorNode(element, !JavaMetamodel.getManager().isElementGuarded(element));
        n.setActions(DEFAULT_ACTIONS, SystemAction.get(OpenAction.class));
        if (tree) {
            return n;
        } else {
            Node oldNode = getCompatibleFactory().createConstructorNode(element);
            return new CompatibleNode(n, oldNode);
        }
    }

    public Node createMethodNode(Method element) {
        MethodNode n = new MethodNode(element, !JavaMetamodel.getManager().isElementGuarded(element));
        n.setActions(DEFAULT_ACTIONS, SystemAction.get(OpenAction.class));
        if (tree) {
            return n;
        } else {
            Node oldNode = getCompatibleFactory().createMethodNode(element);
            return new CompatibleNode(n, oldNode);
        }
    }

    public Node createInitializerNode(Initializer element) {
        InitializerNode n = new InitializerNode(element, !JavaMetamodel.getManager().isElementGuarded(element));
        n.setActions(INITIALIZER_ACTIONS, SystemAction.get(OpenAction.class));
        if (tree) {
            return n;
        } else {
            Node oldNode = getCompatibleFactory().createInitializerNode(element);
            return new CompatibleNode(n, oldNode);
        }
    }

    public Node createEnumNode(JavaEnum element) {
        SourceNodeFactory snf;
        EnumNode n;
        if (tree) {
            snf = SourceNodes.getBrowserFactory();
            EnumChildren children = new EnumChildren(snf, element);
            EnumFilter filter = new EnumFilter();
            n = new EnumNode(element, children, !JavaMetamodel.getManager().isElementGuarded(element));

            n.setElementFormat(new ElementFormat (
                                   NbBundle.getBundle(JavaDataObject.class).getString("CTL_Class_name_format") // NOI18N
                               ));

            // filter out inner classes
            filter.setOrder(new int[] {
                                 EnumFilter.CONSTANTS,
                                 ClassElementFilter.FIELD,
                                 ClassElementFilter.CONSTRUCTOR,
                                 ClassElementFilter.METHOD,
                             });
            children.setFilter(filter);
        } else {
            snf = SourceNodes.getExplorerFactory();
            n = new EnumNode(element, Categories.createEnumChildren(element, snf, true), !JavaMetamodel.getManager().isElementGuarded(element));
        }
        n.setActions(CLASS_ACTIONS, SystemAction.get(OpenAction.class));
        return n;
    }

    public Node createEnumConstantNode(EnumConstant element) {
        EnumConstantNode n = new EnumConstantNode(element, Children.LEAF, !JavaMetamodel.getManager().isElementGuarded(element));
        n.setActions(DEFAULT_ACTIONS, SystemAction.get(OpenAction.class));
        return n;
    }

    public Node createAnnotationTypeNode(AnnotationType element) {
        SourceNodeFactory snf;
        AnnotationTypeNode n;
        
        if (tree) {
            snf = SourceNodes.getBrowserFactory();
            AnnotationTypeChildren children = new AnnotationTypeChildren(snf, element);
            AnnotationTypeFilter filter = new AnnotationTypeFilter();
            n = new AnnotationTypeNode(element, children, !JavaMetamodel.getManager().isElementGuarded(element));
            n.setElementFormat(new ElementFormat(
                    NbBundle.getBundle (JavaDataObject.class).getString("CTL_Class_name_format") // NOI18N
            ));

            // filter out inner classes
            filter.setOrder(new int[] {
                ClassElementFilter.FIELD,
                ClassElementFilter.METHOD,
            });
        } else {
            snf = SourceNodes.getExplorerFactory();
            n = new AnnotationTypeNode(
                    element, Categories.createAnnotationTypeChildren(element, snf, true), 
                    !JavaMetamodel.getManager().isElementGuarded(element));
        }
        n.setActions(CLASS_ACTIONS, SystemAction.get(OpenAction.class));
        return n;
    }

    public Node createAnnotationTypeMethodNode(Attribute element) {
        ElementNode n = new AnnotationTypeMethodNode(element, !JavaMetamodel.getManager().isElementGuarded(element));
        n.setActions(DEFAULT_ACTIONS, SystemAction.get(OpenAction.class));
        return n;
    }

    public Node createClassNode(JavaClass element) {
        if (element == null) {
            // XXX hack for beans module; see JavaElementNodeFactory.FACTORY_GETTER_NODE and org.netbeans.modules.beans.PatternsBrowserFactory.createClassNode
            return super.createClassNode(element);
        }
        
        Node n;
        if (tree) {
            ClassNode cn ;
            SourceNodeFactory snf = SourceNodes.getBrowserFactory();
            ClassChildren children = new ClassChildren(snf, element);
            ClassElementFilter filter = new ClassElementFilter();
            cn = new ClassNode(element, children, !JavaMetamodel.getManager().isElementGuarded(element));

            cn.setElementFormat(new ElementFormat (
                                   NbBundle.getBundle (JavaDataObject.class).getString("CTL_Class_name_format") // NOI18N
                               ));

            // filter out inner classes
            filter.setOrder (new int[] {
                                 ClassElementFilter.FIELD,
                                 ClassElementFilter.CONSTRUCTOR,
                                 ClassElementFilter.METHOD,
                             });
            children.setFilter (filter);
            cn.setActions(CLASS_ACTIONS, SystemAction.get(OpenAction.class));
            n = cn;
        } else {
            ClassNode cn ;
            SourceNodeFactory snf = SourceNodes.getExplorerFactory();
            cn = new ClassNode(element, Categories.createClassChildren(element, snf, true), !JavaMetamodel.getManager().isElementGuarded(element));
            cn.setActions(CLASS_ACTIONS, SystemAction.get(OpenAction.class));
            
            Node oldNode = getCompatibleFactory().createClassNode(element);
            n = new CompatibleNode(cn, oldNode);
        }
        
        return n;
        
    }

    public Node createErrorNode() {
        AbstractNode n = new AbstractNode(Children.LEAF);
        n.setName(NbBundle.getMessage(ElementNode.class, "Error")); // NOI18N
        n.setIconBase(IconStrings.ERROR);
        return n;
    }

    public Node createWaitNode() {
        AbstractNode n = new AbstractNode(Children.LEAF);
        n.setName(NbBundle.getMessage(ElementNode.class, JavaMetamodel.getManager().isScanInProgress()? "WaitScan": "Wait")); // NOI18N
        n.setIconBase(IconStrings.WAIT);
        return n;
    }

    /** If true generate nodes for tree.
    */
    public void setGenerateForTree (boolean tree) {
        this.tree = tree;
    }

    /** Returns true if generate nodes for tree.
    * @return true if generate nodes for tree.
    */
    public boolean getGenerateForTree () {
        return tree;
    }


    /** Array of the actions of the java methods, constructors and fields. */
    private static final SystemAction[] DEFAULT_ACTIONS = new SystemAction[] {
                SystemAction.get(OpenAction.class),
                null,
//                SystemAction.get(CutAction.class),
//                SystemAction.get(CopyAction.class),
                null,
                SystemAction.get(DeleteAction.class),
//                SystemAction.get(RenameAction.class),
                null,
                SystemAction.get(ToolsAction.class),
                SystemAction.get(PropertiesAction.class)
            };

    /** Array of the actions of the java intializers. */
    private static final SystemAction[] INITIALIZER_ACTIONS = new SystemAction[] {
                SystemAction.get(OpenAction.class),
                null,
//                SystemAction.get(CutAction.class),
//                SystemAction.get(CopyAction.class),
                null,
                SystemAction.get(DeleteAction.class),
                null,
                SystemAction.get(ToolsAction.class),
                SystemAction.get(PropertiesAction.class)
            };

    /** Array of the actions of the java classes. */
    private static final SystemAction[] CLASS_ACTIONS = new SystemAction[] {
                SystemAction.get(OpenAction.class),
                null,
//                SystemAction.get(CutAction.class),
//                SystemAction.get(CopyAction.class),
//                SystemAction.get(PasteAction.class),
//                null,
                SystemAction.get(DeleteAction.class),
//                SystemAction.get(RenameAction.class),
                null,
                SystemAction.get(NewAction.class),
                null,
                SystemAction.get(ToolsAction.class),
                SystemAction.get(PropertiesAction.class)
            };
    
    private static SourceNodeFactory COMPATIBLE_FACTORY;
    
    /** gets the node facory of the old source hierarchy (java/srcmodel) */
    private static SourceNodeFactory getCompatibleFactory() {
        if (COMPATIBLE_FACTORY == null) {
            COMPATIBLE_FACTORY = SourceNodes.createSourceNodeFactory(
                    new ExFilterFactory(org.openide.src.nodes.DefaultFactory.READ_ONLY)
            );
        }
        return COMPATIBLE_FACTORY;
    }
    
    /** supplies cookies to ensure proper functionality of code that uses the old source hierarchy api */
    private static final class CompatibleNode extends FilterNode {
        public CompatibleNode(Node current, Node srcModelNode) {
            super(current,
                    null,
                    new ProxyLookup(new Lookup[] {
                        current.getLookup(),
                        new OldElementNodeWithoutDO(srcModelNode).getLookup()
                    })
            );
        }
    }
    
    /** this filter the data object instance out of the node's lookup
     * since ElementNode supplies own DO instance (issue #50644).
     * @see CompatibleNode
     */ 
    private static final class OldElementNodeWithoutDO extends FilterNode {
        public OldElementNodeWithoutDO(Node original) {
            super(original);
        }

        public Node.Cookie getCookie(Class type) {
            Node.Cookie nc = null;
            if (!(DataObject.class.isAssignableFrom(type)
                    || OpenCookie.class.isAssignableFrom(type)
                    || SaveCookie.class.isAssignableFrom(type))) {
                
                nc = super.getCookie(type);
            }
            return nc;
        }
    }
}
