/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes.editors;

import org.openide.ErrorManager;
import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;
import org.openide.util.NbBundle;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.java.ui.nodes.elements.ElementNode;
import org.netbeans.modules.java.ui.nodes.elements.ElementFormat;

import java.beans.FeatureDescriptor;
import java.beans.PropertyEditorSupport;
import java.util.StringTokenizer;


/** Property editor for the org.netbeans.jmi.javamodel.Type
*
* @author Petr Hamernik, Jan Pokorsky
*/
public final class TypeEditor extends PropertyEditorSupport implements ExPropertyEditor {

    /**
     * Value tags for L-values. It does not list "void" type.
     */
    public static final String[] LVALUE_TYPES = {
        "boolean", "int", "char", "byte", "short", "long", "float", "double", // NOI18N
        "Object", "String" // NOI18N
    };

    /**
     * Value tags for R-values. 
     */
    public static final String[] RVALUE_TYPES = {
        "void", "boolean", "int", "char", "byte", "short", "long", "float", "double", // NOI18N
        "Object", "String" // NOI18N
    };

    /**
     * Value tags for L-values. 
     */
    public static final String[] ANNTYPE_VALUE_TYPES = {
        "boolean", "int", "char", "byte", "short", "long", "float", "double", // NOI18N
        "String", "String[]" // NOI18N
    };
    
    public static final String ANN_TYPE_EDITOR = "annTypeEditor"; // NOI18N
    
    private boolean acceptsVoid;
    private boolean isAnnTypeEditor = false;
    
    private JavaModelPackage model;
    
    /** Creates new editor */
    public TypeEditor () {
    }

    public String getAsText () {
        Object val = getValue();
        Type t = (Type) val;
        return (t == null) ? "" : ElementFormat.elementName(t); // NOI18N
    }

    /**
    * Set the property value by parsing a given String.
    * @param text The string to be parsed.
    */
    public void setAsText (String text) throws IllegalArgumentException {
        Type type = model.getType().resolve(text);

        if (!isValidTypeSyntax(text)) {
            throwUserWarning("MSG_InvalidTypeDecl"); // NOI18N
        }
        if (isAnnTypeEditor) {
            if (!isValidTypeOfAnnTypeMethod(type)) {
                throwUserWarning("MSG_InvalidTypeDecl"); // NOI18N
            }
        } else if (!acceptsVoid && type instanceof PrimitiveType &&
                PrimitiveTypeKindEnum.VOID.equals(((PrimitiveType) type).getKind())) {
            throwUserWarning("MSG_VoidTypeNotPermitted"); // NOI18N
        }
        setValue(type);
    }
    
    private static void throwUserWarning(String key) throws IllegalArgumentException {
        IllegalArgumentException ex = new IllegalArgumentException(key);
        ErrorManager.getDefault().annotate(ex,
                ErrorManager.USER, null, getString(key), null, null);
        throw ex;
    }
    
    public static boolean isValidTypeOfAnnTypeMethod(Type type) {
        Type t = type;
        while (t instanceof Array) {
            t = ((Array) t).getType();
        }
        
        if (t instanceof PrimitiveType) {
            return !PrimitiveTypeKindEnum.VOID.equals(((PrimitiveType) t).getKind());
        }
        
        if (t instanceof JavaEnum || t instanceof AnnotationType || t instanceof UnresolvedClass) {
            return true;
        }
        
        if (t instanceof JavaClass) {
            String cname = t.getName();
            return cname.equals(String.class.getName()) || cname.equals(Class.class.getName());
        }
        throw new IllegalStateException("unknown type: " + type); // NOI18N
    }
    
    /**
     * validates generics syntax for now to not break the mdr
     * @param type type to check
     * @return is valid or not
     */ 
    public static boolean isValidTypeSyntax(String type) {
        if (type.length() == 0 || type.charAt(0) == '<') {
            return false;
        }
        
        StringTokenizer tukac = new StringTokenizer(type, "<,>", true); // NOI18N
        int depth = 0;
        while (tukac.hasMoreTokens()) {
            String token = tukac.nextToken();
            if (depth <= 0 && ",".equals(token)) // NOI18N
                return false;
            else if ("<".equals(token)) // NOI18N
                depth++;
            else if (">".equals(token)) { // NOI18N
                if (depth <= 0) return false;
                depth--;
            }
        }
        
        return depth == 0;
    }

    /**
    * @param v new value
    */
    public void setValue(Object v) {
        
        if ( v == null || v instanceof Type )
            super.setValue(v);
        else
            throw new IllegalArgumentException();        
    }

    /**
    * @return A fragment of Java code representing an initializer for the
    * current value.
    */
    public String getJavaInitializationString () {
        return getAsText();
    }

    /**
    * @return The tag values for this property.
    */
    public String[] getTags () {
        if (isAnnTypeEditor) {
            return ANNTYPE_VALUE_TYPES;
        }
        return acceptsVoid ? RVALUE_TYPES : LVALUE_TYPES;
    }

    /**
     * This method is called by the IDE to pass
     * the environment to the property editor.
     */
    public void attachEnv(PropertyEnv env) {
        FeatureDescriptor desc = env.getFeatureDescriptor();
        Object o;
        
        o = desc.getValue("acceptVoidType"); // NOI18N
        if (o instanceof Boolean) {
            acceptsVoid = ((Boolean)o).booleanValue();
        } else {
            acceptsVoid = true;
        }
        
        o = desc.getValue(ANN_TYPE_EDITOR);
        if (o instanceof Boolean) {
            isAnnTypeEditor = ((Boolean) o).booleanValue();
        }
        
        // inplace editor will permit to type own Types  
        desc.setValue("canEditAsText", Boolean.TRUE); // NOI18N
        
        model = ElementNode.getModel(desc);
    }
    
    private static String getString(String key) {
        return NbBundle.getMessage(TypeEditor.class, key);
    }

}
