/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.java.ui.nodes.elements;

import org.openide.nodes.Node;
import org.openide.nodes.Children;
import org.openide.util.RequestProcessor;
import org.openide.util.Mutex;
import org.openide.ErrorManager;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.jmi.javamodel.Element;

import javax.jmi.reflect.JmiException;
import javax.jmi.reflect.InvalidObjectException;
import java.util.*;

/**
 * provides children (jmi elements and their nodes) in async way. It should
 * be used if you cannot create keys and nodes in the AWT event queue. 
 */
final class ChildrenProvider implements Runnable {
        
    /**
     * computes elements (children), creates their nodes, presents them.
     */ 
    interface KeyHandler {
        /**
         * subclasses implement this to supply child keys
         * @return keys
         */ 
        List collectKeys();
        
        /**
         * creates nodes for a given key. The method is invoked inside
         * Children.MUTEX and MDR transaction.
         * @param key the key
         * @return child nodes for this key or null if there should be no 
         *         nodes for this key
         */ 
        Node[] prepareNodes(Object key);
        
        /**
         * subclasses implement this to add nodes to Children.
         * Children.Key.setKeys() will be most often implementation.
         * @param keys child keys
         * @param nodes list of nodes corresponding to keys
         */ 
        void presentKeys(List/*<Element>*/ keys, List/*<Node[]>*/ nodes);
    }
        
    /**
     * processor computing keys and their nodes to not block
     * the awt event queue
     */ 
    static final RequestProcessor RP = new RequestProcessor("Java Children Provider"); // NOI18N
        
    private RequestProcessor.Task currentTask;
        
    private Map/*<Element, Node[]>*/ nodes = initNodeMap();
    
    /** keys */
    private List elements;
        
    private final KeyHandler keyHandler;

    public ChildrenProvider(KeyHandler keyHandler) {
        this.keyHandler = keyHandler;
    }

    public void run() {
        JMManager.getTransactionMutex().addPriorityThread();
        List/*<Element*/ keys;
        Map/*<Element, Node[]>*/ nodesNew;
        try {
            keys = keyHandler.collectKeys();
            nodesNew = createNodeMap(nodes, keys);
        } catch (InvalidObjectException e) {
            // some element is invalid. MDR will notify listeners about that change later
            keys = Collections.EMPTY_LIST;
            nodesNew = Collections.EMPTY_MAP;
        } catch (JmiException e) {
            keys = Collections.EMPTY_LIST;
            nodesNew = Collections.EMPTY_MAP;
            ErrorManager.getDefault().notify(ErrorManager.WARNING, e);
        }

        synchronized(this) {
            elements = keys;
            nodes = nodesNew;
        }
        presentKeys(keys, nodesNew);
    }

    private void presentKeys(final List keys, final Map nodesNew) {
        List nodes = new ArrayList(keys.size());
        for (Iterator it = keys.iterator(); it.hasNext();) {
            Object key = it.next();
            Node[] ns = (Node[]) nodesNew.get(key);
            nodes.add(ns);
            
        }
        keyHandler.presentKeys(keys, nodes);
    }

    private Map/*<Element, Node[]>*/ initNodeMap() {
        Map/*<Element, Node[]>*/ nodesNew = new WeakHashMap/*<Element, Node[]>*/();
        return nodesNew;
    }
        
    /**
     * non-blocking operation that recomputes keys and update their nodes
     */ 
    public synchronized void recomputeChildren() {
        if (currentTask == null) {
            currentTask = RP.post(this);
        } else {
            currentTask.schedule(0);
        }
    }
    
    /**
     * gets node for element. It is a non-blocking operation
     * @param element element as a key
     * @return node or <code>null</null> if the node has not been created yet
     */ 
    public Node getNode(Object element) {
        Node[] ns = null;
        synchronized (this) {
            ns = (Node[]) nodes.get(element);
        }
        return ns == null? null: ns[0];
    }
    
    /**
     * gets keys; blocking opreration
     * @return keys
     */ 
    public Collection getKeys() {
        computeChildren();
        synchronized (this) {
            return elements;
        }
    }
    
    /**
     * waits untill the keys and nodes are ready.
     */ 
    public void waitFinished() {
        computeChildren();
    }
    
    /**
     * stops the running task and sets children as empty collection 
     */ 
    public void clear() {
        RP.post(new CleanTask());
    }
    
    private void clearImpl() {
        synchronized (this) {
            elements = Collections.EMPTY_LIST;
            nodes = Collections.EMPTY_MAP;
        }
        presentKeys(elements, nodes);
    }
        
    private void computeChildren() {
        synchronized (this) {
            if (currentTask == null) {
                recomputeChildren();
            }
        }
        currentTask.waitFinished();
    }
    
    private Map/*<Element, Node[]>*/ createNodeMap(
            final Map/*<Element, Node[]>*/ nodes, final List keys) throws JmiException {
        final Map/*<Element, Node[]>*/ nodesNew = initNodeMap();
        if (keys.isEmpty()) {
            return nodesNew;
        }
        
        JmiException ex = (JmiException) Children.MUTEX.readAccess(new Mutex.Action() {
            public Object run() {
                try {
                    JMManager.getTransactionMutex().addPriorityThread();
                    JavaMetamodel.getDefaultRepository().beginTrans(false);
                    try {
                        for (Iterator it = keys.iterator(); it.hasNext();) {
                            Object key = it.next();
                            if (key instanceof Element && !((Element) key).isValid()) {
                                // some element is invalid. MDR will notify listeners about that change later
                                // throw out the map
                                nodesNew.clear();
                                keys.clear();
                                return null;
                            } else {
                                Node[] ns = (Node[]) nodes.get(key);
                                if (ns == null) {
                                    ns = keyHandler.prepareNodes(key);
                                }
                                nodesNew.put(key, ns);
                            }
                        }
                    } finally {
                        JavaMetamodel.getDefaultRepository().endTrans();
                    }
                } catch (JmiException ex) {
                    return ex;
                }
                return null;
            }

        });
        
        if (ex != null) {
            throw ex;
        }
        return nodesNew;
    }
    
    private final class CleanTask implements Runnable {
        
        public void run() {
            clearImpl();
        }

    }
        
}
