/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes.elements;

import java.awt.BorderLayout;
import java.lang.reflect.Modifier;

import javax.swing.*;
import javax.swing.border.*;
import javax.jmi.reflect.JmiException;

import org.openide.*;
import org.openide.explorer.propertysheet.PropertyPanel;
import org.openide.util.Utilities;
import org.openide.util.NbBundle;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;

/** Customizer for ClassElement
 *
 * @author Petr Hamernik
 */
final class ClassCustomizer extends JPanel {
    /** Predefined types in the type combo */
    private static final String[] COMMON_TYPES = {
        "java.lang.Object", // NOI18N
        "java.awt.Component", // NOI18N
        "javax.swing.JComponent", // NOI18N
        "javax.swing.JPanel" // NOI18N
    };

    /** The edited class */
    private final JavaClass element;
    /**
     * class where the customized class or interface should be added; can be null
     */ 
    private final JavaClass jclass;
    /** java resource where the customized class or interface should be added; can be null */
    private final Resource resource;
    
    private boolean isOK = true;
    
    /** Create new ClassCustomizer component for inner classes
    * @param jclass class where the customized inner class or inner interface should be added
    * @param element The class to be customized
    */
    public ClassCustomizer(JavaClass jclass, JavaClass element) {
        this(null, jclass, element);
    }
    
    /** Create new ClassCustomizer component for top-level classes
    * @param resource java resource where the customized class or interface should be added
    * @param element The class to be customized
    */
    public ClassCustomizer(Resource resource, JavaClass element) {
        this(resource, null, element);
    }
    
    private ClassCustomizer(Resource resource, JavaClass jclass, JavaClass element) {
        this.resource = resource;
        this.jclass = jclass;
        this.element = element;
        initComponents ();
        
        //borders
        interfacesPanel.setBorder(new TitledBorder(NbBundle.getMessage(ClassCustomizer.class, "CTL_Interfaces"))); // NOI18N

        // modifiers
        int mask = Modifier.PUBLIC | Modifier.ABSTRACT;
        if (resource == null) { // inner class
            mask |= Modifier.PROTECTED | Modifier.PRIVATE | Modifier.STATIC;
        }
        if (!element.isInterface()) {
            mask |= Modifier.FINAL;
        }
        accessPanel.add(SourceEditSupport.createAccessModifiersPanel(element, mask), BorderLayout.CENTER);
        modifierPanel.add(SourceEditSupport.createOtherModifiersPanel(element, mask), BorderLayout.CENTER);

        if (element.isInterface()) {
            superClassCombo.setEnabled(false);
            jLabel2.setVisible(false);
            superClassCombo.setVisible(false);
        } else {
            JavaClass superclassId = element.getSuperClass();
            superClassCombo.setSelectedItem(
                    (superclassId == null) ? "" : superclassId.getName()); // NOI18N
            
        }

        // name
        nameTextField.setText(element.getSimpleName());

        // interfaces
        interfacesPanel.add(
                new PropertyPanel(ClassNode.createInterfacesProperty(element, true), PropertyPanel.PREF_CUSTOM_EDITOR),
                BorderLayout.CENTER
        );

        //mnemonics
        jLabel1.setDisplayedMnemonic(NbBundle.getMessage(ClassCustomizer.class, "CTL_Name_Mnemonic").charAt(0)); // NOI18N
        jLabel2.setDisplayedMnemonic(NbBundle.getMessage(ClassCustomizer.class, "CTL_Type_Mnemonic").charAt(0)); // NOI18N
        
        // XXX generics not yet implemented
        jLabel3.setVisible(false);
        typeTextField.setVisible(false);
        
        initAccessibility();
    }

    public void addNotify() {
        super.addNotify();

        // select the name
        int l = nameTextField.getText().length();
        nameTextField.setCaretPosition(0);
        nameTextField.moveCaretPosition(l);
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                nameTextField.requestFocus();
            }
        });
    }
    
    /** This method is called from within the class to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jLabel1 = new javax.swing.JLabel();
        nameTextField = new javax.swing.JTextField();
        jLabel3 = new javax.swing.JLabel();
        typeTextField = new javax.swing.JTextField();
        jLabel2 = new javax.swing.JLabel();
        superClassCombo = new javax.swing.JComboBox(COMMON_TYPES);
        jLabel4 = new javax.swing.JLabel();
        accessPanel = new javax.swing.JPanel();
        jLabel5 = new javax.swing.JLabel();
        modifierPanel = new javax.swing.JPanel();
        interfacesPanel = new javax.swing.JPanel();

        setLayout(new java.awt.GridBagLayout());

        setBorder(new javax.swing.border.EmptyBorder(new java.awt.Insets(6, 6, 6, 6)));
        jLabel1.setLabelFor(nameTextField);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel1, org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "CTL_Name"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(12, 0, 0, 0);
        add(jLabel1, gridBagConstraints);

        nameTextField.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                nameTextFieldFocusLost(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 8, 0, 0);
        add(nameTextField, gridBagConstraints);
        nameTextField.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "ACSD_ClassNameTextField"));

        jLabel3.setLabelFor(typeTextField);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel3, org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "CTL_Generic_Type"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 0, 0, 0);
        add(jLabel3, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 0, 0);
        add(typeTextField, gridBagConstraints);
        typeTextField.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "ACSD_ClassType"));

        jLabel2.setLabelFor(superClassCombo);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel2, org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "CTL_Superclass"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 0, 0, 0);
        add(jLabel2, gridBagConstraints);

        superClassCombo.setEditable(true);
        superClassCombo.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                superClassComboActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 0, 0);
        add(superClassCombo, gridBagConstraints);
        superClassCombo.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "ACSD_SuperClass"));

        jLabel4.setLabelFor(accessPanel);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel4, org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "CTL_AccessRights"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 0, 0, 0);
        add(jLabel4, gridBagConstraints);

        accessPanel.setLayout(new java.awt.BorderLayout());

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 0, 0);
        add(accessPanel, gridBagConstraints);

        jLabel5.setLabelFor(modifierPanel);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel5, org.openide.util.NbBundle.getMessage(ClassCustomizer.class, "CTL_Modifiers"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 0, 0, 0);
        add(jLabel5, gridBagConstraints);

        modifierPanel.setLayout(new java.awt.BorderLayout());

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 0, 0);
        add(modifierPanel, gridBagConstraints);

        interfacesPanel.setLayout(new java.awt.BorderLayout());

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        add(interfacesPanel, gridBagConstraints);

    }
    // </editor-fold>//GEN-END:initComponents

    private void superClassComboActionPerformed (java.awt.event.ActionEvent evt) {//GEN-FIRST:event_superClassComboActionPerformed
        if (!superClassCombo.isEnabled()) return;
        
        JavaClass oldValue = element.getSuperClass();
        String oldValueStr = (oldValue == null) ? "" : oldValue.getName(); // NOI18N
        Object item = superClassCombo.getSelectedItem();

        if (item == null) {
            // user held a CTRL -- bug in Swing ?!?
            superClassCombo.setSelectedItem(oldValueStr);
            return;
        }
        String newValueStr = item.toString().trim();
        boolean ok = false;
        if (!oldValueStr.equals(newValueStr)) {
            try {
                MultipartId mid = getJModel().getMultipartId().createMultipartId(newValueStr, null, null);
                element.setSuperClassName(mid);
                ok = true;
            } catch (JmiException e) {
                ErrorManager.getDefault().notify(e);
            }
        } else { 
            return; // nothing to change
        }
        isOK = ok;
        if (!ok)
            superClassCombo.setSelectedItem(oldValueStr);
    }//GEN-LAST:event_superClassComboActionPerformed

    private void nameTextFieldFocusLost (java.awt.event.FocusEvent evt) {//GEN-FIRST:event_nameTextFieldFocusLost
        if (evt != null && (evt.isTemporary() || !this.isAncestorOf(evt.getOppositeComponent())))
            return;

        String newName = nameTextField.getText().trim();
        String oldName = element.getSimpleName();
        boolean ok = false;
        
        if (!Utilities.isJavaIdentifier(newName)) {
            IllegalArgumentException x = new IllegalArgumentException("Invalid name: " + newName); // NOI18N
            ErrorManager.getDefault().annotate(
                x, ErrorManager.USER, null, 
                NbBundle.getMessage(ClassCustomizer.class, "MSG_Not_Valid_Identifier"), // NOI18N
                null, null);
            ErrorManager.getDefault().notify (x);
        } else if (oldName.equals(newName)) {
            return; // nothing to change
        } else if (this.resource == null && jclass.getInnerClass(newName, true) != null) {
            IllegalArgumentException x = new IllegalArgumentException("Invalid name: " + newName); // NOI18N
            ErrorManager.getDefault().annotate(
                x, ErrorManager.USER, null, 
                NbBundle.getMessage(ClassCustomizer.class, "MSG_Not_Valid_Identifier"), // NOI18N
                null, null);
            ErrorManager.getDefault().notify (x);
        } else if (this.jclass == null && SourceEditSupport.findTopLevelClass(this.resource, newName) != null) {
            IllegalArgumentException x = new IllegalArgumentException("Invalid name: " + newName); // NOI18N
            ErrorManager.getDefault().annotate(
                x, ErrorManager.USER, null, 
                NbBundle.getMessage(ClassCustomizer.class, "MSG_Not_Valid_Identifier"), // NOI18N
                null, null);
            ErrorManager.getDefault().notify (x);
        } else {
            try {
                element.setSimpleName(newName);
                ok = true;
            } catch (JmiException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
        isOK = ok;
        if (!ok) {
            nameTextField.setText(oldName);
        }
    }//GEN-LAST:event_nameTextFieldFocusLost


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel accessPanel;
    private javax.swing.JPanel interfacesPanel;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JPanel modifierPanel;
    private javax.swing.JTextField nameTextField;
    private javax.swing.JComboBox superClassCombo;
    private javax.swing.JTextField typeTextField;
    // End of variables declaration//GEN-END:variables

    
    private void initAccessibility() {
        nameTextField.getAccessibleContext().setAccessibleName(NbBundle.getMessage(ClassCustomizer.class, "ACS_ClassNameTextField")); // NOI18N
        nameTextField.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(ClassCustomizer.class, "ACS_ClassNameTextField")); // NOI18N
        this.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(ClassCustomizer.class, "ACSD_ClassCustomizerDialog")); // NOI18N
    }
    
    public boolean isOK() {
        superClassComboActionPerformed(null);
        if (isOK) {
            nameTextFieldFocusLost(null);
        }
        return isOK;
    }
    
    private JavaModelPackage getJModel() {
        return jclass != null?
                JavaMetamodel.getManager().getJavaExtent(jclass):
                JavaMetamodel.getManager().getJavaExtent(resource);
    }
}
