/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes.elements;

import java.awt.BorderLayout;
import java.lang.reflect.Modifier;

import javax.swing.*;
import javax.swing.border.*;
import javax.jmi.reflect.JmiException;

import org.openide.*;
import org.openide.explorer.propertysheet.PropertyPanel;
import org.openide.util.Utilities;
import org.openide.util.NbBundle;
import org.openide.util.HelpCtx;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaEnum;
import org.netbeans.jmi.javamodel.Resource;

/** Customizer for {@link org.netbeans.jmi.javamodel.JavaEnum}
 *
 * @author Jan Pokorsky
 */
final class EnumCustomizer extends JPanel {
    /** The edited enumeration */
    private final JavaEnum element;
    /**
     * class where the customized class or interface should be added; can be null
     */ 
    private final JavaClass jclass;
    /** java resource where the customized class or interface should be added; can be null */
    private final Resource resource;
    
    private boolean isOK = true;
    
    /** Create new EnumCustomizer component for inner classes
    * @param jclass class where the customized inner class or inner interface should be added
    * @param element The enumeration to be customized
    */
    public EnumCustomizer(JavaClass jclass, JavaEnum element) {
        this(null, jclass, element);
    }
    
    /** Create new EnumCustomizer component for top-level classes
    * @param resource java resource where the customized class or interface should be added
    * @param element The enumeration to be customized
    */
    public EnumCustomizer(Resource resource, JavaEnum element) {
        this(resource, null, element);
    }
    
    private EnumCustomizer(Resource resource, JavaClass jclass, JavaEnum element) {
        this.resource = resource;
        this.jclass = jclass;
        this.element = element;
        initComponents ();
        
        //borders
        interfacesPanel.setBorder (new TitledBorder(NbBundle.getMessage(EnumCustomizer.class, "CTL_Interfaces"))); // NOI18N

        // modifiers
        int mask = Modifier.ABSTRACT;
        if (resource == null) { // inner class
            mask |= Modifier.PUBLIC | Modifier.PROTECTED | Modifier.PRIVATE | Modifier.STATIC;
        }
        if (!element.isInterface()) {
            mask |= Modifier.FINAL;
        }
        accessPanel.add(SourceEditSupport.createAccessModifiersPanel(element, mask), BorderLayout.CENTER);
        modifierPanel.add(SourceEditSupport.createOtherModifiersPanel(element, mask), BorderLayout.CENTER);

        // name
        nameTextField.setText(element.getSimpleName());

        // interfaces
        interfacesPanel.add(
                new PropertyPanel(ClassNode.createInterfacesProperty(element, true), PropertyPanel.PREF_CUSTOM_EDITOR),
                BorderLayout.CENTER
        );

        HelpCtx.setHelpIDString (this, "java.enum.customizer"); // NOI18N

        //mnemonics
        jLabel1.setDisplayedMnemonic(NbBundle.getMessage(EnumCustomizer.class, "CTL_Name_Mnemonic").charAt(0)); // NOI18N
        
        initAccessibility();
    }

    public void addNotify() {
        super.addNotify();

        // select the name
        int l = nameTextField.getText().length();
        nameTextField.setCaretPosition(0);
        nameTextField.moveCaretPosition(l);
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                nameTextField.requestFocus();
            }
        });
    }
    
    /** This method is called from within the class to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jLabel1 = new javax.swing.JLabel();
        nameTextField = new javax.swing.JTextField();
        jLabel2 = new javax.swing.JLabel();
        accessPanel = new javax.swing.JPanel();
        jLabel3 = new javax.swing.JLabel();
        modifierPanel = new javax.swing.JPanel();
        interfacesPanel = new javax.swing.JPanel();

        setLayout(new java.awt.GridBagLayout());

        setBorder(new javax.swing.border.EmptyBorder(new java.awt.Insets(6, 6, 6, 6)));
        jLabel1.setLabelFor(nameTextField);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel1, org.openide.util.NbBundle.getMessage(EnumCustomizer.class, "CTL_Name"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(12, 0, 0, 0);
        add(jLabel1, gridBagConstraints);

        nameTextField.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                nameTextFieldFocusLost(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 8, 0, 0);
        add(nameTextField, gridBagConstraints);

        jLabel2.setLabelFor(accessPanel);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel2, org.openide.util.NbBundle.getMessage(EnumCustomizer.class, "CTL_AccessRights"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 0, 0, 0);
        add(jLabel2, gridBagConstraints);

        accessPanel.setLayout(new java.awt.BorderLayout());

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 0, 0);
        add(accessPanel, gridBagConstraints);

        jLabel3.setLabelFor(modifierPanel);
        org.openide.awt.Mnemonics.setLocalizedText(jLabel3, org.openide.util.NbBundle.getMessage(EnumCustomizer.class, "CTL_Modifiers"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 0, 0, 0);
        add(jLabel3, gridBagConstraints);

        modifierPanel.setLayout(new java.awt.BorderLayout());

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(8, 8, 0, 0);
        add(modifierPanel, gridBagConstraints);

        interfacesPanel.setLayout(new java.awt.BorderLayout());

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 0, 0);
        add(interfacesPanel, gridBagConstraints);

    }
    // </editor-fold>//GEN-END:initComponents

    private void nameTextFieldFocusLost (java.awt.event.FocusEvent evt) {//GEN-FIRST:event_nameTextFieldFocusLost
        if (evt != null && (evt.isTemporary() || !this.isAncestorOf(evt.getOppositeComponent())))
            return;

        String newName = nameTextField.getText().trim();
        String oldName = element.getSimpleName();
        boolean ok = false;
        
        if (!Utilities.isJavaIdentifier(newName)) {
            IllegalArgumentException x = new IllegalArgumentException("Invalid name: " + newName); // NOI18N
            ErrorManager.getDefault().annotate(
                x, ErrorManager.USER, null, 
                NbBundle.getMessage(EnumCustomizer.class, "MSG_Not_Valid_Identifier"), // NOI18N
                null, null);
            ErrorManager.getDefault().notify (x);
        } else if (oldName.equals(newName)) {
            return; // nothing to change
        } else if (this.resource == null && jclass.getInnerClass(newName, true) != null) {
            IllegalArgumentException x = new IllegalArgumentException("Invalid name: " + newName); // NOI18N
            ErrorManager.getDefault().annotate(
                x, ErrorManager.USER, null, 
                NbBundle.getMessage(EnumCustomizer.class, "MSG_Not_Valid_Identifier"), // NOI18N
                null, null);
            ErrorManager.getDefault().notify (x);
        } else if (this.jclass == null && SourceEditSupport.findTopLevelClass(this.resource, newName) != null) {
            IllegalArgumentException x = new IllegalArgumentException("Invalid name: " + newName); // NOI18N
            ErrorManager.getDefault().annotate(
                x, ErrorManager.USER, null, 
                NbBundle.getMessage(EnumCustomizer.class, "MSG_Not_Valid_Identifier"), // NOI18N
                null, null);
            ErrorManager.getDefault().notify (x);
        } else {
            try {
                element.setSimpleName(newName);
                ok = true;
            } catch (JmiException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
        isOK = ok;
        if (!ok) {
            nameTextField.setText(oldName);
        }
    }//GEN-LAST:event_nameTextFieldFocusLost


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JPanel accessPanel;
    private javax.swing.JPanel interfacesPanel;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JPanel modifierPanel;
    private javax.swing.JTextField nameTextField;
    // End of variables declaration//GEN-END:variables

    
    private void initAccessibility() {
        nameTextField.getAccessibleContext().setAccessibleName(NbBundle.getMessage(EnumCustomizer.class, "ACS_EnumNameTextField")); // NOI18N
        nameTextField.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(EnumCustomizer.class, "ACS_EnumNameTextField")); // NOI18N
        this.getAccessibleContext().setAccessibleDescription(NbBundle.getMessage(EnumCustomizer.class, "ACSD_EnumCustomizerDialog")); // NOI18N
    }
    
    public boolean isOK() {
        nameTextFieldFocusLost(null);
        return isOK;
    }
}
