/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.java.ui.nodes.elements;

import org.openide.util.NbBundle;

/**
* These options determine the display name format
* of each kind of element.
* <p>Also included are read-only properties for the "long formats",
* which are in practice used for {@link ElementNode#getHintElementFormat}.
*
* @author Petr Hamernik
*/
public final class SourceOptions {
    /** Kinds of the format. */
    private static final byte T_INITIALIZER = 0;
    private static final byte T_FIELD = 1;
    private static final byte T_CONSTRUCTOR = 2;
    private static final byte T_METHOD = 3;
    private static final byte T_CLASS = 4;
    private static final byte T_INTERFACE = 5;
    private static final byte T_ENUM = 6;
    private static final byte T_CONSTANT = 7;
    private static final byte T_ANNOTATION_TYPE = 8;
    private static final byte T_ANNOTATION_TYPE_METHOD = 9;

    /** Names of all properties. */
    static final String[] PROP_NAMES = {
        "initializerElementFormat", "fieldElementFormat", // NOI18N
        "constructorElementFormat", "methodElementFormat", // NOI18N
        "classElementFormat", "interfaceElementFormat", // NOI18N
        "enumElementFormat", "constantElementFormat", // NOI18N
        "annTypeElementFormat", "annTypeMethodElementFormat", // NOI18N
    };
    
    /** default values for the formats - short form. */
    private static final ElementFormat[] DEFAULT_FORMATS_SHORT = new ElementFormat[PROP_NAMES.length];

    /** default values for the formats - long form. */
    private static final ElementFormat[] DEFAULT_FORMATS_LONG = new ElementFormat[PROP_NAMES.length];
    private static final SourceOptions INSTANCE = new SourceOptions();

    private static void loadDefaultFormats() {
        synchronized (SourceOptions.class) {
            if (DEFAULT_FORMATS_SHORT[0] != null)
                return;
            for (int i = 0; i < PROP_NAMES.length; i++) {
                DEFAULT_FORMATS_SHORT[i] = new ElementFormat(getString("SHORT_" + PROP_NAMES[i])); // NOI18N
                DEFAULT_FORMATS_LONG[i] = new ElementFormat(getString("LONG_" + PROP_NAMES[i])); // NOI18N
            }
        }
    }
    
    /** Property name of the initializer display format. */
    public static final String PROP_INITIALIZER_FORMAT = PROP_NAMES[T_INITIALIZER];

    /** Property name of the field display format. */
    public static final String PROP_FIELD_FORMAT = PROP_NAMES[T_FIELD];

    /** Property name of the constructor display format. */
    public static final String PROP_CONSTRUCTOR_FORMAT = PROP_NAMES[T_CONSTRUCTOR];

    /** Property name of the method display format. */
    public static final String PROP_METHOD_FORMAT = PROP_NAMES[T_METHOD];

    /** Property name of the class display format. */
    public static final String PROP_CLASS_FORMAT = PROP_NAMES[T_CLASS];

    /** Property name of the interface display format. */
    public static final String PROP_INTERFACE_FORMAT = PROP_NAMES[T_INTERFACE];

    /** Property name of the enum display format. */
    public static final String PROP_ENUM_FORMAT = PROP_NAMES[T_ENUM];

    /** Property name of the enum constant display format. */
    public static final String PROP_CONSTANT_FORMAT = PROP_NAMES[T_CONSTANT];

    /** Property name of the annotation type display format. */
    public static final String PROP_ANNOTATION_TYPE_FORMAT = PROP_NAMES[T_ANNOTATION_TYPE];

    /** Property name of the annotation type display format. */
    public static final String PROP_ANNOTATION_TYPE_METHOD_FORMAT = PROP_NAMES[T_ANNOTATION_TYPE_METHOD];

    /** Property name of the 'categories usage' property. */
    public static final String PROP_CATEGORIES_USAGE = "categoriesUsage"; // NOI18N

    /** CategoriesUsage property current value */
    private static boolean categories = true;

    static final long serialVersionUID = 1;
    
    private SourceOptions() {
    }
    
    // ============= public methods ===================
    
    public static SourceOptions getInstance() {
        return INSTANCE;
    }

    /** Get the initializer format.
    * @return the current format
    */
    public ElementFormat getInitializerElementFormat() {
        return getElementFormat(T_INITIALIZER);
    }

    private ElementFormat getElementFormat(int type) {
        loadDefaultFormats();
        return DEFAULT_FORMATS_SHORT[type];
    }

    /** Get the field format.
    * @return the current format
    */
    public ElementFormat getFieldElementFormat() {
        return getElementFormat(T_FIELD);
    }

    /** Get the constructor format.
    * @return the current format
    */
    public ElementFormat getConstructorElementFormat() {
        return getElementFormat(T_CONSTRUCTOR);
    }

    /** Get the method format.
    * @return the current format
    */
    public ElementFormat getMethodElementFormat() {
        return getElementFormat(T_METHOD);
    }

    /** Get the class format.
    * @return the current format
    */
    public ElementFormat getClassElementFormat() {
        return getElementFormat(T_CLASS);
    }

    /** Get the interface format.
    * @return the current format
    */
    public ElementFormat getInterfaceElementFormat() {
        return getElementFormat(T_INTERFACE);
    }

    /** Get the enum format.
    * @return the current format
    */
    public ElementFormat getEnumElementFormat() {
        return getElementFormat(T_ENUM);
    }

    /** Get the enum constant format.
    * @return the current format
    */
    public ElementFormat getConstantElementFormat() {
        return getElementFormat(T_CONSTANT);
    }

    /** Get the annotation type format.
    * @return the current format
    */
    public ElementFormat getAnnTypeElementFormat() {
        return getElementFormat(T_ANNOTATION_TYPE);
    }

    /** Get the annotation type method format.
    * @return the current format
    */
    public ElementFormat getAnnTypeMethodElementFormat() {
        return getElementFormat(T_ANNOTATION_TYPE_METHOD);
    }

    // ============= getters for long form of formats =================

    /** Get the initializer format for longer hints.
    * @return the current format
    */
    public ElementFormat getInitializerElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_INITIALIZER];
    }

    /** Get the field format for longer hints.
    * @return the current format
    */
    public ElementFormat getFieldElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_FIELD];
    }

    /** Get the constructor format for longer hints.
    * @return the current format
    */
    public ElementFormat getConstructorElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_CONSTRUCTOR];
    }

    /** Get the method format for longer hints.
    * @return the current format
    */
    public ElementFormat getMethodElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_METHOD];
    }

    /** Get the class format for longer hints.
    * @return the current format
    */
    public ElementFormat getClassElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_CLASS];
    }

    /** Get the interface format for longer hints.
    * @return the current format
    */
    public ElementFormat getInterfaceElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_INTERFACE];
    }

    /** Get the enum format for longer hints.
    * @return the current format
    */
    public ElementFormat getEnumElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_ENUM];
    }

    /** Get the enum constant format for longer hints.
    * @return the current format
    */
    public ElementFormat getConstantElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_CONSTANT];
    }

    /** Get the annotation type format for longer hints.
    * @return the current format
    */
    public ElementFormat getAnnTypeElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_ANNOTATION_TYPE];
    }

    /** Get the annotation type method format for longer hints.
    * @return the current format
    */
    public ElementFormat getAnnTypeMethodElementLongFormat() {
        loadDefaultFormats();
        return DEFAULT_FORMATS_LONG[T_ANNOTATION_TYPE_METHOD];
    }
    // ============= categories of elements usage ===================

    /** Set the property whether categories under class elements should be used or not.
    * @param cat if <CODE>true</CODE> the elements under class elements are divided into
    *     categories: fields, constructors, methods. Otherwise (<CODE>false</CODE>) all elements
    *     are placed directly under class element.
    */
    public void setCategoriesUsage(boolean cat) {
        categories = cat;
    }

    /** Test whether categiries under class elements are used or not.
    * @return <CODE>true</CODE> if the elements under class elements are divided into
    *     categories: fields, constructors, methods. Otherwise <CODE>false</CODE> (all elements
    *     are placed directly under class element).
    */
    public boolean getCategoriesUsage() {
        return categories;
    }

    // ============= private methods ===================

    private static String getString(String key) {
        return NbBundle.getMessage(SourceOptions.class, key);
    }
}
