/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.explorer.propertysheet.editors;

import java.awt.*;
import java.beans.*;
import java.lang.reflect.Modifier;
import java.util.Locale;
import java.util.StringTokenizer;

import javax.swing.JPanel;

import org.openide.src.ElementProperties;
import org.openide.util.HelpCtx;
import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;
import org.openide.ErrorManager;
import org.openide.util.NbBundle;

/** Property editors for java modifiers.
*
* @author Petr Hamernik
*/
public class ModifierEditor extends JPanel implements ExPropertyEditor {

    /** Instance of custom property editor - visual panel. */
    private ModifierPanel panel;

    /** Serial version UID */
    static final long serialVersionUID = 6324048239020120791L;
    
    /** Current mask */
    private int mask;

    /** Current value */
    private int modifier;
    
    private PropertyEnv env;

    /** Creates new modifiers editor with full mask.
    */
    public ModifierEditor() {
        this(ModifierPanel.EDITABLE_MASK);
    }

    /** Creates new modifiers editor.
    * @param mask The mask of modifier values which should be possible to change.
    */
    public ModifierEditor(int mask) {
        modifier = 0;
        setMask(mask & ModifierPanel.EDITABLE_MASK);

        getAccessibleContext().setAccessibleDescription(getString("ACSD_ModifierPanel"));
    }
    
    public void addNotify() {
        setLayout(new BorderLayout());
        panel = new ModifierPanel(this);
        panel.setMnemonics(env);
        add(panel, BorderLayout.CENTER);
        
        super.addNotify();
    }

    public void removeNotify() {
        super.removeNotify();
        if (panel != null) {
            remove(panel);
            panel = null;
        }
    }
    
    /** Getter for property mask.
     *@return Value of property mask.
     */
    int getMask() {
        return mask;
    }

    /** Set the mask of editable modifiers.
     * @param mask new value of the mask.
     */
    public void setMask(int mask) {
        if (this.mask != mask) {
            int oldMask = this.mask;
            this.mask = mask & ModifierPanel.EDITABLE_MASK;
            firePropertyChange (ModifierPanel.PROP_MASK, new Integer (oldMask), new Integer (mask));
            setModifier(modifier & mask);
        }
    }
    
    /** Getter for property modifier.
     *@return Value of property modifier.
     */
    int getModifier() {
        return modifier;
    }

    /** Setter for property modifier.
     *@param modifier New value of property modifier.
     */
    void setModifier(int modifier) {
        if (this.modifier != modifier) {
            int oldModifier = this.modifier;
            this.modifier = modifier;
            // for our panel
            firePropertyChange (ModifierPanel.PROP_MODIFIER, new Integer (oldModifier), new Integer (modifier));
            // for the outside world
            firePropertyChange(ElementProperties.PROP_MODIFIERS, new Integer (oldModifier), new Integer (modifier));
        }
    }

    /** Set new value */
    public void setValue(Object object) throws IllegalArgumentException {
        if (object == null) {
            setModifier(0);
            return;
        }
        if (object instanceof Integer) {
            setModifier(((Integer) object).intValue());
        }
        else {
            throw new IllegalArgumentException();
        }
    }

    /** @return the java source code representation
    * of the current value.
    */
    public String getJavaInitializationString() {
        return new Integer(getModifier()).toString();
    }

    /** Get the value */
    public Object getValue() {
        return new Integer(getModifier());
    }

    /** @return <CODE>false</CODE> */
    public boolean isPaintable() {
        return false;
    }

    /** Does nothing. */
    public void paintValue(Graphics g, Rectangle rectangle) {
    }

    /** @return textual representition of current value of the modifiers. */
    public String getAsText() {
        return Modifier.toString(getModifier());
    }

    /** Parse the text and sets the modifier editor value */
    public void setAsText(String string) throws IllegalArgumentException {
        int newValue = 0;
        int oldValue = modifier;

        StringTokenizer tukac = new StringTokenizer(string, ", ", false); // NOI18N
        while (tukac.hasMoreTokens()) {
            String token = tukac.nextToken();
            boolean known = false;
            for (int i = 0; i < ModifierPanel.MODIFIER_COUNT; i++) {
                if ((ModifierPanel.MODIFIER_VALUES[i] & mask) != 0) {
                    if (token.equals(ModifierPanel.MODIFIER_NAMES[i])) {
                        if (((ModifierPanel.MODIFIER_VALUES[i] == Modifier.FINAL) && ((newValue & Modifier.ABSTRACT) != 0)) ||
                                ((ModifierPanel.MODIFIER_VALUES[i] == Modifier.ABSTRACT) && ((newValue & Modifier.FINAL) != 0)))
                            break;
                        newValue |= ModifierPanel.MODIFIER_VALUES[i];
                        known = true;
                        break;
                    }
                }
            }
            if ((newValue & ModifierPanel.ACCESS_MASK) == 0) {
                for (int i = 1; i <= 3; i++) {
                    if ((ModifierPanel.ACCESS_VALUES[i] & mask) != 0) {
                        if (token.equals(ModifierPanel.ACCESS_NAMES[i])) {
                            newValue |= ModifierPanel.ACCESS_VALUES[i];
                            known = true;
                            break;
                        }
                    }
                }
            }
            if (!known) {
                IllegalArgumentException x = new IllegalArgumentException(
                    "Invalid modifier: " + token); // NOI18N
                String message = java.text.MessageFormat.format(
                    getString("MSG_IllegalModifierString"), // NOI18N
                    new Object[] { token });
                ErrorManager.getDefault().annotate(x,
			ErrorManager.USER, null, message, null, null);
                throw x;
            }
        }
        if (oldValue != newValue) {
            modifier = newValue;
            firePropertyChange(ModifierPanel.PROP_MODIFIER,
                    new Integer(oldValue), new Integer(modifier));
        }
    }

    /** @return <CODE>null</CODE> */
    public String[] getTags() {
        return null;
    }

    /** @return <CODE>this</CODE> */
    public Component getCustomEditor() {
        return this;
    }

    /** @return <CODE>true</CODE> */
    public boolean supportsCustomEditor() {
        return true;
    }

    /** Get the customized property value.
     * @return the property value
     * @exception InvalidStateException when the custom property editor does not contain a valid property value
     *           (and thus it should not be set)
     */
    public Object getPropertyValue() throws IllegalStateException {
        return getValue();
    }

    /**
     * This method is called by the IDE to pass
     * the environment to the property editor.
     */
    public void attachEnv(PropertyEnv env) {
        this.env = env;
    }
     
    private static String getString(String key) {
        return NbBundle.getBundle("org.openide.explorer.propertysheet.editors.Bundle2", Locale.getDefault(), ModifierEditor.class.getClassLoader()).getString(key);
    }
}
