/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.explorer.propertysheet.editors;

import java.awt.event.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import java.beans.FeatureDescriptor;
import java.lang.reflect.Modifier;
import java.util.Locale;

import javax.swing.*;

import org.openide.util.NbBundle;
import org.openide.util.WeakListeners;
import org.openide.explorer.propertysheet.PropertyEnv;

/** JPanel extension containing components which allows visual
 * editing Modifier object.
 * This class has two main properties: mask (possible values mask) 
 * and modifier (current value).
 *
 * @author Petr Hamernik
 */
class ModifierPanel extends javax.swing.JPanel {

    // ------------------------- Statics -------------------------------

    /** Name of 'mask' property */
    public static final String PROP_MASK = "mask"; // NOI18N

    /** Name of 'modifier' property (current value) */
    public static final String PROP_MODIFIER = "modifier"; // NOI18N
    
    private static final int CHECK_ABSTRACT         = 0;
    private static final int CHECK_FINAL            = 1;
    private static final int CHECK_STATIC           = 2;
    private static final int CHECK_SYNCHRONIZED     = 3;
    private static final int CHECK_TRANSIENT        = 4;
    private static final int CHECK_VOLATILE         = 5;
    private static final int CHECK_NATIVE           = 6;

    /** Names of modifiers */
    static final String MODIFIER_NAMES[] = {
        "abstract", "final", "static", "synchronized", "transient", "volatile", "native" // NOI18N
    };

    private static final String[] MODIFIER_DESCRIPTION_KEYS = {
        "ACSD_ModifierPanel_Modifier_Abstract",         // NOI18N
        "ACSD_ModifierPanel_Modifier_Final",            // NOI18N
        "ACSD_ModifierPanel_Modifier_Static",           // NOI18N
        "ACSD_ModifierPanel_Modifier_Synchronized",     // NOI18N
        "ACSD_ModifierPanel_Modifier_Transient",        // NOI18N
        "ACSD_ModifierPanel_Modifier_Volatile",         // NOI18N
        "ACSD_ModifierPanel_Modifier_Native"            // NOI18N
    };

    private static final String[] MODIFIER_MNEMONICS_KEYS = {
        "ModifierPanel_Modifier_Abstract_Mnemonic",     // NOI18N
        "ModifierPanel_Modifier_Final_Mnemonic",        // NOI18N
        "ModifierPanel_Modifier_Static_Mnemonic",       // NOI18N
        "ModifierPanel_Modifier_Synchronized_Mnemonic", // NOI18N
        "ModifierPanel_Modifier_Transient_Mnemonic",    // NOI18N
        "ModifierPanel_Modifier_Volatile_Mnemonic",     // NOI18N
        "ModifierPanel_Modifier_Native_Mnemonic"        // NOI18N
    };

    /** Values of modifiers */
    static final int MODIFIER_VALUES[] = {
        Modifier.ABSTRACT, Modifier.FINAL, Modifier.STATIC, Modifier.SYNCHRONIZED,
        Modifier.TRANSIENT, Modifier.VOLATILE, Modifier.NATIVE
    };

    /** Count of the modifiers */
    static final int MODIFIER_COUNT = MODIFIER_VALUES.length;

    /** Names of accessibility */
    static final String ACCESS_NAMES[] = {
        "<default>", "private", "protected", "public" // NOI18N
    };

    /** Values of accessibility */
    static final int ACCESS_VALUES[] = {
        0, Modifier.PRIVATE, Modifier.PROTECTED, Modifier.PUBLIC
    };

    /** Mask of access modifiers */
    static final int ACCESS_MASK = Modifier.PRIVATE | Modifier.PROTECTED | Modifier.PUBLIC;

    /** Mask of all possible modifiers. */
    static final int EDITABLE_MASK = ACCESS_MASK | Modifier.ABSTRACT |
                                     Modifier.FINAL | Modifier.STATIC | Modifier.SYNCHRONIZED |
                                     Modifier.TRANSIENT | Modifier.VOLATILE | Modifier.NATIVE;

    // ------------------ Instance Fields --------------------------

    /** Reference back to the editor that created this panel. */
    private ModifierEditor myEditor;
    
    /** Current access values shown in the combo box */
    private int currentAccessValues[];

    /** Current access names shown in the combo box */
    private String currentAccessNames[];

    /** JCheckBox array */
    private JCheckBox[] checks;

    /** listener for visual changes */
    private ActionListener listener;

    /** Ignored flag - used during firing change events */
    private boolean ignored = false;


    static final long serialVersionUID =6884758007403225916L;
    /** Creates new form ModifiersPanel */
    public ModifierPanel(ModifierEditor ed) {
        myEditor = ed;
        currentAccessValues = ACCESS_VALUES;
        currentAccessNames = ACCESS_NAMES;
        
        myEditor.addPropertyChangeListener(
            WeakListeners.propertyChange(
                new PropertyChangeListener() {
                    public void propertyChange(PropertyChangeEvent evt) {
                        if (PROP_MODIFIER.equals(evt.getPropertyName()) || 
                            PROP_MASK.equals(evt.getPropertyName())) {
                            updateAccess();
                            ignored = true;
                            updateComponents();
                            ignored = false;
                        }
                    }
                },
                myEditor
            )
        );

        listener = new ActionListener() {
                       public void actionPerformed(ActionEvent evt) {
                           // remove abstract, if private access is being selected.
			   int selIndex = accessCombo.getSelectedIndex();
                           // disallow abstract, if private access is selected.
			   if (evt.getSource() == accessCombo && selIndex < 0) {
			       // revert the combo to the previous value.
			       return;
			   }			   
                           if (checks[CHECK_ABSTRACT].isSelected() &&
                            ((myEditor.getModifier() & MODIFIER_VALUES[CHECK_ABSTRACT]) == 0) &&
                            ((myEditor.getModifier() & Modifier.PRIVATE) > 0)) {
                               checks[CHECK_ABSTRACT].setSelected(false);
                           }
                           if (selIndex >= 0 &&
			       (currentAccessValues[accessCombo.getSelectedIndex()] & Modifier.PRIVATE) > 0 &&
	            		(myEditor.getModifier() & Modifier.PRIVATE) == 0) 
                               checks[CHECK_ABSTRACT].setSelected(false);
			       
                           excludeChecks(CHECK_ABSTRACT, CHECK_FINAL);
                           excludeChecks(CHECK_ABSTRACT, CHECK_NATIVE);
                           excludeChecks(CHECK_ABSTRACT, CHECK_STATIC);
                           excludeChecks(CHECK_ABSTRACT, CHECK_SYNCHRONIZED);
                           excludeChecks(CHECK_VOLATILE, CHECK_FINAL);
                           
                           if (!ignored)
                               updateValue();
                       }
                   };

        ignored = true;
        initComponents();

        modifPanel.setBorder (new javax.swing.border.CompoundBorder(
                                  new javax.swing.border.TitledBorder(getString("LAB_Modifiers")),
                                  new javax.swing.border.EmptyBorder(new java.awt.Insets(3, 3, 3, 3))
                              ));

        updateAccess();
        updateModifiers();
        updateComponents();
        ignored = false;

        jLabel1.setDisplayedMnemonic(getString("LAB_AccessRights_Mnemonic").charAt(0));

        jLabel1.getAccessibleContext().setAccessibleDescription(getString("ACSD_AccessRights"));
    }

    /** Makes sure that the specified two checkboxes are mutually exclusive by
     * unselecting one if the other one becomes selected.
     */
    private void excludeChecks(int check1, int check2) {
        if (checks[check1].isSelected() && ((myEditor.getModifier() & MODIFIER_VALUES[check1]) == 0))
            checks[check2].setSelected(false);
        else if (checks[check2].isSelected() && ((myEditor.getModifier() & MODIFIER_VALUES[check2]) == 0))
            checks[check1].setSelected(false);
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        modifPanel = new javax.swing.JPanel();
        jPanel2 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        accessCombo = new javax.swing.JComboBox();

        setLayout(new java.awt.BorderLayout());

        setBorder(new javax.swing.border.EmptyBorder(new java.awt.Insets(6, 7, 6, 7)));
        modifPanel.setLayout(new java.awt.GridLayout(4, 2, 4, 4));

        checks = new JCheckBox[MODIFIER_COUNT];
        for (int i = 0; i < MODIFIER_COUNT; i++) {
            checks[i] = new JCheckBox(MODIFIER_NAMES[i]);
            checks[i].getAccessibleContext().setAccessibleDescription(getModifierDescription(i));
            modifPanel.add(checks[i]);
            checks[i].setEnabled((myEditor.getMask() & MODIFIER_VALUES[i]) != 0);
            checks[i].addActionListener(listener);
        }

        add(modifPanel, java.awt.BorderLayout.CENTER);

        jPanel2.setLayout(new java.awt.BorderLayout(8, 8));

        jPanel2.setBorder(new javax.swing.border.EmptyBorder(new java.awt.Insets(5, 5, 5, 5)));
        jLabel1.setText(getString("LAB_AccessRights"));
        jLabel1.setLabelFor(accessCombo);
        jPanel2.add(jLabel1, java.awt.BorderLayout.WEST);

        accessCombo.addActionListener(listener);
        jPanel2.add(accessCombo, java.awt.BorderLayout.CENTER);

        add(jPanel2, java.awt.BorderLayout.NORTH);

    }//GEN-END:initComponents


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JComboBox accessCombo;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel modifPanel;
    // End of variables declaration//GEN-END:variables

    /** Update access ComboBox values depending on new 'mask' property
     */
    private void updateAccess() {
        int selValue = myEditor.getModifier() & ACCESS_MASK;
        int selIndex = -1;

        int counter = 1;
        for (int i = 1; i < ACCESS_VALUES.length; i++) {
            if ((ACCESS_VALUES[i] & myEditor.getMask()) != 0)
                counter++;
        }
        currentAccessValues = new int[counter];
        currentAccessNames = new String[counter];

        currentAccessValues[0] = ACCESS_VALUES[0];
        currentAccessNames[0] = ACCESS_NAMES[0];
        counter = 1;

        for (int i = 1; i < ACCESS_VALUES.length; i++) {
            if ((ACCESS_VALUES[i] & myEditor.getMask()) != 0) {
                currentAccessValues[counter] = ACCESS_VALUES[i];
                currentAccessNames[counter] = ACCESS_NAMES[i];
                if (ACCESS_VALUES[i] == selValue) {
                    selIndex = counter;
                }
                counter++;
            }
        }
        if (selIndex == -1 && selValue == 0)
            selIndex = 0;

        ignored = true;
        accessCombo.setModel(new DefaultComboBoxModel(currentAccessNames));
        accessCombo.setSelectedIndex(selIndex);
        ignored = false;
    }

    /** Update enable status of all modifiers check boxes
     */
    private void updateModifiers() {
        for (int i = 0; i < MODIFIER_COUNT; i++) {
            checks[i].setEnabled((myEditor.getMask() & MODIFIER_VALUES[i]) != 0);
        }
    }

    /** Update the components inside the ModifierPanel depending on new value
     * of 'modifier' property.
     */
    private void updateComponents() {
	updateAccessCombo();
	updateModifiers();
        for (int i = 0; i < MODIFIER_COUNT; i++) {
            checks[i].setSelected((myEditor.getModifier() & MODIFIER_VALUES[i]) != 0);
        }
    }
    
    private void updateAccessCombo() {
        int selIndex = -1;
        if (myEditor.getModifier() == 0) {
            selIndex = 0;
        } else {
            for (int i = 1; i < currentAccessValues.length; i++) {
                if ((currentAccessValues[i] & myEditor.getModifier()) != 0) {
		            if (selIndex != -1) {
		                selIndex = -1;
		                break;
		            }
                    selIndex = i;
                }
            }
        }
	    if (accessCombo.getSelectedIndex() != selIndex) {
            accessCombo.setSelectedIndex(selIndex);
	    }
    }

    /** Updates the value depending on the status of the components. */
    private void updateValue() {
        int newValue = 0;
	int comboIndex = accessCombo.getSelectedIndex();
	
	if (comboIndex == -1) {
	    newValue = myEditor.getModifier() & ACCESS_MASK;
	} else {
            newValue |= currentAccessValues[comboIndex];
	}

        for (int i = 0; i < MODIFIER_COUNT; i++) {
            if (checks[i].isSelected() & checks[i].isEnabled())
                newValue |= MODIFIER_VALUES[i];
        }
        if (myEditor.getModifier() != newValue) {
            myEditor.setModifier(newValue);
        }
    }
    
    /* package private */void setMnemonics(PropertyEnv env) {
        FeatureDescriptor desc = env.getFeatureDescriptor();

        for (int i = 0; i < MODIFIER_COUNT; i++) {
            checks[i].setMnemonic(getModifierMnemonics(i));
        
            Object o = desc.getValue(MODIFIER_MNEMONICS_KEYS[i]);
            if (o instanceof String) {
                checks[i].setMnemonic(((String)o).charAt(0));
            } else {
                checks[i].setMnemonic(getModifierMnemonics(i));
            }
        }
    }
    
    private static String getString(String key) {
        return NbBundle.getBundle("org.openide.explorer.propertysheet.editors.Bundle2", Locale.getDefault(), ModifierPanel.class.getClassLoader()).getString(key);
    }
    
    static String getModifierDescription(int i) {
        return getString(MODIFIER_DESCRIPTION_KEYS[i]);
    }
    
    static char getModifierMnemonics(int i) {
        return getString(MODIFIER_MNEMONICS_KEYS[i]).charAt(0);
    }
}
