/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.explorer.propertysheet.editors;

import java.awt.*;
import java.beans.*;
import java.util.Locale;

import org.openide.ErrorManager;
import org.openide.src.Type;
import org.openide.explorer.propertysheet.editors.EnhancedPropertyEditor;
import org.openide.explorer.propertysheet.ExPropertyEditor;
import org.openide.explorer.propertysheet.PropertyEnv;
import org.openide.util.NbBundle;


/** Property editor for the org.netbeans.src.Type
*
* @author Petr Hamernik
*/
public class TypeEditor extends PropertyEditorSupport 
    implements EnhancedPropertyEditor, ExPropertyEditor {

    /**
     * Value tags for L-values. It does not list "void" type.
     */
    private static final String[] LVALUE_TYPES = {
        "boolean", "int", "char", "byte", "short", "long", "float", "double", // NOI18N
        "Object", "String" // NOI18N
    };

    /**
     * Value tags for R-values. 
     */
    private static final String[] RVALUE_TYPES = {
        "void", "boolean", "int", "char", "byte", "short", "long", "float", "double", // NOI18N
        "Object", "String" // NOI18N
    };
    
    /** generated Serialized Version UID */
    static final long serialVersionUID = 1423443523462351952L;

    private boolean acceptsVoid;
    
    /** Creates new editor */
    public TypeEditor () {
    }

    /**
    * @return The property value as a human editable string.
    * <p>   Returns null if the value can't be expressed as an editable string.
    * <p>   If a non-null value is returned, then the PropertyEditor should
    *       be prepared to parse that string back in setAsText().
    */
    public String getAsText () {
        Object val = getValue();
        return (val == null) ? "" : val.toString(); // NOI18N
    }

    /**
    * Set the property value by parsing a given String.
    * @param string The string to be parsed.
    */
    public void setAsText (String string) throws IllegalArgumentException {
        Type t;
        
        try {
            t = Type.parse(string);
        } catch (IllegalArgumentException ex) {
            ErrorManager.getDefault().annotate(ex,
		    ErrorManager.USER, null, 
                    getString("MSG_InvalidTypeDecl"), // NOI18N
                    null, null);
            throw ex;
        }
        if (!acceptsVoid && t == Type.VOID) {
            IllegalArgumentException ex = new IllegalArgumentException("Void not allowed"); // NOI18N
            ErrorManager.getDefault().annotate(ex,
		    ErrorManager.USER, null, 
                    getString("MSG_VoidTypeNotPermitted"), // NOI18N
                    null, null);
            throw ex;
        }
        setValue(t);
    }

    /**
    * @param v new value
    */
    public void setValue(Object v) {
        
        if ( v == null || v instanceof Type )
            super.setValue(v);
        else
            throw new IllegalArgumentException();        
    }

    /**
    * @return A fragment of Java code representing an initializer for the
    * current value.
    */
    public String getJavaInitializationString () {
        return getAsText();
    }

    /**
    * @return The tag values for this property.
    */
    public String[] getTags () {
        return acceptsVoid ? RVALUE_TYPES : LVALUE_TYPES;
    }

    /**
    * @return Returns custom property editor to be showen inside the property
    *         sheet.
    */
    public Component getInPlaceCustomEditor () {
        return null;
    }

    /**
    * @return true if this PropertyEditor provides a enhanced in-place custom
    *              property editor, false otherwise
    */
    public boolean hasInPlaceCustomEditor () {
        return false;
    }

    /**
    * @return true if this property editor provides tagged values and
    * a custom strings in the choice should be accepted too, false otherwise
    */
    public boolean supportsEditingTaggedValues () {
        return true;
    }
    
    /**
     * This method is called by the IDE to pass
     * the environment to the property editor.
     */
    public void attachEnv(PropertyEnv env) {
        FeatureDescriptor desc = env.getFeatureDescriptor();
        Object o;
        
        o = desc.getValue("acceptVoidType"); // NOI18N
        if (o instanceof Boolean) {
            acceptsVoid = ((Boolean)o).booleanValue();
        } else {
            acceptsVoid = true;
        }
    }
    
    private static String getString(String key) {
        return NbBundle.getBundle("org.openide.explorer.propertysheet.editors.Bundle2", Locale.getDefault(), TypeEditor.class.getClassLoader()).getString(key);
    }

}
