/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.awt.Component;
import java.beans.*;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.lang.reflect.InvocationTargetException;

import org.openide.src.*;
import org.openide.nodes.*;

/** Node representing a constructor.
* @see ConstructorElement
* @author Petr Hamernik
*/
public class ConstructorElementNode extends MemberElementNode {
    /** Create a new constructor node.
    * @param element constructor element to represent
    * @param writeable <code>true</code> to be writable
    */
    public ConstructorElementNode(ConstructorElement element, boolean writeable) {
        super(element, Children.LEAF, writeable);
        setElementFormat0(sourceOptions.getConstructorElementFormat());
    }

    public org.openide.util.HelpCtx getHelpCtx () {
        return new org.openide.util.HelpCtx ("org.openide.src.ConstructorNode"); // NOI18N
    }

    /* Resolve the current icon base.
    * @return icon base string.
    */
    protected String resolveIconBase() {
        int modif = ((ConstructorElement)element).getModifiers();
        if (Modifier.isPrivate(modif))
            return CONSTRUCTOR_PRIVATE;
        else if (Modifier.isProtected(modif))
            return CONSTRUCTOR_PROTECTED;
        else if (Modifier.isPublic(modif))
            return CONSTRUCTOR_PUBLIC;
        else
            return CONSTRUCTOR_PACKAGE;
    }

    /* This method resolve the appropriate hint format for the type
    * of the element. It defines the short description.
    */
    protected ElementFormat getHintElementFormat() {
        return sourceOptions.getConstructorElementLongFormat();
    }

    /* Creates property set for this node */
    protected Sheet createSheet () {
        Sheet sheet = Sheet.createDefault();
        Sheet.Set ps = sheet.get(Sheet.PROPERTIES);
        ps.put(createModifiersProperty(writeable));
        ps.put(createNameProperty(false));
        ps.put(createParametersProperty(writeable));
        ps.put(createExceptionsProperty(writeable));
        return sheet;
    }

    /* Removes the element from the class and calls superclass.
    *
    * @exception IOException if SourceException is thrown
    *            from the underlayed Element.
    */
    public void destroy() throws IOException {
        if (!(element instanceof MethodElement)) {
            SourceEditSupport.invokeAtomicAsUser(element, new SourceEditSupport.ExceptionalRunnable() {
                                                     public void run() throws SourceException {
                                                         ConstructorElement el = (ConstructorElement) element;
                                                         el.getDeclaringClass().removeConstructor(el);
                                                     }
                                                 });
        }
        super.destroy();
    }

    /** Indicate that this node cannot be renamed.
    * An constructor must have the same name like class
    * @return <code>false</code>
    */
    public boolean canRename() {
        return false;
    }

    public Component getCustomizer() {
        return new MethodCustomizer((ConstructorElement)element);
    }

    public boolean hasCustomizer() {
        return isWriteable();
    }

    /** Create a node property for constructor parameters.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    protected Node.Property createParametersProperty(boolean canW) {
        Node.Property p = new ElementProp(PROP_PARAMETERS, MethodParameter[].class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((ConstructorElement)element).getParameters();
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);
                       if (!(val instanceof MethodParameter[]))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((ConstructorElement)element).setParameters((MethodParameter[])val);
                                     }
                                 });
                   }
               };
        p.setValue("changeImmediate" /* PropertyEnv.PROP_CHANGE_IMMEDIATE */,Boolean.FALSE); // NOI18N
        return p;
    }

    /** Create a node property for constructor exceptions.
    * @param canW <code>false</code> to force property to be read-only
    * @return the property
    */
    protected Node.Property createExceptionsProperty(boolean canW) {
        Node.Property p = new ElementProp(PROP_EXCEPTIONS, Identifier[].class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((ConstructorElement)element).getExceptions();
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);
                       if (!(val instanceof Identifier[]))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((ConstructorElement)element).setExceptions((Identifier[])val);
                                     }
                                 });
                   }
               };
        p.setValue("changeImmediate" /* PropertyEnv.PROP_CHANGE_IMMEDIATE */,Boolean.FALSE); // NOI18N
        return p;
    }
}
