/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.openide.src.nodes;

import java.io.IOException;
import java.beans.*;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;

import org.openide.ErrorManager;
import org.openide.explorer.propertysheet.editors.ModifierEditor;
import org.openide.src.*;
import org.openide.nodes.*;
import org.openide.util.Utilities;

/** Node representing some type of member element.
*
* @author Petr Hamernik
*/
public abstract class MemberElementNode extends ElementNode {
    /** Create a new node.
    *
    * @param element member element to represent
    * @param children list of children
    * @param writeable <code>true</code> to be writable
    */
    public MemberElementNode(MemberElement element, Children children, boolean writeable) {
        super(element, children, writeable);
        superSetName(element.getName().getName());
    }

    /** Set the node's (system) name.
    * Attempts to change the element's name as well using {@link MemberElement#setName}.
    * Read-only elements cannot have their name set.
    * The display name will also be updated according to the proper format,
    * if necessary (typically it will be).
    *
    * @param str the new element and node name
    */
    public void setName(final String str) {
        try {
            if (testJavaId(str)) {
                SourceEditSupport.invokeAtomicAsUser(element, new SourceEditSupport.ExceptionalRunnable() {
                                                         public void run() throws SourceException {
                                                             ((MemberElement)element).setName(Identifier.create(str));
                                                             superSetName(str);
                                                         }
                                                     });
            }
        }
        catch (IOException e) {
            MessageFormat fmt = new MessageFormat(bundle.getString("MSG_ElementCantRename"));
            String[] params = new String[] { ((MemberElement)element).getName().toString(), e.getMessage() };
            if (params[1] == null)
                params[1] = ""; // NOI18N
	    
	    IllegalArgumentException ex = new IllegalArgumentException("Invalid name"); // NOI18N
	    ErrorManager.getDefault().annotate(ex, ErrorManager.USER,
		    null, fmt.format(params), e, null);
	    throw ex;
        }
    }

    /** Tests if the given string is java identifier and if not, notifies
    * the user.
    * @return true if it is ok.
    */
    boolean testJavaId(String str) throws IllegalArgumentException {
        boolean ok = Utilities.isJavaIdentifier(str);
        if (!ok) {
	    IllegalArgumentException ex = new IllegalArgumentException("Invalid identifier"); // NOI18N
	    ErrorManager.getDefault().annotate(ex, ErrorManager.USER,
		null, bundle.getString("MSG_Not_Valid_Identifier"), null, null);
            throw ex;
        }
        return ok;
    }

    /** Create a node property for the modifiers of the element.
    * This property will typically display with a custom editor
    * allowing individual modifiers to be examined.
    * @param canW if <code>false</code>, the property will be read-only irrespective of
    *       the underlying element's ability to change the modifiers
    * @return the property
    */
    protected Node.Property createModifiersProperty(boolean canW) {
        Node.Property p = new ElementProp(PROP_MODIFIERS, Integer.class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return new Integer(((MemberElement) element).getModifiers());
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);

                       if (!(val instanceof Integer))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         ((MemberElement)element).setModifiers(((Integer)val).intValue());
                                     }
                                 });
                   }

                   /** Define property editor for this property. */
                   public PropertyEditor getPropertyEditor () {
                       return new ModifierEditor(((MemberElement)element).getModifiersMask());
                   }
               };
        p.setValue("changeImmediate" /* PropertyEnv.PROP_CHANGE_IMMEDIATE */,Boolean.FALSE); // NOI18N
        return p;
    }

    /** Create a node property representing the element's name.
    * @param canW if <code>false</code>, property will be read-only
    * @return the property.
    */
    protected Node.Property createNameProperty(boolean canW) {
        return new ElementProp(ElementProperties.PROP_NAME, String.class, canW) {
                   /** Gets the value */
                   public Object getValue () {
                       return ((MemberElement)element).getName().getName();
                   }

                   /** Sets the value */
                   public void setValue(final Object val) throws IllegalArgumentException,
                       IllegalAccessException, InvocationTargetException {
                       super.setValue(val);
                       if (!(val instanceof String))
                           throw new IllegalArgumentException();

                       runAtomic(element, new SourceEditSupport.ExceptionalRunnable() {
                                     public void run() throws SourceException {
                                         String sourceName = ((String) val).trim();
                                         String fullName = sourceName;

                                         String prevName = ((MemberElement)element).getName().getFullName();
                                         int dot = prevName.lastIndexOf("."); // NOI18N
                                         if (dot != -1) {
                                             fullName = prevName.substring(0, dot + 1) + sourceName;
                                         }

                                         if (testJavaId(sourceName)) {
                                             ((MemberElement)element).setName(Identifier.create(fullName, sourceName));
                                         }
                                     }
                                 });
                   }
               };
    }
}
