/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.test.java.gui.wizards;

import org.netbeans.junit.*;
import org.netbeans.jellytools.*;
import org.netbeans.jellytools.nodes.*;
import org.netbeans.jemmy.JemmyProperties;
import org.netbeans.jemmy.TestOut;
import org.netbeans.jemmy.operators.*;
import org.netbeans.test.java.Utilities;
import org.netbeans.test.java.gui.GuiUtilities;
import java.io.*;

/**
 * Tests the New File Wizard.
 * @author Roman Strobl
 */
public class NewFileWizardTest extends JellyTestCase {

    // default timeout for actions in miliseconds
    private static final int ACTION_TIMEOUT = 1000; 

    // name of sample project
    private static final String TEST_PROJECT_NAME = "TestProject"; 

    // name of sample package
    private static final String TEST_PACKAGE_NAME = "test"; 

    // name of sample class
    private static final String TEST_CLASS_NAME = "TestClass"; 

    // name of invalid package
    private static final String TEST_PACKAGE_NAME_INVALID = "a/b"; 
    
    /**
     * error log
     */
    protected static PrintStream err;
    /**
     * standard log
     */
    protected static PrintStream log;
   
    // workdir, default /tmp, changed to NBJUnit workdir during test
    private String workDir = "/tmp";
    
    static String projectDir;
        
    /**
     * Adds tests into the test suite.
     * @return suite
     */
    public static NbTestSuite suite() {
        NbTestSuite suite = new NbTestSuite();        
        suite.addTest(new NewFileWizardTest("testCreateProject"));
        // test requires an opened project
        suite.addTest(new NewFileWizardTest("testCreatePackage"));
        // test requires an opened project and created package
        suite.addTest(new NewFileWizardTest("testDeletePackage"));
        // test requires an opened project
        suite.addTest(new NewFileWizardTest("testDeleteProject"));
        suite.addTest(new NewFileWizardTest("testNewFileWizardComplex"));
        suite.addTest(new NewFileWizardTest("testCreatePackageFailure"));
        return suite;
    }
            
    /**
     * Main method for standalone execution.
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        junit.textui.TestRunner.run(suite());
    }    

    /**
     * Sets up logging facilities.
     */
    public void setUp() {
        System.out.println("########  "+getName()+"  #######");
        err = getLog();
        log = getRef();        
        JemmyProperties.getProperties().setOutput(new TestOut(null,
                new PrintWriter(err, true), new PrintWriter(err, false), null));
        try {
            File wd = getWorkDir();
            workDir = wd.toString();
        } catch (IOException e) { }
    }
    
    /**
     * Creates a new instance of Main
     * @param testName name of test
     */
    public NewFileWizardTest(String testName) {
        super(testName);
    }
     
    /**
     * Tests creating a project.
     */
    public void testCreateProject() {
        projectDir = GuiUtilities.createProject(TEST_PROJECT_NAME, workDir);
    }

    /**
     * Tests deleting a project including files on hard drive.
     */
    public void testDeleteProject() {
        GuiUtilities.deleteProject(TEST_PROJECT_NAME, TEST_CLASS_NAME, 
                projectDir, false);
    }

    /**
     * Tests creating of a package.
     */
    public void testCreatePackage() {
        GuiUtilities.createPackage(TEST_PROJECT_NAME, TEST_PACKAGE_NAME);
    }

    /**
     * Tests deleting of a package.
     */
    public void testDeletePackage() {
        // delete a package
        Node pn = new ProjectsTabOperator().getProjectRootNode(
                TEST_PROJECT_NAME);
        pn.select();
        
        Node n = new Node(pn, org.netbeans.jellytools.Bundle.getString(
                "org.netbeans.modules.java.j2seproject.Bundle", 
                "NAME_src.dir")+"|"+TEST_PACKAGE_NAME); 
        n.select();
        n.performPopupAction(org.netbeans.jellytools.Bundle.getString(
                "org.netbeans.core.projects.Bundle", "LBL_action_delete")); 

        // confirm
        new NbDialogOperator(Bundle.getString("org.openide.explorer.Bundle", 
                "MSG_ConfirmDeleteObjectTitle")).yes();

    }
    
    /**
     * Tests New File wizard.
     * - create test project
     * - create test package
     * - create test class through New File wizard (core of the test)
     * - close opened file and project
     * - delete the project incl. all files on disc
     */
    public void testNewFileWizardComplex() {
        // create test project
        testCreateProject();
        
        // create test package
        testCreatePackage();

        // select project node
        Node pn = new ProjectsTabOperator().getProjectRootNode(
                TEST_PROJECT_NAME);
        pn.select();
        
        // create test class                
        NewFileWizardOperator op = NewFileWizardOperator.invoke();
        
        op.selectCategory(Bundle.getString("org.netbeans.modules.java.Bundle",
                "Templates/Classes"));        
        op.selectFileType(Bundle.getString("org.netbeans.modules.java.Bundle",
                "Templates/Classes/Class.java"));        
        op.next();
        
        JTextFieldOperator tf = new JTextFieldOperator(op);
        tf.setText(TEST_CLASS_NAME);
        
        op.finish();
        
        // check generated source
        EditorOperator editor = new EditorOperator(TEST_CLASS_NAME);
        String text = editor.getText();
        
        // check if class name is generated 4 times in the source code
        int oldIndex = 0;
        for (int i=0; i<4; i++) {
            oldIndex = text.indexOf(TEST_CLASS_NAME, oldIndex);
            if (oldIndex>-1) oldIndex++;
        }
                
        assertTrue("Error in generated class "+TEST_CLASS_NAME+".java.",
                oldIndex!=-1);

        editor.close();

        // delete test package
        testDeletePackage();
        
        // delete test project
        testDeleteProject();

    }    

    /**
     * Negative test for creating of a package.
     */
    public void testCreatePackageFailure() {
        NewFileWizardOperator op = NewFileWizardOperator.invoke();

        // wait till all fields are loaded
        JDialogOperator jdo = new JDialogOperator(
                org.netbeans.jellytools.Bundle.getString(
                "org.netbeans.modules.project.ui.Bundle", 
                "LBL_NewFileWizard_Title"));                
        JTreeOperator jto = new JTreeOperator(jdo, 0);
        boolean exitLoop = false;
        for (int i=0; i<10; i++) {
            for (int j=0; j<jto.getChildCount(jto.getRoot()); j++) {
                if (jto.getChild(jto.getRoot(), j).toString()==
                        Bundle.getString("org.netbeans.modules.java.Bundle",
                        "Templates/Classes")) {
                    exitLoop = true;
                    break;
                }
            }
            if (exitLoop) break;
            Utilities.takeANap(1000);
        }
        
        // choose package
        op.selectCategory(Bundle.getString("org.netbeans.modules.java.Bundle", 
                "Templates/Classes"));
        op.selectFileType(Bundle.getString("org.netbeans.modules.java.Bundle", 
                "Templates/Classes/Package"));        
        op.next();
        
        // try to set an invalid name
        JTextFieldOperator tfp = new JTextFieldOperator(op, 0);
        tfp.setText(TEST_PACKAGE_NAME_INVALID);
        for (int i=0; i<10; i++) {
            JButtonOperator jbo = new JButtonOperator(op, 
                Bundle.getString("org.openide.Bundle", "CTL_FINISH"));
            if (!jbo.isEnabled()) break;
            Utilities.takeANap(1000);
        }        
        
        // check finish button
        JButtonOperator jbo = new JButtonOperator(op, 
                Bundle.getString("org.openide.Bundle", "CTL_FINISH"));        
        assertFalse("Finish button should be disabled for package with "
                +"invalid name.", jbo.isEnabled());        
        
        new NbDialogOperator(Bundle.getString(
                "org.netbeans.modules.project.ui.Bundle", 
                "LBL_NewProjectWizard_Subtitle")+" "
                +Bundle.getString("org.netbeans.modules.java.Bundle", 
                "Templates/Package")).cancel(); 
    }
    
}
