/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Ralph Krueger. 
 */

package org.netbeans.modules.changelog;


import java.util.*;
import org.openide.*;
import java.io.*;

import org.netbeans.modules.changelog.html.*;

/**
 * prints the processed groups to text file..
 * @author  ralph
 */
public class LogPrinter_HTML implements LogPrinter {
    
    private static final String SPECIAL_KEY = "SPECIAL"; //NOI18N
    private static final String PREV_REVISION = "previous_revision"; //NOI18N
    private static final String FILE_PATH = "file_path"; //NOI18N
    private static final String CURR_REVISION = "current_revision"; //NOI18N
    
    
    private java.io.PrintWriter writer;
    
    private File file;
    
    private boolean view;
    
    private boolean includeSummary;
    
    private boolean hasCommonBranch;
    
    private ChangeLogHTMLService service;
    
    public LogPrinter_HTML(File fileParam, boolean view) {
        this.file = fileParam;
        this.view = view;
        writer = null;
        try {
            if (this.file == null) {
                this.file = File.createTempFile("chlog", ".html");
                file.deleteOnExit();
            }
            if (!file.exists()) {
                try {
                    if (file.getParentFile() != null) {
                        file.getParentFile().mkdirs();
                    }
                    file.createNewFile();
                } catch (IOException exc) {
                    org.openide.ErrorManager.getDefault().notify(exc);
                    System.out.println("error while creating file..");
                }
            }
            writer = new PrintWriter(new FileOutputStream(file));
            
        } catch (IOException exc) {
            org.openide.ErrorManager.getDefault().notify(exc);
            System.out.println("error while opening file..");
        }
    }
    
    public LogPrinter_HTML(File file, boolean view, ChangeLogHTMLService service) {
        this(file, view);
        this.service = service;
    }
    
    public void printHeader(ChangeLogProcessor processor) {
        includeSummary = processor.isIncludeSummary();
        if (writer == null) {
            return;
        }
        writer.println("<HTML>");
        writer.println("<HEAD>");
        writer.println("<TITLE> CVS Repository Query Results </TITLE>");
        writer.println("</HEAD>");
        writer.println("<BODY>");
        
        
        if (processor.isIncludeQueryDescription()) {
            writer.println("<H1> Query: </H1>");
            if (processor.getDateRange() != null) {
                writer.println("   Date Range: " + ChangeLogUtils.escapeString(processor.getDateRange()));
                writer.println("<P>");
            }
            if (processor.getRevisionRange() != null) {
                writer.println("   Revision Filter:" + ChangeLogUtils.escapeString(processor.getRevisionRange()));
                writer.println("<P>");
            }
            if (processor.getUser() != null) {
                writer.println("    User Filter:" + ChangeLogUtils.escapeString(processor.getUser()));
                writer.println("<P>");
            }
            if (processor.getMessageFilter() != null) {
                String messageType = "";
                if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_SUBSTRING) {
                    messageType = "Substring";
                } else if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_SOME_WORDS) {
                    messageType = "Any of words";
                } else if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_ALL_WORDS) {
                    messageType = "All of words";
                } else if (processor.getMessageFilterType() == ChangeLogProcessor.MESSAGE_FILTER_REGEXP) {
                    messageType = "Regular expression";
                }
                writer.println("   Message Filter (" + ChangeLogUtils.escapeString(messageType) + "):" + ChangeLogUtils.escapeString(processor.getMessageFilter()));
                writer.println("<P>");
            }
            if (processor.getFileFilter() != null) {
                String fileType = "";
                if (processor.getFileFilterType() == ChangeLogProcessor.FILE_FILTER_SUBSTRING) {
                    fileType = "Substring";
                } else if (processor.getFileFilterType() == ChangeLogProcessor.FILE_FILTER_REGEXP) {
                    fileType = "Regular expression";
                }
                writer.println("   Contained Files Filter (" + ChangeLogUtils.escapeString(fileType) + "):" + ChangeLogUtils.escapeString(processor.getFileFilter()));
                writer.println("<P>");
            }
            if (processor.getSortMode() == ChangeLogProcessor.SORT_BY_DATE) {
                writer.print("   Sort: by Date");
            } else if (processor.getSortMode() == ChangeLogProcessor.SORT_BY_USER) {
                writer.print("   Sort: by User");
            }
            if (processor.isDescendingSort()) {
                writer.println(" (Descending)");
            } else {
                writer.println(" (Ascending)");
            }
            writer.println("<P>");
            if (processor.isIncludeBranchNames()) {
                writer.println("Includes Branch Names Information <P>");
            }
            writer.println("<BR>");
            writer.println("<BR>");
        }
        writer.println("<H1> Results: </H1>");
    }
    
    public void printGroupHeader(RevisionsGroup group) {
        if (writer == null) {
            return;
        } 
        hasCommonBranch = false;
        writer.println("<UL>");
          writer.println("--------------------------------------------------");
          writer.println("<LI>User: <I>" + ChangeLogUtils.escapeString(group.getUser()) + "</I>");
          writer.println("<LI>Date: <I>" + ChangeLogUtils.escapeString(group.getStartingDate().toString()) + "</I>");
          if (group.getCommonBranch() != null) {
              writer.println("<LI>Branch: <B><I>" + ChangeLogUtils.escapeString(group.getCommonBranch()) + "</I></B>");
              hasCommonBranch = true;
          }
          writer.println("<LI>Message: <B>" + ChangeLogUtils.escapeString(group.getMessage()) + "</B>");
          writer.println("<UL>");
    }
    
    public void printSingleRevision(LogInfoRevision revision) {
        if (writer == null ) {
            return;
        }
        String repoFileName = revision.getLogInfoHeader().getRepositoryFilename();
        // remove the trailing ,v part..
        repoFileName = repoFileName.substring(0, repoFileName.length() - 2);

        writer.println("<LI>"); //NOI18N
        HashMap replacementMap = computeReplacements(revision);
        if (service != null) {
            String url = ChangeLogUtils.replaceArguments(service.getRevisionURL(), replacementMap);
            writer.println("<A href='" +  url +  "'>" +  revision.getNumber() + "</A> "); //NOI18N
        } else {
            writer.println(revision.getNumber());
        }
        writer.println("  " + repoFileName);
        if ((!replacementMap.containsKey(SPECIAL_KEY))  && //NOI18N
               replacementMap.containsKey(PREV_REVISION) && //NOI18N
               service != null) {
            String diffUrl = ChangeLogUtils.replaceArguments(service.getDiffURL(), replacementMap);
//            String diffUrl = "http://www.netbeans.org/source/browse/" + reducedRepoFileName 
//                   + ".diff?r1=text&tr1=" + prevRevision + "&r2=text&tr2=" + revision.getNumber() + "&f=h";
            writer.println("[<A href='" + diffUrl + "'> Diff </A>]"); 
        } else {
            // do nothing..
        }
        if (replacementMap.containsKey(SPECIAL_KEY)) { //NOI18N
            writer.println(" ["+ replacementMap.get(SPECIAL_KEY) + "]");
        }
        if (!revision.getBranch().equals("") && !hasCommonBranch) {
            writer.println(" On Branch: <B>" + revision.getBranch() + "</B>");
        }
    }
    
    private HashMap computeReplacements(LogInfoRevision revision) {
        String repoFileName = revision.getLogInfoHeader().getRepositoryFilename();
        // remove the trailing ,v part..
        repoFileName = repoFileName.substring(0, repoFileName.length() - 2);
        HashMap replacementMap = new HashMap();
        if (service != null) {
            String reducedRepoFileName;
            if (repoFileName.startsWith(service.getCvsRootPrefix())) {
                reducedRepoFileName = repoFileName.substring(service.getCvsRootPrefix().length() + 1);
            } else {
                reducedRepoFileName = repoFileName;
            }
            replacementMap.put(FILE_PATH, reducedRepoFileName); //NOI18N
        } else {
            replacementMap.put(FILE_PATH,  repoFileName);
        }
        replacementMap.put(CURR_REVISION, revision.getNumber()); //NOI18N
        String revNumber = revision.getNumber();
        if (revision.getState().equals("dead")) { //NOI18N
            replacementMap.put(SPECIAL_KEY, "Removed"); //NOI18N
        } else {
            int[] revNumbers = ChangeLogUtils.convertRevisionToIntArray(revision.getNumber());
            if (revNumbers[revNumbers.length - 2] % 2 != 0 && revNumbers.length != 2) {
                replacementMap.put(SPECIAL_KEY, "Imported"); //NOI18N
            } else {
                if (revNumbers[revNumbers.length - 1] == 1) {
                    if (revNumbers.length == 2) {
                        replacementMap.put(SPECIAL_KEY, "Added"); //NOI18N
                    } else {
                        revNumbers[revNumbers.length - 1] = 0;
                        revNumbers[revNumbers.length - 2] = 0;
                        String prevrev = ChangeLogUtils.convertIntArrayToRevision(revNumbers);
                        replacementMap.put(PREV_REVISION, prevrev);
                    }
                } else {
                    revNumbers[revNumbers.length - 1] = revNumbers[revNumbers.length - 1] - 1;
                    String prevrev = ChangeLogUtils.convertIntArrayToRevision(revNumbers);
                    replacementMap.put(PREV_REVISION, prevrev);
                }
            }
        }
        return replacementMap;
        
    }
    
    
    public void printGroupFooter(RevisionsGroup group) {
        if (writer == null) {
            return;
        }
        writer.println("</UL>");
        writer.println("</UL>");
    }
    
    public void printSummary(SummaryProcessor processor) {
        if (writer == null) {
            return;
        }
        if (includeSummary) {
            writer.println("=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=");
            writer.println("<H1>Summary:</H1>");
            writer.println("<B>Number of changes: " + processor.getCommitCount() + "</B>");
            writer.println("<P>");
            writer.print("<B>Developers: </B>");
            String[] users = processor.getUserList();
            for (int i = 0; i < users.length; i++) {
                if (i != users.length - 1) {
                    writer.print(ChangeLogUtils.escapeString(users[i]) + ", ");
                } else {
                    writer.print(ChangeLogUtils.escapeString(users[i]));
                }
            }
            writer.println("<P> <BR>");
            String[] mostChanged = processor.getMostChangedFiles();
            writer.println("<B>Most frequently changed files:</B>");
            writer.println("<UL>");
            for (int j = 0; j < mostChanged.length; j++) {
                writer.println("<LI>" + ChangeLogUtils.escapeString(mostChanged[j]));
            }
            writer.println("</UL>");
            writer.println("<P> <BR>");
            writer.println("<B>Most active developers (based on number of commits): </B>");
            String[] mostActive= processor.getMostActiveUsers();
            writer.println("<UL>");
            for (int k = 0; k < mostActive.length; k++) {
                writer.println("<LI>" + ChangeLogUtils.escapeString(mostActive[k]));
            }
            writer.println("</UL>");
            writer.println("<P> <BR>");
        }
    }
    
    public void printFooter(ChangeLogProcessor processor) {
        if (writer == null) {
            return;
        }
        writer.println("</BODY>");
        writer.println("</HTML>");
        writer.flush();
        writer.close();
        if (view) {
            try {
                TopManager.getDefault().showUrl(file.toURL());
            } catch (java.net.MalformedURLException exc) {
                System.out.println("malformed URL.");
            } 
        }
        
        
    }
    
}
