/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.versioning.system.cvss;

import org.netbeans.modules.turbo.TurboProvider;
import org.openide.ErrorManager;

import java.io.*;
import java.util.Map;
import java.util.Properties;
import java.util.Iterator;
import java.util.HashMap;

/**
 * Persistent FileStatusCache layer. It supports
 * {@link #ATTR_STATUS_MAP} attribute
 * on <code>File</code> entities.
 *
 * @author Petr Kuzel
 */
final class CacheTurboProvider implements TurboProvider {

    static final String ATTR_STATUS_MAP = "org.netbeans.modules.versioning.system.cvss.CacheTurboProvider.STATUS_MAP";  // NOI18N

    private static final String FILE_NAME_VERSION_2 = "nb_cvs2.properties"; // NOI18N


    public boolean recognizesAttribute(String name) {
        return ATTR_STATUS_MAP == name;
    }

    public boolean recognizesEntity(Object key) {
        return key instanceof File;
    }

    public Object readEntry(Object key, String name, MemoryCache memoryCache) {
        if (ATTR_STATUS_MAP == name) {
            File dir = (File) key;
            assert dir.isDirectory() : "Dir expected: " + dir.getPath();  // NOI18N

            File cacheFile = getCacheFile(dir);
            if (cacheFile == null) {
                // non versioned folder, it's resolved by FileStatusCache
                return null;
            }

            if (cacheFile.exists() == false) {
                // - very first request, it's resolved by FileStatusCache
                // - all were up-to-date, it's resolved by FileStatusCache
                return null;
            }

            // load from properties

            Properties props = new Properties();
            InputStream in = null;
            try {
                in = new FileInputStream(cacheFile);
                in = new BufferedInputStream(in);
                props.load(in);
            } catch (IOException ex) {
                if (in != null) {
                    try {
                        in.close();
                    } catch (IOException alreadyClosed) {
                    }
                }
            }

            // convert to target format

            Map ret = new HashMap(props.size());
            Iterator it = props.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                String fileName = (String) entry.getKey();
                String statusString = (String) entry.getValue();
                FileInformation info = FileInformation.fromExternalForm(statusString);
                File fileKey = new File(dir, fileName);  // XXX use dir relative names
                ret.put(fileKey, info);
            }
            return ret;
        }
        return null;
    }

    public boolean writeEntry(Object key, String name, Object value) {
        if (ATTR_STATUS_MAP == name) {
            File dir = (File) key;
            assert dir.isDirectory() : "Dir expected: " + dir.getPath();  // NOI18N

            Map infoMap = (Map) value;

            // remove stale file for unmanaged subtrees and null requests
            if (infoMap == null) {
                File cacheFile = getCacheFile(dir);
                if (cacheFile != null && cacheFile.exists()) {
                    if (cacheFile.delete() == false) {
                        ErrorManager.getDefault().log("CVS: can not delete: " + cacheFile.getPath()); // NOI18N
                    }
                }
                return true;
            }

            File cacheFile = getCacheFile(dir);
            if (cacheFile == null) {
                // unversioned folder, it's handled in read
                return true;
            }

            // write down to propeties file format

            if (infoMap.size() == 0) return true;

            Properties props = new Properties();
            Iterator it = infoMap.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry entry = (Map.Entry) it.next();
                String fileName = ((File) entry.getKey()).getName(); // XXX use dir relative names
                FileInformation status = (FileInformation) entry.getValue();
                props.setProperty(fileName, status.toExternalForm());
            }

            OutputStream out = null;
            try {
                out = new FileOutputStream(cacheFile);
                out = new BufferedOutputStream(out);
                props.store(out, "Maps dir name => recent status.");  // NOI18N
            } catch (IOException ex) {
                ErrorManager.getDefault().notify(ex);
            } finally {
                if (out != null) {
                    try {
                        out.close();
                    } catch (IOException alreadyClosed) {
                    }
                }
            }

            return true;
        }
        return false;
    }

    /** Locate cache file or null for unversioned folders. */
    private File getCacheFile(File folder) {
        File cvs = new File(folder, "CVS");  // NOI18N
        if (cvs.exists() == false) return null;
        return new File(folder, "CVS" + File.separatorChar + FILE_NAME_VERSION_2); // NOI18N
    }
}
