/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.settings;

import org.openide.options.SystemOption;

import java.util.Collections;
import java.util.List;
import java.util.ArrayList;
import java.io.Serializable;

/**
 * Persistent singleton storage of various recent values.
 * Persistence is driven by associated BeanInfo.
 *
 * @author Petr Kuzel
 */
public final class HistorySettings extends SystemOption {

    private static final long serialVersionUID = 1L;

    /** Recently used CVS roots <code>List&lt;String></code>. Most recent is the first one. */
    public static final String PROP_CVS_ROOTS = "cvsRoots";  // NOI18N

    /** Recently used working directories for checkout. Typical restore value is parent folder. */
    public static final String PROP_CHECKOUT_DIRECTORY = "checkoutDirectory"; // NOI18N

    /** If true checkout end with poping up dialog. */
    public static final String PROP_SHOW_CHECKOUT_COMPLETED = "showCheckoutCompleted"; // NOI18N

    /** For lists, it defines number of kept entries */
    public static final int HISTORY_LEN = 8;

    /** Recent proxy settings */
    public static final String PROP_HTTP_PROXY_HOST = "httpProxyHost"; // NOI18N
    public static final String PROP_HTTP_PROXY_PORT = "httpProxyPort"; // NOI18N
    public static final String PROP_SOCKS_PROXY_HOST = "socksProxyHost"; // NOI18N
    public static final String PROP_SOCKS_PROXY_PORT = "socksProxyPort"; // NOI18N

    public static final String PROP_EXT_COMMAND = "extCommand"; // NOI18N

    /** Creates new instance, for externalizion purposes only. */
    public HistorySettings() {        
    }
    
    public static HistorySettings getDefault() {
        return (HistorySettings) HistorySettings.findObject(HistorySettings.class, true);
    }

        
    /**
     * Loads recent list of given name.
     *
     * @param name identifies the list to load
     * @return loaded read-only list never <code>null</code>
     */
    public static List getRecent(String name) {
        List values = (List) getDefault().getProperty(name);
        if (values == null) {
            return Collections.EMPTY_LIST;
        } else {
            return values;
        }
    }

    /**
     * Stores recent list of given name.
     *
     * @param name identifies the list to load
     * @param values updated recent values
     */
    public static void setRecent(String name, List values) {
        getDefault().putProperty(name, new ArrayList(values), true);
    }

    /**
     * Prepend given value into list given by name. Possibly
     * eliminates already existing value in older slots.
     *
     * @param name identifies the list to load
     * @param recent new value
     */
    public static void addRecent(String name, Serializable recent) {
        List values = (List) getDefault().getProperty(name);
        List list;
        if (values == null) {
            list = new ArrayList(1);
            list.add(recent);
        } else {
            list = new ArrayList(values);
            list.remove(recent);
            while (list.size() > HISTORY_LEN) {
                list.remove(list.size() -1);
            }
            list.add(0, recent);
        }
        getDefault().putProperty(name, list, true);
    }

    public static void setFlag(String name, int value) {
        getDefault().putProperty(name, new Integer(value));
    }

    public static int getFlag(String name, int defaultValue) {
        Integer i = (Integer) getDefault().getProperty(name);
        if (i != null) {
            return i.intValue();
        }
        return defaultValue;
    }

    public String displayName() {
        return getClass().getName();
    }

    public List getCvsRoots() {
        return getRecent(PROP_CVS_ROOTS);
    }
    
    public void setCvsRoots(List values) {
        setRecent(PROP_CVS_ROOTS, values);
    }

    public List getCheckoutDirectory() {
        return getRecent(PROP_CHECKOUT_DIRECTORY);
    }

    public void setCheckoutDirectory(List values) {
        setRecent(PROP_CHECKOUT_DIRECTORY, values);
    }

    public boolean getShowCheckoutCompleted() {
        int flag = getFlag(PROP_SHOW_CHECKOUT_COMPLETED, -1);
        return flag != 0;
    }

    public void setShowCheckoutCompleted(boolean show) {
        setFlag(PROP_SHOW_CHECKOUT_COMPLETED, show ? -1 : 0);
    }

    public String getHttpProxyHost() {
        String host = (String) getProperty(PROP_HTTP_PROXY_HOST);
        if (host == null) {
            host = "";  // NOI18N
        }
        return host;
    }

    public void setHttpProxyHost(String httpProxyHost) {
        putProperty(PROP_HTTP_PROXY_HOST, httpProxyHost);
    }

    public String getHttpProxyPort() {
        String port = (String) getProperty(PROP_HTTP_PROXY_PORT);
        if (port == null) {
            port = "80"; // NOI18N
        }
        return port;
    }

    public void setHttpProxyPort(String httpProxyPort) {
        putProperty(PROP_HTTP_PROXY_PORT, httpProxyPort);
    }

    public String getSocksProxyHost() {
        String host = (String) getProperty(PROP_SOCKS_PROXY_HOST);
        if (host == null) {
            host = ""; // NOI18N
        }
        return host;        
    }

    public void setSocksProxyHost(String socksProxyHost) {
        putProperty(PROP_SOCKS_PROXY_HOST, socksProxyHost);
    }

    public String getSocksProxyPort() {
        String port = (String) getProperty(PROP_SOCKS_PROXY_PORT);
        if (port == null) {
            port = "1080"; // NOI18N
        }
        return port;        
    }

    public void setSocksProxyPort(String socksProxyPort) {
        putProperty(PROP_SOCKS_PROXY_PORT, socksProxyPort);
    }

    public void setExtCommand(String command) {
        putProperty(PROP_EXT_COMMAND, command);
    }

    public String getExtCommand() {
        return (String) getProperty(PROP_EXT_COMMAND);
    }

}
