/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.settings;

import org.openide.options.SystemOption;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.netbeans.modules.versioning.system.cvss.CvsMetadata;

import java.util.*;
import java.io.Serializable;
import java.io.File;

/**
 * Stores CVS metadata from deleted folders. They will be used when user commits his deletes.
 *
 * <p>It also keep scheduled (project) folders that does not
 * exist in repository to be added into repository.
 *
 * @author Maros Sandor
 */
public class MetadataAttic extends SystemOption {
    
    private static final long serialVersionUID = 1L;

    public static final String PROP_FOLDERS_MAP = "foldersMap";  // NOI18N

    public static final String PROP_SCHEDULED_FOLDERS = "scheduledProjects";  // NOI18N

    /** For exteralization purposes. */
    public MetadataAttic() {
    }

    static int id;

    // clients code ~~~~~~~~~~~~~~~~~~~~~~~~~

    public static void cleanUp() {
        Map map = getDefault().getFoldersMap();
        Set directories = map.keySet();
        for (Iterator i = directories.iterator(); i.hasNext();) {
            File dir = (File) i.next();
            if (dir.exists()) i.remove();
        }
        map = new PersistentMap(map);
        getDefault().putProperty(PROP_FOLDERS_MAP, map, true);
    }
    
    public static CvsMetadata getMetadata(File dir) {
        Metadata data = (Metadata) getDefault().getFoldersMap().get(dir);
        return data != null ? data.data : null;
    }
    
    public static void setMetadata(File dir, CvsMetadata data) {
        Map map = getDefault().getFoldersMap();
        if (data != null) {
            Metadata mdata = new Metadata();
            mdata.data = data;
            map.put(dir, mdata);
        } else {
            map.remove(dir);
        }
        map = new PersistentMap(map);
        getDefault().putProperty(PROP_FOLDERS_MAP, map, true);
    }

    /**
     * Registers new folder to be added into repository.
     *
     * @param fileObject folder to be added
     * @param root cvs root <code>null</code> for removal
     * @param repository repository path or <code>null</code> for removal
     */
    public static void scheduleFolder(FileObject fileObject, String root, String repository) {
        File file = FileUtil.toFile(fileObject);
        Map map = getDefault().getScheduledFolders();
        if (root == null || repository == null) {
            map.remove(file);
        } else {
            Repository repo = new Repository();
            repo.cvsRoot = root;
            repo.cvsRepository = repository;
            map.put(file, repository);
        }
        getDefault().setScheduledFolders(map);
    }

    /**
     * @return repository path or <code>null</code> if not scheduled
     */
    public static String getScheduledRepository(FileObject fileObject) {
        Map map = getDefault().getScheduledFolders();
        File f = FileUtil.toFile(fileObject);
        Repository repo = (Repository) map.get(f);
        if (repo != null) {
            return repo.cvsRepository;
        }
        return null;
    }

    /**
     * @return cvs root or <code>null</code> if not scheduled
     */
    public static String getScheduledRoot(FileObject fileObject) {
        Map map = getDefault().getScheduledFolders();
        File f = FileUtil.toFile(fileObject);
        Repository repo = (Repository) map.get(f);
        if (repo != null) {
            return repo.cvsRoot;
        }
        return null;
    }

    // System option implementation ~~~~~~~~~~~~~~~~~~

    private static MetadataAttic getDefault() {
        return (MetadataAttic) MetadataAttic.findObject(MetadataAttic.class, true);
    }

    public String displayName() {
        return getClass().getName();
    }

    public Map getFoldersMap() {
        Map map = (Map) getProperty(PROP_FOLDERS_MAP);
        if (map == null) {
            map = new MetadataAttic.PersistentMap();
        }
        return map;
    }

    public void setFoldersMap(Map rootsMap) {
        putProperty(PROP_FOLDERS_MAP, new MetadataAttic.PersistentMap(rootsMap), true);
    }

    public Map getScheduledFolders() {
        Map map = (Map) getProperty(PROP_SCHEDULED_FOLDERS);
        if (map == null) {
            map = new MetadataAttic.PersistentMap();
        }
        return map;
    }

    public void setScheduledFolders(Map projects) {
        putProperty(PROP_SCHEDULED_FOLDERS, new MetadataAttic.PersistentMap(projects), true);
    }

    /**
     * Holds associated settings.
     */
    private static final class Metadata implements Serializable {
        
        private static final long serialVersionUID = 1L;

        private CvsMetadata  data;
    }

    private static final class Repository implements Serializable {
        private static final long serialVersionUID = 1L;
        public String cvsRoot;
        public String cvsRepository;
    }

    /**
     * Map that tricks system option setter optimalization
     */
    private final static class PersistentMap extends HashMap {
        private static final long serialVersionUID = 1L;

        public PersistentMap() {
        }

        public PersistentMap(Map map) {
            super(map);
        }

        public boolean equals(Object o) {
            return o == this;
        }

        public int hashCode() {
            return System.identityHashCode(this);
        }
    }
    
}
