/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.ui.actions.diff;

import org.netbeans.api.diff.Diff;
import org.netbeans.api.diff.DiffView;
import org.netbeans.api.diff.StreamSource;
import org.netbeans.modules.versioning.system.cvss.*;
import org.netbeans.modules.versioning.system.cvss.ui.actions.update.UpdateExecutor;
import org.netbeans.modules.versioning.system.cvss.ui.actions.commit.CommitAction;
import org.netbeans.modules.versioning.system.cvss.util.Utils;
import org.netbeans.modules.versioning.system.cvss.util.Context;
import org.netbeans.modules.versioning.util.VersioningListener;
import org.netbeans.modules.versioning.util.VersioningEvent;
import org.netbeans.lib.cvsclient.command.update.UpdateCommand;
import org.netbeans.lib.cvsclient.command.GlobalOptions;
import org.openide.util.NbBundle;
import org.openide.util.RequestProcessor;
import org.openide.util.lookup.Lookups;
import org.openide.ErrorManager;
import org.openide.LifecycleManager;
import org.openide.nodes.Node;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;
import org.openide.windows.TopComponent;
import org.openide.windows.WindowManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

import javax.swing.*;
import javax.swing.plaf.basic.BasicComboBoxRenderer;
import javax.swing.border.CompoundBorder;
import java.io.IOException;
import java.io.File;
import java.util.*;
import java.util.List;
import java.awt.event.*;
import java.awt.*;
import java.awt.image.BufferedImage;

/**
 * Main DIFF GUI panel that includes the navidation toolbar and DIFF component.
 * 
 * @author Maros Sandor
 */
public class DiffMainPanel extends javax.swing.JPanel implements ActionListener, VersioningListener, DiffSetupSource {

    /**
     * Array of DIFF setups that we show in the DIFF view. Contents of this array is changed if
     * the user switches DIFF types.
     */
    private Setup[] setups;

    /**
     * Context in which to DIFF.
     */
    private final Context context;

    private int                     displayStatuses;

    /**
     * Display name of the context of this diff.
     */ 
    private final String contextName;
    
    private int currentType;
    private int currentDifferenceIndex;
    private int currentIndex = -1;
    
    private RequestProcessor.Task prepareTask;
    private DiffPrepareTask dpt;

    private AbstractAction          nextAction;
    private AbstractAction          prevAction;
    
    // TODO: reuse this code ... the same pattern is in SynchronizePanel
    private static final RequestProcessor   rp = new RequestProcessor("CVS-VersioningView", 1);  // NOI18N

    /**
     * null for view that are not
     */
    private RequestProcessor.Task   refreshTask;

    private ExecutorGroup           group;
    private volatile boolean        executed;

    private JComponent              diffView;

    /**
     * Creates diff panel and immediatelly starts loading...
     */
    public DiffMainPanel(Context context, int initialType, String contextName, ExecutorGroup group) {
        this.context = context;
        this.contextName = contextName;
        this.group = group;
        currentType = initialType;
        initComponents();
        setupComponents();
        refreshSetups();
        refreshComponents();
        refreshTask = rp.create(new RefreshViewTask());
    }

    /**
     * Construct diff component showing just one file.
     * It hides All, Local, Remote toggles and file chooser combo.
     */
    public DiffMainPanel(File file, String rev1, String rev2) {
        context = null;
        contextName = file.getName();
        initComponents();
        setupComponents();
        localToggle.setVisible(false);
        remoteToggle.setVisible(false);
        allToggle.setVisible(false);
        navigationCombo.setVisible(false);
        refreshButton.setVisible(false);
        updateButton.setVisible(false);
        commitButton.setVisible(false);

        // mimics refreshSetups()
        setups = new Setup[] {
            new Setup(file, rev1, rev2)
        };
        setDiffIndex(0, 0);
        dpt = new DiffPrepareTask(setups);
        prepareTask = RequestProcessor.getDefault().post(dpt);
    }

    /**
     * Called by the enclosing TopComponent to interrupt the fetching task.
     */
    void componentClosed() {
        setups = null;
        if (prepareTask != null) {
            prepareTask.cancel();
        }
    }

    public synchronized void setGroup(ExecutorGroup group) {
        this.group = group;
        if (executed) {
            group.executed();
        }
    }

    void requestActive() {
        if (diffView != null) {
            diffView.requestFocusInWindow();
        }
    }

    private static class ColoredComboRenderer extends BasicComboBoxRenderer {

        public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            if (isSelected) {
                value = ((Setup) value).getBaseFile().getName();
            }
            return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
        }
    }

    private void setupComponents() {
        controlsToolbar.putClientProperty("JToolBar.isRollover", Boolean.TRUE);  // NOI18N
        controlsToolbar.setLayout(new DiffMainPanel.ToolbarLayout());
        
        navigationCombo.addActionListener(this);
        navigationCombo.setRenderer(new ColoredComboRenderer());
        refreshButton.addActionListener(this);
        updateButton.addActionListener(this);
        commitButton.addActionListener(this);
        localToggle.addActionListener(this);
        remoteToggle.addActionListener(this);
        allToggle.addActionListener(this);
        
        refreshButton.setToolTipText(NbBundle.getMessage(DiffMainPanel.class, "MSG_RefreshDiff_Tooltip"));
        updateButton.setToolTipText(NbBundle.getMessage(DiffMainPanel.class, "MSG_UpdateDiff_Tooltip", contextName));
        commitButton.setToolTipText(NbBundle.getMessage(DiffMainPanel.class, "MSG_CommitDiff_Tooltip", contextName));
        ButtonGroup grp = new ButtonGroup();
        grp.add(localToggle);
        grp.add(remoteToggle);
        grp.add(allToggle);
        if (currentType == Setup.DIFFTYPE_LOCAL) localToggle.setSelected(true);
        else if (currentType == Setup.DIFFTYPE_REMOTE) remoteToggle.setSelected(true);
        else if (currentType == Setup.DIFFTYPE_ALL) allToggle.setSelected(true);
        
        nextAction = new AbstractAction(null, new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/diff-next.png"))) {  // NOI18N
            {
                putValue(Action.SHORT_DESCRIPTION, java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").
                                                   getString("CTL_DiffPanel_Next_Tooltip"));                
            }
            public void actionPerformed(ActionEvent e) {
                onNextButton();
            }
        };
        prevAction = new AbstractAction(null, new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/diff-prev.png"))) { // NOI18N
            {
                putValue(Action.SHORT_DESCRIPTION, java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").
                                                   getString("CTL_DiffPanel_Prev_Tooltip"));                
            }
            public void actionPerformed(ActionEvent e) {
                onPrevButton();
            }
        };
        nextButton.setAction(nextAction);
        prevButton.setAction(prevAction);
    }
    
    private void refreshComponents() {
        DiffView view = setups != null ? setups[currentIndex].getView() : null;
        if (view != null) {
            nextAction.setEnabled(currentIndex < setups.length - 1 || currentDifferenceIndex < view.getDifferenceCount() - 1);
        } else {
            nextAction.setEnabled(false);
        }
        prevAction.setEnabled(currentIndex > 0 || currentDifferenceIndex > 0);
    }
    
    public void addNotify() {
        super.addNotify();
        if (refreshTask != null) {
            CvsVersioningSystem.getInstance().getStatusCache().addVersioningListener(this);
        }
        JComponent parent = (JComponent) getParent();
        parent.getActionMap().put("jumpNext", nextAction);  // NOI18N
        parent.getActionMap().put("jumpPrev", prevAction); // NOI18N
    }

    public void removeNotify() {
        CvsVersioningSystem.getInstance().getStatusCache().removeVersioningListener(this);
        super.removeNotify();
    }
    
    // TODO: reuse this code ... the same pattern is in SynchronizePanel
    public void versioningEvent(VersioningEvent event) {
        if (event.getId() == FileStatusCache.EVENT_FILE_STATUS_CHANGED) {
            if (!affectsView(event)) return;
            if (CvsVersioningSystem.getInstance().getParameter(CvsVersioningSystem.PARAM_BATCH_REFRESH_RUNNING) != null) {
                refreshTask.schedule(1000);
            } else {
                refreshTask.schedule(200);
            }
        } else if (event.getId() == CvsVersioningSystem.EVENT_PARAM_CHANGED) {
            if (event.getParams()[0].equals(CvsVersioningSystem.PARAM_BATCH_REFRESH_RUNNING)) {
                if (CvsVersioningSystem.getInstance().getParameter(CvsVersioningSystem.PARAM_BATCH_REFRESH_RUNNING) == null) {
                    refreshTask.schedule(0);
                }
            }
        }
    }
    
    private boolean affectsView(VersioningEvent event) {
        File file = (File) event.getParams()[0];
        FileInformation oldInfo = (FileInformation) event.getParams()[1];
        FileInformation newInfo = (FileInformation) event.getParams()[2];
        if (oldInfo == null) {
            if ((newInfo.getStatus() & displayStatuses) == 0) return false;
        } else {
            if ((oldInfo.getStatus() & displayStatuses) + (newInfo.getStatus() & displayStatuses) == 0) return false;
        }
        return context.contains(file);
    }
    
    private void setDiffIndex(int idx, int location) {
        currentIndex = idx;
        DiffView view = setups[currentIndex].getView();

        // enable Select in .. action
        TopComponent tc = (TopComponent) getClientProperty(TopComponent.class);
        if (tc != null) {
            Node node = Node.EMPTY;
            File baseFile = setups[currentIndex].getBaseFile();
            if (baseFile != null) {
                FileObject fo = FileUtil.toFileObject(baseFile);
                if (fo != null) {
                    node = new AbstractNode(Children.LEAF, Lookups.singleton(fo));
                }
            }
            tc.setActivatedNodes(new Node[] {node});
        }

        boolean focus = false;
        removeDiffComponent();
        if (view != null) {
            if (location == -1) {
                location = view.getDifferenceCount() - 1;
            }
            if (location >=0 && location < view.getDifferenceCount()) {
                view.setCurrentDifference(location);
            }
            diffView = (JComponent) view.getComponent();
            diffView.getActionMap().put("jumpNext", nextAction);  // NOI18N
            diffView.getActionMap().put("jumpPrev", prevAction);  // NOI18N
            Component toc = WindowManager.getDefault().getRegistry().getActivated();
            if (SwingUtilities.isDescendingFrom(this, toc)) {
                focus = true;
            }
        } else {
            diffView = new SourcesUnavailableComponent(NbBundle.getMessage(DiffMainPanel.class, "MSG_DiffPanel_NoContent"));
        }
        add(diffView);

        currentDifferenceIndex = location;
        if (navigationCombo.isVisible()) {            
            navigationCombo.setSelectedIndex(currentIndex);
        }
        refreshComponents();
        revalidate();
        repaint();
        if (focus) {
            diffView.requestFocusInWindow();
        }
    }

    private void removeDiffComponent() {
        if (diffView != null) {
            remove(diffView);
            diffView = null;
        }
    }

    public void actionPerformed(ActionEvent e) {
        Object source = e.getSource();
        if (source == navigationCombo) onNavigationCombo();
        else if (source == refreshButton) onRefreshButton();
        else if (source == updateButton) onUpdateButton();
        else if (source == commitButton) onCommitButton();
        else if (source == localToggle || source == remoteToggle || source == allToggle) onDiffTypeChanged();
    }

    private void onRefreshButton() {
        LifecycleManager.getDefault().saveAll();
        refreshSetups();
        executeUpdateCommand(true);
    }

    private void onUpdateButton() {
        executeUpdateCommand(false);
    }

    private void onCommitButton() {
        CommitAction.invokeCommit(contextName, context, null);
    }

    // TODO: reuse this code ... the same pattern is in SynchronizePanel
    private void executeUpdateCommand(boolean doNoChanges) {
        if (context == null || context.getRoots().size() == 0) return;
        UpdateCommand cmd = new UpdateCommand();
        String msg;
        if (doNoChanges) {
            msg =  NbBundle.getMessage(DiffMainPanel.class, "BK0001");
        } else {
            msg = NbBundle.getMessage(DiffMainPanel.class, "BK0002");
        }
        cmd.setDisplayName(msg);
        GlobalOptions options = CvsVersioningSystem.createGlobalOptions();
        if (context.getExclusions().size() > 0) {
            options.setExclusions((File[]) context.getExclusions().toArray(new File[context.getExclusions().size()]));
        }
        cmd.setFiles(context.getRootFiles());
        cmd.setBuildDirectories(true);
        cmd.setPruneDirectories(true);
        options.setDoNoChanges(doNoChanges);

        ExecutorGroup group = new ExecutorGroup(msg);
        group.addExecutors(UpdateExecutor.splitCommand(cmd, CvsVersioningSystem.getInstance(), options));
        group.execute();
    }

    /** Next that is driven by visibility. It continues to next not yet visible difference. */
    private void onNextButton() {
        if (navigationCombo.isVisible()) {
            currentIndex = navigationCombo.getSelectedIndex();
        }

        DiffView view = setups[currentIndex].getView();
        if (view != null) {
            int visibleDiffernce = view.getCurrentDifference();
            if (visibleDiffernce < view.getDifferenceCount() - 1) {
                currentDifferenceIndex = Math.max(currentDifferenceIndex, visibleDiffernce);
            }
            if (++currentDifferenceIndex >= view.getDifferenceCount()) {
                if (++currentIndex >= setups.length) {
                    currentIndex--;
                } else {
                    setDiffIndex(currentIndex, 0);
                }
            } else {
                view.setCurrentDifference(currentDifferenceIndex);
            }
        } else {
            if (++currentIndex >= setups.length) currentIndex = 0;
            setDiffIndex(currentIndex, 0);
        }
        refreshComponents();
    }

    private void onPrevButton() {
        DiffView view = setups[currentIndex].getView();
        if (view != null) {
            if (--currentDifferenceIndex < 0) {
                if (--currentIndex < 0) {
                    currentIndex++;
                } else {
                    setDiffIndex(currentIndex, -1);
                }
            } else {
                view.setCurrentDifference(currentDifferenceIndex);
            }
        } else {
            if (--currentIndex < 0) currentIndex = setups.length - 1;
            setDiffIndex(currentIndex, -1);
        }
        refreshComponents();
    }

    private void onNavigationCombo() {
        int idx = navigationCombo.getSelectedIndex();
        if (idx != currentIndex) setDiffIndex(idx, 0);
    }

    /**
     * @return setups, takes into account Local, Remote, All switch
     */
    public Collection getSetups() {
        if (setups == null) {
            return Collections.EMPTY_SET;
        } else {
            return Arrays.asList(setups);
        }
    }

    public String getSetupDisplayName() {
        return contextName;
    }


    private void refreshSetups() {
        if (dpt != null) {
            prepareTask.cancel();
        }

        File [] files;
        switch (currentType) {
        case Setup.DIFFTYPE_LOCAL:
            displayStatuses = FileInformation.STATUS_LOCAL_CHANGE;
            break;
        case Setup.DIFFTYPE_REMOTE:
            displayStatuses = FileInformation.STATUS_REMOTE_CHANGE; 
            break;
        case Setup.DIFFTYPE_ALL:
            displayStatuses = FileInformation.STATUS_LOCAL_CHANGE | FileInformation.STATUS_REMOTE_CHANGE; 
            break;
        default:
            throw new IllegalStateException("Unknown DIFF type:" + currentType); // NOI18N
        }
        files = DiffExecutor.getModifiedFiles(context, displayStatuses);
        
        Setup [] newSetups = new Setup[files.length];
        for (int i = 0; i < newSetups.length; i++) {
            File file = files[i];
            newSetups[i] = new Setup(file, currentType);
        }
        Arrays.sort(newSetups, new SetupsComparator());

        setups = newSetups;
        navigationCombo.setModel(new DefaultComboBoxModel(setups));

        if (setups.length == 0) {
            String noContentLabel;
            switch (currentType) {
            case Setup.DIFFTYPE_LOCAL:
                noContentLabel = NbBundle.getMessage(DiffMainPanel.class, "MSG_DiffPanel_NoLocalChanges");
                break;
            case Setup.DIFFTYPE_REMOTE:
                noContentLabel = NbBundle.getMessage(DiffMainPanel.class, "MSG_DiffPanel_NoRemoteChanges");
                break;
            case Setup.DIFFTYPE_ALL:
                noContentLabel = NbBundle.getMessage(DiffMainPanel.class, "MSG_DiffPanel_NoAllChanges");
                break;
            default:
                throw new IllegalStateException("Unknown DIFF type:" + currentType); // NOI18N
            }
            setups = null;
            navigationCombo.setModel(new DefaultComboBoxModel(new Object [] { noContentLabel }));
            navigationCombo.setEnabled(false);
            navigationCombo.setPreferredSize(null);
            Dimension dim = navigationCombo.getPreferredSize();
            navigationCombo.setPreferredSize(new Dimension(dim.width + 1, dim.height));
            removeDiffComponent();
            diffView = new SourcesUnavailableComponent(noContentLabel);
            add(diffView);
            nextAction.setEnabled(false);
            prevAction.setEnabled(false);
            revalidate();
            repaint();
            executed();
        } else {
            navigationCombo.setEnabled(true);
            navigationCombo.setPreferredSize(null);
            Dimension dim = navigationCombo.getPreferredSize();
            navigationCombo.setPreferredSize(new Dimension(dim.width + 1, dim.height));
            setDiffIndex(0, 0);
            dpt = new DiffPrepareTask(setups);
            prepareTask = RequestProcessor.getDefault().post(dpt);
        }
    }
    
    private void onDiffTypeChanged() {
        if (localToggle.isSelected()) {
            if (currentType == Setup.DIFFTYPE_LOCAL) return;
            currentType = Setup.DIFFTYPE_LOCAL;
        } else if (remoteToggle.isSelected()) {
            if (currentType == Setup.DIFFTYPE_REMOTE) return;
            currentType = Setup.DIFFTYPE_REMOTE;
        } else if (allToggle.isSelected()) {
            if (currentType == Setup.DIFFTYPE_ALL) return;
            currentType = Setup.DIFFTYPE_ALL;
        }
        refreshSetups();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        controlsToolbar = new javax.swing.JToolBar();
        allToggle = new javax.swing.JToggleButton();
        localToggle = new javax.swing.JToggleButton();
        remoteToggle = new javax.swing.JToggleButton();
        navigationCombo = new javax.swing.JComboBox();
        nextButton = new javax.swing.JButton();
        prevButton = new javax.swing.JButton();
        jPanel1 = new javax.swing.JPanel();
        refreshButton = new javax.swing.JButton();
        updateButton = new javax.swing.JButton();
        commitButton = new javax.swing.JButton();

        setLayout(new java.awt.BorderLayout());

        controlsToolbar.setFloatable(false);
        allToggle.setText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_All"));
        allToggle.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_All_Tooltip"));
        controlsToolbar.add(allToggle);

        localToggle.setText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Local"));
        localToggle.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Local_Tooltip"));
        controlsToolbar.add(localToggle);

        remoteToggle.setText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Remote"));
        remoteToggle.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Remote_Tooltip"));
        controlsToolbar.add(remoteToggle);

        controlsToolbar.add(navigationCombo);

        nextButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/diff-next.png")));
        nextButton.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Next_Tooltip"));
        nextButton.setEnabled(false);
        controlsToolbar.add(nextButton);

        prevButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/diff-prev.png")));
        prevButton.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Prev_Tooltip"));
        prevButton.setEnabled(false);
        controlsToolbar.add(prevButton);

        jPanel1.setMaximumSize(new java.awt.Dimension(15, 1));
        jPanel1.setPreferredSize(new java.awt.Dimension(15, 1));
        controlsToolbar.add(jPanel1);

        refreshButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/refresh.png")));
        refreshButton.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Prev_Tooltip"));
        controlsToolbar.add(refreshButton);

        updateButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/update.png")));
        updateButton.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Prev_Tooltip"));
        controlsToolbar.add(updateButton);

        commitButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/org/netbeans/modules/versioning/system/cvss/resources/icons/commit.png")));
        commitButton.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/versioning/system/cvss/ui/actions/diff/Bundle").getString("CTL_DiffPanel_Prev_Tooltip"));
        controlsToolbar.add(commitButton);

        add(controlsToolbar, java.awt.BorderLayout.NORTH);

    }
    // </editor-fold>//GEN-END:initComponents

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JToggleButton allToggle;
    private javax.swing.JButton commitButton;
    private javax.swing.JToolBar controlsToolbar;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JToggleButton localToggle;
    private javax.swing.JComboBox navigationCombo;
    private javax.swing.JButton nextButton;
    private javax.swing.JButton prevButton;
    private javax.swing.JButton refreshButton;
    private javax.swing.JToggleButton remoteToggle;
    private javax.swing.JButton updateButton;
    // End of variables declaration//GEN-END:variables

    
    private class DiffPrepareTask implements Runnable {
        
        private final Setup[] prepareSetups;

        public DiffPrepareTask(Setup [] prepareSetups) {
            this.prepareSetups = prepareSetups;
        }

        public void run() {
            final Diff diff = Diff.getDefault();
            try {
                for (int i = 0; i < prepareSetups.length; i++) {
                    if (prepareSetups != setups) return;
                    try {
                        prepareSetups[i].initSources(group);  // slow network I/O
                        final int fi = i;
                        StreamSource ss1 = prepareSetups[fi].getFirstSource();
                        StreamSource ss2 = prepareSetups[fi].getSecondSource();
                        final DiffView view = diff.createDiff(ss1, ss2);  // possibly executing slow external diff

                        SwingUtilities.invokeLater(new Runnable() {
                            public void run() {
                                prepareSetups[fi].setView(view);
                                if (prepareSetups != setups) {
                                    return;
                                }
                                if (currentIndex == fi) {
                                    setDiffIndex(fi, 0);
                                }
                            }
                        });
                    } catch (IOException e) {
                        ErrorManager.getDefault().notify(e);
                    }
                }
            } finally {
                executed();
            }
        }
    }

    private synchronized void executed() {
        if (group != null) {
            group.executed();
        } else {
            // to warn setGroup that DiffPrepareTask already finished
            executed = true;
        }
    }

    private static class SourcesUnavailableComponent extends JComponent {
        
        public SourcesUnavailableComponent(String message) {
            JLabel label = new JLabel(message);
            setLayout(new BorderLayout());
            label.setHorizontalAlignment(JLabel.CENTER);
            add(label, BorderLayout.CENTER);
        }
    }
    
    /**
     * Hardcoded toolbar layout. It eliminates need
     * for nested panels their look is hardly maintanable
     * accross several look and feels
     * (e.g. strange layouting panel borders on GTK+).
     *
     * <p>It sets authoritatively component height and takes
     * "prefered" width from components itself.
     *
     */
    private class ToolbarLayout implements LayoutManager {

        /** Expected border height */
        private int TOOLBAR_HEIGHT_ADJUSTMENT = 10;

        private int TOOLBAR_SEPARATOR_MIN_WIDTH = 12;

        /** Cached toolbar height */
        private int toolbarHeight = -1;

        /** Guard for above cache. */
        private Dimension parentSize;

        private Set adjusted = new HashSet();

        public void removeLayoutComponent(Component comp) {
        }

        public void layoutContainer(Container parent) {
            Rectangle [] sizes = layout(parent);
            for (int i = 0; i < sizes.length; i++) {
                JComponent comp = (JComponent) parent.getComponent(i);
                if (comp.isVisible()) {
                    comp.setBounds(sizes[i]);
                }
            }
        }

        private Rectangle [] layout(Container parent) {
            Dimension dim = DiffMainPanel.this.getSize();
            Dimension max = parent.getSize();
            int rowOffset = 0;
            int maxHeigth = 0;

            List sizes = new ArrayList();
            int components = parent.getComponentCount();
            int horizont = 0;
            for (int i = 0; i < components; i++) {
                Rectangle rect = new Rectangle();
                JComponent comp = (JComponent) parent.getComponent(i);
                if (comp.isVisible()) {
                    rect.setLocation(horizont, rowOffset);
                    if (comp instanceof AbstractButton) {
                        adjustToobarButton((AbstractButton)comp);
                    } else {
                        adjustToolbarComponentSize((JComponent)comp);
                    }                    
                    Dimension pref = comp.getPreferredSize();
                    int width = pref.width;
                    if (comp instanceof JSeparator && ((dim.height - dim.width) <= 0)) {
                        width = Math.max(width, TOOLBAR_SEPARATOR_MIN_WIDTH);
                    }
    
                    // in column layout use taller toolbar
                    int height = getToolbarHeight(dim) -1;
                    maxHeigth = Math.max(maxHeigth, height);
                    rect.setSize(width, height);  // 1 verySoftBevel compensation
                    horizont += width;
                    if (horizont > max.width) {
                        rowOffset += maxHeigth + 2;
                        horizont = 0;
                        maxHeigth = 0;
                    }
                }
                sizes.add(rect);
            }
            return (Rectangle[]) sizes.toArray(new Rectangle[sizes.size()]);
        }        
        
        public void addLayoutComponent(String name, Component comp) {
        }

        public Dimension minimumLayoutSize(Container parent) {

            // in column layout use taller toolbar
            Dimension dim = DiffMainPanel.this.getSize();
            int height = getToolbarHeight(dim);

            int components = parent.getComponentCount();
            int horizont = 0;
            for (int i = 0; i<components; i++) {
                Component comp = parent.getComponent(i);
                if (comp.isVisible() == false) continue;
                if (comp instanceof AbstractButton) {
                    adjustToobarButton((AbstractButton)comp);
                } else {
                    adjustToolbarComponentSize((JComponent)comp);
                }
                Dimension pref = comp.getPreferredSize();
                int width = pref.width;
                if (comp instanceof JSeparator && ((dim.height - dim.width) <= 0)) {
                    width = Math.max(width, TOOLBAR_SEPARATOR_MIN_WIDTH);
                }
                horizont += width;
            }

            return new Dimension(horizont, height);
        }

        public Dimension preferredLayoutSize(Container parent) {
            Rectangle [] bounds = layout(parent);
            Rectangle union = new Rectangle();
            for (int i = 0; i < bounds.length; i++) {
                union.add(bounds[i]);
            }
            return new Dimension(union.width, union.height);
        }

        /**
         * Computes vertical toolbar components height that can used for layout manager hinting.
         * @return size based on font size and expected border.
         */
        private int getToolbarHeight(Dimension parent) {

            if (parentSize == null || (parentSize.equals(parent) == false)) {
                parentSize = parent;
                toolbarHeight = -1;
            }

            if (toolbarHeight == -1) {
                BufferedImage image = new BufferedImage(1,1,BufferedImage.TYPE_BYTE_GRAY);
                Graphics2D g = image.createGraphics();
                UIDefaults def = UIManager.getLookAndFeelDefaults();

                int height = 0;
                String[] fonts = {"Label.font", "Button.font", "ToggleButton.font"};      // NOI18N
                for (int i=0; i<fonts.length; i++) {
                    Font f = def.getFont(fonts[i]);
                    FontMetrics fm = g.getFontMetrics(f);
                    height = Math.max(height, fm.getHeight());
                }
                toolbarHeight = height + TOOLBAR_HEIGHT_ADJUSTMENT;
            }

            return toolbarHeight;
        }


        /** Toolbar controls must be smaller and should be transparent*/
        private void adjustToobarButton(final AbstractButton button) {

            if (adjusted.contains(button)) return;

            // workaround for Ocean L&F clutter - toolbars use gradient.
            // To make the gradient visible under buttons the content area must not
            // be filled. To support rollover it must be temporarily filled
            if (button instanceof JToggleButton == false) {
                button.setContentAreaFilled(false);
                button.setMargin(new Insets(0, 3, 0, 3));
                button.setBorderPainted(false);
                button.addMouseListener(new MouseAdapter() {
                    public void mouseEntered(MouseEvent e) {
                        button.setContentAreaFilled(true);
                        button.setBorderPainted(true);
                    }

                    public void mouseExited(MouseEvent e) {
                        button.setContentAreaFilled(false);
                        button.setBorderPainted(false);
                    }
                });
            }

            adjustToolbarComponentSize(button);
        }

        private void adjustToolbarComponentSize(JComponent button) {

            if (adjusted.contains(button)) return;

            // as we cannot get the button small enough using the margin and border...
            if (button.getBorder() instanceof CompoundBorder) { // from BasicLookAndFeel
                Dimension pref = button.getPreferredSize();

                // XXX #41827 workaround w2k, that adds eclipsis (...) instead of actual text
                if ("Windows".equals(UIManager.getLookAndFeel().getID())) {  // NOI18N
                    pref.width += 9;
                }
                button.setPreferredSize(pref);
            }

            adjusted.add(button);
        }
    }

    private static class SetupsComparator extends Utils.ByImportanceComparator {

        private FileStatusCache cache;

        public SetupsComparator() {
            cache = CvsVersioningSystem.getInstance().getStatusCache();
        }

        public int compare(Object o1, Object o2) {
            File file1 = ((Setup) o1).getBaseFile();
            File file2 = ((Setup) o2).getBaseFile();
            int cmp = super.compare(cache.getStatus(file1), cache.getStatus(file2));
            if (cmp == 0) {
                return file1.getName().compareToIgnoreCase(file2.getName());
            }
            return cmp;
        }
    }

    private class RefreshViewTask implements Runnable {
        public void run() {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    refreshSetups();
                }
            });
        }
    }
}
