/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.versioning.system.cvss.ui.selectors;

import org.openide.util.NbBundle;
import org.openide.ErrorManager;

import java.io.Serializable;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

/**
 * Describes proxy server that should be used while opening socket
 * connections. The descriptor is static. It's responsibility of
 * an external entity to create proper proxy descriptor for given
 * connection.
 * <p>
 * It's serializable and supports storing 'use system proxy'
 * option that is dynamically reevaluated on deserialization.
 *
 * @author Petr Kuzel
 */
public final class ProxyDescriptor implements Serializable {

    private static final long serialVersionUID = 1L;

    /**
     * This constant descriptor states for TYPE_DIRECT.   
     */
    public static final ProxyDescriptor DIRECT = new ProxyDescriptor(ProxyDescriptor.TYPE_DIRECT, null, -1, null, null);

    /**
     * This constant descriptor states for proxy settings obtained from system.
     */
    public static final ProxyDescriptor SYSTEM = new ProxyDescriptor();

    /**
     * Represents direct internet connection - no proxy.
     */
    public static final int TYPE_DIRECT = 0;

    /**
     * Means that socket connection should be astablished using
     * HTTP proxy. Here does not apply to HttpURLConnectons.
     */
    public static final int TYPE_HTTP = 1;

    /**
     * Means that socket connection should be astablished using
     * SOCKS proxy.
     */
    public static final int TYPE_SOCKS = 2;

    private int type;
    private int port;
    private String host;
    private String userName;
    private String password;
    private String description;
    private final boolean system;
    private transient ProxyDescriptor systemDescriptor;


    ProxyDescriptor(int type, String host, int port, String userName, String password) {
        this(type, host, port, userName, password, null);
    }

    /** Creates fixed decriptor. */
    ProxyDescriptor(int type, String host, int port, String userName, String password, String desc) {
        this.type = type;
        this.host = host;
        this.password = password;
        this.port = port;
        this.userName = userName;
        system = false;
        this.description = desc;
    }

    /**
     * Creates 'use system proxy' descriptor that dynamically delegates
     * to system so actual values may change according to recent system
     * setting.
     * <p>
     * Many systems support automatic proxy configurations, that
     * return proxy depending on connection address. Do not use this
     * contructor in such case.
     */
    ProxyDescriptor() {
        system = true;
    }

    /** One of TYPE_ constants. */
    public int getType() {
        ProxyDescriptor desc = this;
        if (system && systemDescriptor == null) {
            systemDescriptor = ProxySelector.detectSystemProxy();
        }
        if (system) {
            desc = systemDescriptor;
        }
        if (desc == null) return TYPE_DIRECT;
        return desc.type;
    }

    /** @return true for 'use system proxy' dynamic descriptors. */
    public boolean isSystemProxyDescriptor() {
        return system;
    }

    public String getHost() {
        ProxyDescriptor desc = this;
        if (system && systemDescriptor == null) {
            systemDescriptor = ProxySelector.detectSystemProxy();
        }
        if (system) {
            desc = systemDescriptor;
        }
        return desc != null ? desc.host : null;
    }

    /**
     * @return port number, special value <tt>-1</tt> means default, unspecified by user 
     */
    public int getPort() {
        ProxyDescriptor desc = this;
        if (system && systemDescriptor == null) {
            systemDescriptor = ProxySelector.detectSystemProxy();
        }
        if (system) {
            desc = systemDescriptor;
        }
        return desc != null ? desc.port : -1;
    }

    /** null for undefined */
    public String getUserName() {
        ProxyDescriptor desc = this;
        if (system && systemDescriptor == null) {
            systemDescriptor = ProxySelector.detectSystemProxy();
        }
        if (system) {
            desc = systemDescriptor;
        }
        return desc != null ? desc.userName : null;
    }

    /** null for undefined */
    public String getPassword() {
        ProxyDescriptor desc = this;
        if (system && systemDescriptor == null) {
            systemDescriptor = ProxySelector.detectSystemProxy();
        }
        if (system) {
            desc = systemDescriptor;
        }
        return desc != null ? desc.password : null;
    }

    /** null for undefined */
    public String getDescription() {
        ProxyDescriptor desc = this;
        if (system && systemDescriptor == null) {
            systemDescriptor = ProxySelector.detectSystemProxy();
        }
        if (system) {
            desc = systemDescriptor;
        }
        if (desc == null) {
                return NbBundle.getMessage(ProxyDescriptor.class, "BK2002");
            } else {
            return desc.description;
        }
    }

    public boolean needsProxy(String hostName) {
        try {
            InetAddress address = InetAddress.getByName(hostName);
            boolean local =  address.isLoopbackAddress();
            local |= address.isSiteLocalAddress(); // 192.168...
            return local == false;
        } catch (UnknownHostException e) {
            return true;
        }
    }
}
