/*****************************************************************************
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is the CVS Client Library.
 * The Initial Developer of the Original Software is Robert Greig.
 * Portions created by Robert Greig are Copyright (C) 2000.
 * All Rights Reserved.
 *
 * Contributor(s): Robert Greig.
 *****************************************************************************/
package org.netbeans.lib.cvsclient.util;

import java.io.*;

/**
 * Handles the logging of communication to and from the server
 *
 * @author Robert Greig
 * @author Petr Kuzel rewriten to streams
 */
public final class Logger {
    /**
     * The output stream to use to write communication sent to the server
     */
    private static OutputStream outLogStream;

    /**
     * The output stream to use to write communication received from the server
     */
    private static OutputStream inLogStream;

    /**
     * The log files path. If the property is set to the constant "system"
     * then it uses System.err, otherwise it tries to create a file at the
     * specified path
     */
    private static final String LOG_PROPERTY = "cvsClientLog"; // NOI18N

    /**
     * Whether we are logging or not
     */
    private static boolean logging;

    static {
        setLogging(System.getProperty(LOG_PROPERTY));
    }

    public static void setLogging(String logPath) {
        logging = (logPath != null);

        try {
            if (logging) {
                if (logPath.equals("system")) { // NOI18N
                    outLogStream = System.err;
                    inLogStream = System.err;
                }
                else {
                    outLogStream = new BufferedOutputStream(new FileOutputStream(logPath + ".out")); // NOI18N
                    inLogStream = new BufferedOutputStream(new FileOutputStream(logPath + ".in"));  // NOI18N
                }
            }
        }
        catch (IOException e) {
            System.err.println("Unable to create log files: " + e); // NOI18N
            System.err.println("Logging DISABLED"); // NOI18N
            logging = false;
            try {
                if (outLogStream != null) {
                    outLogStream.close();
                }
            }
            catch (IOException ex2) {
                // ignore, if we get one here we really are screwed
            }

            try {
                if (inLogStream != null) {
                    inLogStream.close();
                }
            }
            catch (IOException ex2) {
                // ignore, if we get one here we really are screwed
            }
        }
    }


    /**
     * Log a message received from the server. The message is logged if
     * logging is enabled
     * @param received the data received from the server
     */
    public static void logInput(byte[] received) {
        logInput(received, 0 , received.length);
    }

    /**
     * Log a message received from the server. The message is logged if
     * logging is enabled
     * @param received the data received from the server
     */
    public static void logInput(byte[] received, int offset, int len) {
        if (!logging) {
            return;
        }

        try {
            inLogStream.write(received, offset, len);
            inLogStream.flush();
        }
        catch (IOException ex) {
            System.err.println("Could not write to log file: " + ex); // NOI18N
            System.err.println("Logging DISABLED."); // NOI18N
            logging = false;
        }
    }

    /**
     * Log a character received from the server. The message is logged if
     * logging is enabled
     * @param received the data received from the server
     */
    public static void logInput(char received) {
        if (!logging) {
            return;
        }

        try {
            inLogStream.write(received);
            inLogStream.flush();
        }
        catch (IOException ex) {
            System.err.println("Could not write to log file: " + ex); // NOI18N
            System.err.println("Logging DISABLED."); // NOI18N
            logging = false;
        }
    }

    /**
     * Log a message sent to the server. The message is logged if
     * logging is enabled
     * @param sent the data sent to the server
     */
    public static void logOutput(byte[] sent) {
        if (!logging) {
            return;
        }

        try {
            outLogStream.write(sent);
            outLogStream.flush();
        }
        catch (IOException ex) {
            System.err.println("Could not write to log file: " + ex); // NOI18N
            System.err.println("Logging DISABLED."); // NOI18N
            logging = false;
        }
    }
}

