/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.jdbcimpl;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.*;

import java.util.*;
import java.io.*;

/**
 * JdbcIndex implements the MDR Index interface using JDBC.
 *
 * @author John V. Sichi
 * @version $Id: JdbcIndex.java,v 1.1.106.1 2006/07/01 05:19:20 jtulach Exp $
 */
abstract class JdbcIndex implements Index
{
    protected JdbcStorage storage;
    protected String tableName;
    protected String name;
    protected String keyColName;
    protected String valColName;
    protected Storage.EntryType keyType;
    protected Storage.EntryType valueType;
    protected boolean needSurrogate;

    protected LazyPreparedStatement sqlKeySetIterator;
    protected LazyPreparedStatement sqlKeySetSize;
    protected LazyPreparedStatement sqlKeySetContains;
    protected LazyPreparedStatement sqlInsert;
    protected LazyPreparedStatement sqlDelete;
    protected LazyPreparedStatement sqlFind;

    void init(
        JdbcStorage storage,
        String tableName,
        String name,
        String keyColName,
        String valColName,
        Storage.EntryType keyType,
        Storage.EntryType valueType,
        boolean needSurrogate)
    {
        this.storage = storage;
        this.tableName = tableName;
        this.name = name;
        this.keyColName = keyColName;
        this.valColName = valColName;
        this.keyType = keyType;
        this.valueType = valueType;
        this.needSurrogate = needSurrogate;
        defineSql();
    }

    protected void defineSql()
    {
        if (isKeyUnique()) {
            sqlKeySetIterator = new LazyPreparedStatement(
                "select " + keyColName + " from " + tableName);
            sqlKeySetSize = new LazyPreparedStatement(
                "select count(*) from " + tableName);
        } else {
            sqlKeySetIterator = new LazyPreparedStatement(
                "select distinct " + keyColName
                + " from " + tableName);
            sqlKeySetSize = new LazyPreparedStatement(
                "select count(distinct " + keyColName
                + ") from " + tableName);
        }
        
        sqlKeySetContains = new LazyPreparedStatement(
            "select count(*) from " + tableName + " where "
            + keyColName + " = ?");
        
        sqlInsert = new LazyPreparedStatement(
            "insert into " + tableName
            + " values(?,?"
            + (needSurrogate ? ",?" : "")
            + ")");
        
        sqlDelete = new LazyPreparedStatement(
            "delete from " + tableName
            + " where " + keyColName + " = ?");
        
        sqlFind = new LazyPreparedStatement(
            "select " + valColName + " from " + tableName
            + " where " + keyColName + " = ?");
    }

    protected boolean isKeyUnique()
    {
        return false;
    }

    // implement Index
    public String getName() throws StorageException
    {
        return name;
    }

    // implement Index
    public Storage.EntryType getValueType() throws StorageException
    {
        return valueType;
    }

    // implement Index
    public Storage.EntryType getKeyType() throws StorageException
    {
        return keyType;
    }

    // implement Index
    public Set keySet() throws StorageException
    {
        return new JdbcSet(
            storage,
            getKeyType(),
            sqlKeySetIterator,sqlKeySetSize,sqlKeySetContains);
    }

    // implement Index
    public void add(Object key, Object value) throws StorageException
    {
        addImpl(key,value);
    }

    protected void addImpl(Object key, Object value) throws StorageException
    {
        Object [] args;
        if (needSurrogate) {
            args = new Object[]{key,value,new Long(storage.getSerialNumber())};
        } else {
            args = new Object[]{key,value};
        }
        storage.executeUpdate(sqlInsert,args);
    }
    
    // implement Index
    public boolean remove(Object key) throws StorageException
    {
        return removeImpl(key);
    }

    protected boolean removeImpl(Object key) throws StorageException
    {
        return storage.executeUpdate(sqlDelete,new Object[]{key}) > 0;
    }
}

// End JdbcIndex.java
