/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.handlers;

import org.netbeans.api.mdr.MDRObject;
import org.netbeans.api.mdr.MDRepository;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.handlers.gen.HandlerGenerator;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.StorageBadRequestException;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.storagemodel.MdrStorage;
import org.netbeans.mdr.storagemodel.StorableBaseObject;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.mdr.storagemodel.Transient;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.util.ImplClass;
import org.netbeans.mdr.util.Logger;
import javax.jmi.reflect.InvalidObjectException;
import javax.jmi.reflect.RefBaseObject;
import javax.jmi.reflect.RefObject;
import javax.jmi.reflect.RefPackage;
import java.util.*;

/** Handles RefBaseObjectCalls
 *
 * @author Petr Hrebejk, Martin Matula
 * @version 
 */
public abstract class BaseObjectHandler extends ImplClass implements MDRObject/*, OclAny*/ {
    
    private static MDRClassLoader defaultLoader = null;
    private static ClassLoaderProvider provider = null;
    
    static Map deletedInstances = new WeakHashMap();
    
    /* --------------------------------------------------------------------- */
    /* -- Static setters/getters ------------------------------------------- */
    /* --------------------------------------------------------------------- */
    
    public static synchronized void setClassLoaderProvider(ClassLoaderProvider provider) {
        BaseObjectHandler.provider = provider;
    }
    
    public static synchronized MDRClassLoader getDefaultClassLoader() {
        if (defaultLoader == null) {
            defaultLoader = new MDRClassLoader(provider);
        }
//	Logger.getDefault().log(defaultLoader.toString() + " : " + defaultLoader.getClass().getName());
        return defaultLoader;   
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Private attributes ----------------------------------------------- */
    /* --------------------------------------------------------------------- */

//    /** The object in strorage to delegate to
//    */
//    private final StorableBaseObject storableDelegate;
    private MOFID mofId;
    private final MdrStorage mdrStorage;
    // The storable hard reference is used only by transient objects
    // to prevent garbage collection.
    // For non transient objects it is null during the whole life cycle.
    private StorableBaseObject storable;
    
    /* --------------------------------------------------------------------- */
    /* -- Static helper methods -------------------------------------------- */
    /* --------------------------------------------------------------------- */

    /**
     * Returns the handler class for JMI interface <code>ifc</code> and storage object <code>s</code>.
     *
     * @param ifc JMI interface to be implemented by the class
     * @param s storage object to be wrapped by an instances of the class
     * @return class implementing <code>ifc</code>
     */
    public static Class getHandlerClass(Class ifc, StorableBaseObject s) throws IllegalArgumentException {
        MDRClassLoader loader = getDefaultClassLoader();
        
        /* check if the loader may load the interface */
        check(loader, ifc);
        /* try to load from cache */
        Map cache = getLoaderCache(loader);
        String className = getName(ifc);
        Class result = getFromCache(cache, ifc, className);
        
        if (result == null) {
            try {
                byte[] handlerClassFile;
                StorableObject metaObject = s.getMetaObject();
                if (s instanceof StorableObject) {
                    handlerClassFile = metaObject.getInstanceClassFile();
                } else {
                    handlerClassFile = metaObject.getClassFile();
                }
                
                if (handlerClassFile == null) {
                    /* generate and define handler class */
                    handlerClassFile = HandlerGenerator.generateHandler(className, ifc, s);
                    if (s instanceof StorableObject) {
                        metaObject.setInstanceClassFile(handlerClassFile);
                    } else {
                        metaObject.setClassFile(handlerClassFile);
                    }
                }
                
                /* [XXX] Allow the use of a system property org.netbeans.mdr.byteCodeDir, write class
                 * files to that directory, if the system property is set.
                 */
//                try {
//                    java.io.FileOutputStream file = new java.io.FileOutputStream("/mnt/work/mdrclasses/" + className.substring(className.lastIndexOf('.') + 1) + ".class");
//                    file.write(handlerClassFile);
//                    file.close();
//                } catch (Exception e) {
//                    Logger.getDefault().notify(Logger.INFORMATIONAL, e);
//                }
                
                result = loader.defineClass(className, handlerClassFile);
            } catch (StorageException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            } finally {
                releaseCache(cache, result, className);
            }
        }
        
        return result;
    }

    /**
     * Returns the JMI interface to be implemented by the handler of <code>s</code>.
     *
     * @param s a storage object
     * @return the JMI interface for <code>s</code>
     */
    public static Class resolveClass(StorableBaseObject s) {
        try {
            StorableObject metaObject = s.getMetaObject();
            return getDefaultClassLoader().resolveInterface(metaObject, !(s instanceof StorableObject));
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    /**
     * Returns interface <code>ifcName</code>. The interface is not generated on the
     * fly, but must be available for loading by the default class-loader
     * (see {@link BaseObjectHandler#getDefaultClassLoader()}).
     *
     * @param ifcName name of the interface to be loaded
     * @return the interface named <code>ifcName</code>
     */
    public static Class resolveInterface(String ifcName) throws ClassNotFoundException {
        try {
            return Class.forName(ifcName, true, BaseObjectHandler.getDefaultClassLoader());
        } catch (RuntimeException e) {
            Logger.getDefault().annotate(e, "ClassLoader: " + BaseObjectHandler.getDefaultClassLoader());
            Logger.getDefault().annotate(e, "ClassName: " + ifcName);
            throw e;
        } catch (Error e) {
            Logger.getDefault().annotate(e, "ClassLoader: " + BaseObjectHandler.getDefaultClassLoader());
            Logger.getDefault().annotate(e, "ClassName: " + ifcName);
            throw e;
	}
    }
        
    /**
     * Loads the class <code>implName</code> from the default class loader.
     *
     * @param implName class name
     * @return the class
     */
    public static Class resolveImplementation(String implName) throws ClassNotFoundException {
        try {
            return Class.forName(implName, true, BaseObjectHandler.getDefaultClassLoader());
        } catch (ClassNotFoundException e) {
            // Logger.getDefault().annotate(e, "Implementation of derived element not found. ClassName: " + implName);
            throw e;
        } catch (Error e) {
            Logger.getDefault().annotate(e, "ClassLoader: " + BaseObjectHandler.getDefaultClassLoader());
            Logger.getDefault().annotate(e, "ClassName: " + implName);
            throw e;
	} catch (RuntimeException e) {
            Logger.getDefault().annotate(e, "ClassLoader: " + BaseObjectHandler.getDefaultClassLoader());
            Logger.getDefault().annotate(e, "ClassName: " + implName);
            throw e;
        }
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Constructor(s) --------------------------------------------------- */
    /* --------------------------------------------------------------------- */

    /** Creates new RefBaseObjectHandler */
    protected BaseObjectHandler(StorableBaseObject storable) {
        _setDelegate(storable);
        this.mdrStorage = storable.getMdrStorage();
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Transaction related methods -------------------------------------- */
    /* --------------------------------------------------------------------- */
    
    protected final void _lock() {
        _lock(false);
    }
    
    protected final void _lock(boolean write) {
        _getRepository().beginTrans(write);
    }
    
    protected final void _unlock() {
        _getRepository().endTrans();
    }

    protected final void _unlock(boolean fail) {
        _getRepository().endTrans(fail);
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Implements org.netbeans.api.mdr.MDRObject ------------------------ */
    /* --------------------------------------------------------------------- */
    
    public final MDRepository repository() {
        return _getRepository();
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Extends java.lang.Object ----------------------------------------- */
    /* --------------------------------------------------------------------- */

    /** Tests this object with other object for identity.
     *  The object is identical if it is of RefBaseObject type
     *  and its mofid equals to this object mofid.
     *  This equals method and hashCode method are the only two
     *  methods guaranteed to work after the object was deleted.
     *  This allows developer to safely remove the deleted object from a container.
     *  @param obj other object
     *  @return true if objects are identical
     */ 
    public final boolean equals(Object obj) {
        if (obj instanceof BaseObjectHandler) {
            return this == obj;
        } else return (obj instanceof RefBaseObject) && ((RefBaseObject) obj).refMofId().equals(refMofId());
    }

    public String toString() {
        String className = getClass().getName();
        className = className.substring( className.lastIndexOf( "." ) + 1 );
        String metaId;
        try {
            metaId = _getDelegate().getMetaObject().getMofId().toString();
        } catch (Exception e) {
            metaId = "(not available)";
        }
        String outP;

        try {
            outP = _getDelegate().getOutermostPackage().getMofId().toString();
        } catch (Exception e) {
            outP = "(not available)";
        }
        
        return className + "  ID: " + refMofId() + "  MID: " + metaId  + "  OPCKG: " + outP;
    }

    /** Returns hash code of this object.
     *  This hashCode method and equals method are the only two
     *  methods guaranteed to work after the object was deleted.
     *  This allows developer to safely remove the deleted object from a container.
     *  @return int the hash code
     */
    public final int hashCode() {
        return this.mofId.hashCode();
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Implements javax.jmi.reflect.RefBaseObject ----------------------- */
    /* --------------------------------------------------------------------- */

    public final RefObject refMetaObject() {
        _lock(false);
        try {
            return (RefObject) _getRepository().getHandler(_getDelegate().getMetaObject());
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        } finally {
            _unlock();
        }
    }

    public final RefPackage refImmediatePackage() {
        _lock(false);
        try {
            return (RefPackage) _getRepository().getHandler(_getDelegate().getImmediatePackage());
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        } finally {
            _unlock();
        }
    }

    public final RefPackage refOutermostPackage() {
        _lock(false);
        try {
            StorableBaseObject pkg = _getDelegate().getOutermostPackage();
            if (pkg.getMofId().equals(_getDelegate().getMofId())) {
                return (RefPackage) this;
            } else {
                return (RefPackage) _getRepository().getHandler(pkg);
            }
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        } finally {
            _unlock();
        }
    }

    public final String refMofId() {
        return mofId.toString();
    }
    
    public final Collection refVerifyConstraints(boolean deepVerify) {
        _lock(false);
        try {
            if (deepVerify) return _recursiveVerify(new ArrayList(), new HashSet());
            else return _verify(new ArrayList());
        } finally {
            _unlock();
        }
    }
    
    /* --------------------------------------------------------------------- */
    /* -- --------------------------------------------------- */
    /* --------------------------------------------------------------------- */

    public final StorableBaseObject _getDelegate() {
        try {
            StorableBaseObject result;
            if (this.storable != null)
                result = this.storable;                   // Transient object
            else
                result = mdrStorage.getObject(mofId);     // Persistent object
            if (result == null) {
                Exception exc = (Exception)deletedInstances.get(this);
                if (exc != null) {
                    Logger.getDefault().notify(Logger.INFORMATIONAL, exc);
                }
                throw new InvalidObjectException(null, "Object with MOFID " + mofId + " no longer exists, class: " + getClass().getName());
            }
            return result;
        } catch (StorageBadRequestException e) {
            Exception exc = (Exception)deletedInstances.get(this);
            if (exc != null) {
                Logger.getDefault().notify(Logger.INFORMATIONAL, exc);
            }
            throw new InvalidObjectException(null, "Object with MOFID " + mofId + " no longer exists, class: " + getClass().getName());
        } catch (StorageException e) {
            throw new DebugException(e.toString());
        }
        
        //return storableDelegate;
    }
    
    public final MOFID _getMofId() {
        return mofId;
    }

    protected final void _setDelegate(StorableBaseObject storable) {
        boolean register = false;
        MOFID newMofId = storable.getMofId();
        if (this.mofId != null && !this.mofId.equals(newMofId)) {
            _getRepository().removeHandler(this.mofId);
            register = true;
        }
        this.mofId = newMofId;
        if (storable instanceof StorableObject && storable instanceof Transient) {
            this.storable = storable;
        } else {
            this.storable = null;
        }
        if (register) {
            _getRepository().addHandler(this);
        }
    }

    protected final MdrStorage _getMdrStorage() {
        return mdrStorage;
    }
    
    protected final NBMDRepositoryImpl _getRepository() {
        return mdrStorage.getRepository();
    }
    
    /* --------------------------------------------------------------------- */
    /* -- Methods to be overridden by sub-classes -------------------------- */
    /* --------------------------------------------------------------------- */

    protected abstract Collection _recursiveVerify(Collection violations, Set visited);
    protected abstract Collection _verify(Collection violations);
}
