/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.handlers.gen;

import java.util.Locale;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.storagemodel.StorableObject;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.util.Logger;
import org.netbeans.mdr.util.MOFConstants;
import javax.jmi.model.MultiplicityType;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

/**
 *
 * @author  Martin Matula, Dusan Balek
 * @version
 */
public class TagSupport extends Object {
    public static final String TAGID_PACKAGE_PREFIX = "javax.jmi.packagePrefix"; //NOI18N
    public static final String TAGID_SUBSTITUTE_NAME = "javax.jmi.substituteName"; //NOI18N
    public static final String TAGID_IMPL_PACKAGE_PREFIX = "org.netbeans.implPackagePrefix"; //NOI18N
    
    public static final int ASSOCIATION = 0;
    public static final int INSTANCE = 1;
    public static final int CLASS = 2;
    public static final int PACKAGE = 3;
    
    private static final Hashtable valueCache = new Hashtable();
    
    private TagSupport() {
    }
    
    /** Returns the full class-name of the JMI interface to be implemented
     *   by a handler for repository object <code>type</code>.
     *
     * @param type a repository object (not: meta-object) looking for a handler
     * @return fully qualified class-name of the JMI interface to be
     *     implemented by a handler for <code>type</code>
     **/
    public static String getTypeFullName(StorableObject type) {
        return getTypeFullName(type, getSubstName(type));
    }
    
    public static String getTypeFullName(StorableObject type, String substName) {
        return getTypePrefix(type, new StringBuffer(50)).append('.').append(substName).toString();
    }
    
    public static String getTypeFullName(StorableObject storable, int type) {
        return getTypePrefix(storable, new StringBuffer(50)).append('.').append(getSubstName(storable)).append((type == CLASS ? "Class" : (type == PACKAGE ? "Package" : ""))).toString(); //NOI18N
    }
    
    /** Returns the full class-name of an optional implementation class
     *  for repository object <code>type</code>. If a class of the returned
     *  name is available in the class-path it is used as super-class for the
     * generated handler class. The returned class-name is:
     *
     * <pre>mdrjmiimpl.<i>package</i>.<i>name</i>Impl</pre>
     * where
     * <pre><i>package</i>.<i>name</i>.</pre>
     * is the fully qualified name of the JMI interface.
     *
     * @param storable a repository object (not: meta-object) looking for a handler
     * @param type kind of repository object
     * @return fully qualified class-name of optional implementation class
     *  */
    public static String getImplFullName(StorableObject storable, int type) {
        return getImplPrefix(storable, new StringBuffer(50)).append('.').append(getSubstName(storable)).append(type == CLASS ? "Class" : (type == PACKAGE ? "Package" : "")).append("Impl").toString(); //NOI18N
    }
    
    public static StorableObject getTag(StorableObject storable, String tagID) {
        StorableObject tag = null;
        
        try {
            Collection tags = (Collection) storable.getImmediatePackage().getAssociation(MOFConstants.SH_MODEL_ATTACHES_TO).queryObjects(MOFConstants.SH_MODEL_ATTACHES_TO_MODEL_ELEMENT, storable.getMofId());
            StorableObject temp;
            for (Iterator it = tags.iterator(); it.hasNext();) {
                temp = (StorableObject) it.next();
                if (tagID.equals(temp.getAttribute(MOFConstants.SH_MODEL_TAG_TAG_ID))) {
                    tag = temp;
                    break;
                }
            }
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
        
        return tag;
    }
    
    public static Collection getTagValues(StorableObject storable, String tagID) {
        StorableObject tag = getTag(storable, tagID);
        
        try {
            if (tag == null) {
                return null;
            } else {
                return (Collection) tag.getAttribute(MOFConstants.SH_MODEL_TAG_VALUES);
            }
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    public static String getTagValue(StorableObject storable, String tagID) {
        if (storable == null)
            return null;
        Object tagKey = new CacheKey(storable.getMofId(), tagID);
        String value = (String) valueCache.get(tagKey);
        
        if (value == null) {
            try {
                Collection values = getTagValues(storable, tagID);
                
                if (values != null && values.size() > 0) {
                    value = (String) values.iterator().next();
                    valueCache.put(tagKey, value);
                }
            } catch (Exception e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            }
        }
        
        return value;
    }
    
    public static String getTagValue(StorableObject storable, String tagID, String defaultValue) {
        String result = getTagValue(storable, tagID);
        if (result == null) {
            result = defaultValue;
        }
        
        return result;
    }
    
    public static String getDataTypeName(StorableObject type) {
        try {
            String typeName = (String) type.getMetaObject().getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME);
            if (typeName.equals(MOFConstants.SH_MODEL_PRIMITIVE_TYPE))
                return "java.lang." + (String) type.getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME); //NOI18N
            else if (typeName.equals(MOFConstants.SH_MODEL_ALIAS_TYPE))
                return getDataTypeName((StorableObject) type.getReference(MOFConstants.SH_MODEL_TYPED_ELEMENT_TYPE));
            else if (typeName.equals(MOFConstants.SH_MODEL_COLLECTION_TYPE)) {
                MultiplicityType multiplicity = (MultiplicityType) type.getAttribute(MOFConstants.SH_MODEL_STRUCTURAL_FEATURE_MULTIPLICITY);
                if (multiplicity.isOrdered())
                    return "java.util.List"; //NOI18N
                else
                    return "java.util.Collection"; //NOI18N
            } else
                return getTypeFullName(type);
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
        
    }
    
    public static String getSubstName(StorableObject storable) {
        return getSubstName(storable, null, null);
    }
    
    public static String getSubstName(StorableObject storable, String name, String meta) {
        String result = getTagValue(storable, TAGID_SUBSTITUTE_NAME);
        try {
            if (result == null) {
                if (name == null)
                    result = (String) storable.getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME);
                else
                    result = name;
            }
            if (meta == null)
                meta = (String) storable.getMetaObject().getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME);
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
        if (MOFConstants.SH_MODEL_CONSTANT.equals(meta))
            result = mapName(result, true, false);
        else {
            boolean flag = MOFConstants.SH_MODEL_CLASS.equals(meta) ||
            MOFConstants.SH_MODEL_PACKAGE.equals(meta) ||
            MOFConstants.SH_MODEL_ASSOCIATION.equals(meta) ||
            MOFConstants.SH_MODEL_EXCEPTION.equals(meta) ||
            MOFConstants.SH_MODEL_IMPORT.equals(meta) ||
            MOFConstants.SH_MODEL_STRUCTURE_TYPE.equals(meta) ||
            MOFConstants.SH_MODEL_ENUMERATION_TYPE.equals(meta) ||
            MOFConstants.SH_MODEL_COLLECTION_TYPE.equals(meta);
            result = mapName(result, false, !flag);
            if (MOFConstants.SH_MODEL_EXCEPTION.equals(meta) && !result.endsWith("Exception")) //NOI18N
                result = result + "Exception"; //NOI18N
        }
        return result;
    }
    
    private static String mapName(String name, boolean toLiteral, boolean firstLower) {
        StringBuffer buffer = new StringBuffer(32);
        boolean wordRead = false;
        boolean lowerCharDetected = false;
        for (int x = 0; x < name.length(); x++) {
            char c = name.charAt(x);
            if (c == '-' || c == '_' || Character.isWhitespace(c)) {
                if (wordRead) {
                    // first whitespace following end of word reached
                    if (toLiteral)
                        buffer.append('_');
                    wordRead = false;
                    lowerCharDetected = false;
                }
            } else {
                if (lowerCharDetected && Character.isUpperCase(c)) {
                    // a next word started ...
                    if (toLiteral)
                        buffer.append('_');
                    wordRead = false;
                    lowerCharDetected = false;
                } else if (Character.isLowerCase(c))
                    lowerCharDetected = true;
                if (!wordRead || toLiteral)
                    buffer.append(Character.toUpperCase(c));
                else
                    buffer.append(Character.toLowerCase(c));
                wordRead = true;
            } // if
        } // for
        if (buffer.length() > 0) {
            if (toLiteral && !wordRead)
                buffer.deleteCharAt(buffer.length() - 1);
            if (firstLower)
                buffer.replace(0, 1, new String(new char [] {(Character.toLowerCase(buffer.charAt(0)))}));
        }
        return buffer.toString();
    }
    
    public static String mapEnumLiteral(String name) {
        return mapName(name, true, false);
    }
    
    private static StringBuffer getImplPrefix(StorableObject storable, StringBuffer sb) {
        try {
            StorableObject pckg = storable;

            while (!MOFConstants.SH_MODEL_PACKAGE.equals(pckg.getMetaObject().getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME))) {
                pckg = (StorableObject) pckg.getReference(MOFConstants.SH_MODEL_MODEL_ELEMENT_CONTAINER);
            }
            
            StorableObject container = (StorableObject) pckg.getReference(MOFConstants.SH_MODEL_MODEL_ELEMENT_CONTAINER);
            if (container == null) {
                String result = getTagValue(pckg, TAGID_IMPL_PACKAGE_PREFIX);
                if (result == null) {
                    result = getTagValue(pckg, TAGID_PACKAGE_PREFIX);
                    if (result == null) {
                        sb.append("impl"); //NOI18N
                    } else {
                        result = result.toLowerCase(Locale.US);
                        if ("javax.jmi".equals(result)) { //NOI18N
                            sb.append("org.netbeans.jmiimpl.mof"); //NOI18N
                        } else if (result.startsWith("org.netbeans.jmi.")) { //NOI18N
                            sb.append("org.netbeans.jmiimpl").append(result.substring(16)); //NOI18N
                        } else if (result.startsWith("org.omg")) { //NOI18N
                            if (result.length() == 7) {
                                sb.append("org.netbeans.jmiimpl.omg"); //NOI18N
                            } else if (result.charAt(7) == '.') {
                                sb.append("org.netbeans.jmiimpl.omg").append(result.substring(7)); //NOI18N
                            } else {
                                sb.append(result).append(".impl"); //NOI18N
                            }
                        } else {
                            sb.append(result).append(".impl"); //NOI18N
                        }
                    }
                } else {
                    sb.append(result.toLowerCase(Locale.US));
                }
            } else {
                getImplPrefix(container, sb);
            }

            String packageName = getTagValue(pckg, TAGID_SUBSTITUTE_NAME);
            if (packageName == null) {
                packageName = mapName((String) pckg.getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME), false, true);
            }

            sb.append('.').append(packageName.toLowerCase(Locale.US));
            
            return sb;
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    public static String getTypePrefix(StorableObject storable) {
        return getTypePrefix(storable, new StringBuffer(50)).toString();
    }
    
    public static StringBuffer getTypePrefix(StorableObject storable, StringBuffer sb) {
        try {
            StorableObject pckg = storable;

            while (!MOFConstants.SH_MODEL_PACKAGE.equals(pckg.getMetaObject().getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME))) {
                pckg = (StorableObject) pckg.getReference(MOFConstants.SH_MODEL_MODEL_ELEMENT_CONTAINER);
            }
            
            StorableObject container = (StorableObject) pckg.getReference(MOFConstants.SH_MODEL_MODEL_ELEMENT_CONTAINER);
            if (container == null) {
                String result = getTagValue(pckg, TAGID_PACKAGE_PREFIX);
                if (result != null) {
                    sb.append(result.toLowerCase(Locale.US)).append('.');
                }
            } else {
                getTypePrefix(container, sb).append('.');
            }

            String packageName = getTagValue(pckg, TAGID_SUBSTITUTE_NAME);
            if (packageName == null) {
                packageName = mapName((String) pckg.getAttribute(MOFConstants.SH_MODEL_MODEL_ELEMENT_NAME), false, true);
            }
            
            return sb.append(packageName.toLowerCase(Locale.US));
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    private static class CacheKey {
        private final MOFID mofId;
        private final String tagId;
        
        public CacheKey(MOFID mofId, String tagId) {
            if (mofId == null) throw new IllegalArgumentException();
            this.mofId = mofId;
            this.tagId = tagId == null ? "null" : tagId;
        }

        public boolean equals(Object o) {
            return (o instanceof CacheKey) && ((CacheKey) o).mofId.equals(mofId) && ((CacheKey) o).tagId.equals(tagId);
        }
        
        public int hashCode() {
            return mofId.hashCode() * 31 + tagId.hashCode();
        }
    }
}
