/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreeindex;

import org.netbeans.mdr.persistence.*;
import java.util.*;
import java.io.*;

/**
 * Iterator over a BtreeCollection, which is the set of values contained in
 * a SinglevaluedBtree.
 *
 * @author	Dana Bergen
 * @version	1.0
 */
public class BtreeIterator extends Object implements Iterator {

    protected Btree btree;
    protected SearchResult current;
    protected BtreePageSource pageSource;
    protected int modCount;

    BtreeIterator(Btree btree) {
        try {
	    this.btree = btree;
	    pageSource = btree.pageSource;
	    btree.beginRead();
	    modCount = btree.modCount;
	    current = btree.getFirst();
	    if (current.entryNum >= 0) {
	        current.entryNum--;
	    }
	} catch (StorageException e) {
	    throw new RuntimeStorageException(e);
	} finally {
	    btree.endRead();
	}
    }

    /**
     * Tests whether there are any more elements to return
     *
     * @return	true if a call to next() would succeed
     */
    public boolean hasNext() {

	try {
	    btree.beginRead();
	    checkModCount();
	    if (!BtreePage.hasNext(null, current)) {
	        pageSource.unpinPage(current.page);
		current.page = null;
		return false;
	    } else {
	        return true;
	    }
	} catch (StorageException e) {
	    throw new RuntimeStorageException(e);
	} finally {
	    btree.endRead();
	}
    }

    /**
     * Gets the next value in the btree.  If this is on a different page
     * from the previous value, the old page will be unpinned.
     * If there are no more records, the last page will be unpinned.
     *
     * @return	The next value in the btree
     *
     * @exception	NoSuchElementException	If there was any error or if
     * 						there are no more records
     */
    public Object next() throws NoSuchElementException {

	BtreePage	oldPage;
	
	if (current.page == null) {
	    throw new NoSuchElementException();
	}
	try {
	    btree.beginRead();
	    checkModCount();
	    oldPage = current.page;
	    BtreePage.getNext(null, current);
	    if (current.page != oldPage) {
	        pageSource.unpinPage(oldPage);
	    }
	    if (current.matched) {
		return getCurrentItem();
	    } else {
	        throw new NoSuchElementException();
	    }
	} catch (StorageException e) {
	    throw new RuntimeStorageException(e);
	} finally {
	    btree.endRead();
	}
    }

    /**
     * Returns the data value at the current position
     */
    Object getCurrentItem() throws StorageException {
	return btree.dataInfo.fromBuffer(
				current.page.getData(current.entryNum));
    }

    void checkModCount() {
	if (btree.modCount > modCount) {
	    throw new ConcurrentModificationException("Index " + btree.getName()
		    + " has been modified since iterator was created.");
	}
    }

    /*
     * Unpin the last page.
     */
    protected void finalize() {
	if (current.page != null) {
            pageSource.unpinPage(current.page);
	    current.page = null;
	}
    }

    /**
     * This is not supported.
     *
     * @exception	UnsupportedOperationException	Always thrown.
     */
    public void remove() throws UnsupportedOperationException {
        throw new UnsupportedOperationException();
    }
}
