/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreeindex;

import org.netbeans.mdr.persistence.*;
import java.util.*;
import java.io.*;

/**
 * Read-only Set view of the distinct keys contained in a Btree.
 *
 * @author	Dana Bergen
 * @version	1.0
 */
public class BtreeKeySet extends AbstractSet {

    private Btree btree;

    BtreeKeySet(Btree btree) {
        this.btree = btree;
    }

    public int size() {
	int count = 0;
	BtreeKeyIterator i = new BtreeKeyIterator();
	while (i.hasNext()) {
	    i.nextKey();
	    count++;
	}
	return count;
    }
    
    public boolean isEmpty() {
        return !iterator().hasNext();
    }

    public Iterator iterator() {
        return new BtreeKeyIterator();
    }

    /**
     * Iterator over a BtreeKeySet
     */
    public class BtreeKeyIterator extends Object implements Iterator {

	protected SearchResult current;
	protected BtreePageSource pageSource;
	protected int modCount;

	BtreeKeyIterator() {
	    try {
	        btree.beginRead();
		modCount = btree.modCount;
		pageSource = btree.pageSource;
		current = btree.getFirst();
		if (current.entryNum >= 0) {
		    current.entryNum--;
		}
	    } catch (StorageException e) {
		    throw new RuntimeStorageException(e);
	    } finally {
	        btree.endRead();
	    }
	}

	/**
	 * Tests whether there are any more elements to return
	 *
	 * @return	true if a call to next() would succeed
	 */
	public boolean hasNext() {

	    boolean result;

	    try {
	        btree.beginRead();
		checkModCount();
		if (!BtreePage.hasNext(null, current)) {
		    pageSource.unpinPage(current.page);
		    current.page = null;
		    result = false;
		} else {
		    result = true;
		}
	    } catch (StorageException se) {
		throw new RuntimeStorageException(se);
	    } finally {
	        btree.endRead();
	    }
	    return result;
	}

	/**
	 * Gets the next distinct key in the btree.  If this is on a different 
	 * page from the previous value, the old page will be unpinned.
	 *
	 * @return	The next distinct key in the btree
	 *
	 * @exception	NoSuchElementException	If there was any error or if
	 * 					there are no more records
	 */
	public Object next() throws NoSuchElementException {

	    byte[] key = nextKey();
	    return btree.keyInfo.fromBuffer(key);
	}

	private byte[] nextKey() throws NoSuchElementException {

	    BtreePage	oldPage;
	    byte[]		key;
	    
	    if (current.page == null) {
		throw new NoSuchElementException();
	    }
	    try {
	        btree.beginRead();
		checkModCount();
		oldPage = current.page;
		BtreePage.getNext(null, current);
		if (current.page != oldPage) {
		    pageSource.unpinPage(oldPage);
		}
		if (current.matched) {
		    key = current.page.getKey(current.entryNum);
		} else {
		    throw new NoSuchElementException();
		}
		/* position ourselves at the next distinct key */
		while (BtreePage.hasNext(key, current)) {
		    oldPage = current.page;
		    BtreePage.getNext(null, current);
		    if (current.page != oldPage) {
			pageSource.unpinPage(oldPage);
		    }
		}
	    } catch (StorageException e) {
		throw new RuntimeStorageException(e);
	    } finally {
	        btree.endRead();
	    }
	    return key;
	}

	private void checkModCount() {
	    if (btree.modCount > modCount) {
	        throw new ConcurrentModificationException("Index " 
				+ btree.getName()
				+ " has been modified since iterator was created.");
	    }
	}

	/*
	 * Unpin the last page.
	 */
	protected void finalize() {
	    if (current.page != null) {
		pageSource.unpinPage(current.page);
		current.page = null;
	    }
	}

	/**
	 * This is not supported.
	 *
	 * @exception	UnsupportedOperationException	Always thrown.
	 */
	public void remove() throws UnsupportedOperationException {
	    throw new UnsupportedOperationException();
	}
    }
}
