/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreeindex;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.MapEntryImpl;
import java.lang.ref.WeakReference;
import java.util.*;
import java.text.*;

/**
 * Btree implementation of MultivaluedIndex interface.
 *
 * @author	Dana Bergen
 * @version	1.0
 */
public class MultivaluedBtree extends Btree implements MultivaluedIndex {

    // (weak) cache storing BtreeListByKey instances
    private WeakHashMap listsByKey_cache = new WeakHashMap ();
    
    public MultivaluedBtree(String name, Storage.EntryType keyType, 
    			     Storage.EntryType dataType, 
			     boolean uniqueValues,
			     BtreePageSource pageSource) 
			     throws StorageException {
	super(name, keyType, dataType, pageSource);
	this.uniqueValues = uniqueValues;
    }

    protected void init() throws StorageException {
        uniqueKeys = false;
	super.init();
    }

    /*
     * No-argument constructor for reconstructing via read().  Only used with
     * Btree's whose pageSource is a BtreeMDRSource.
     */
    public MultivaluedBtree() {
    }

    /** Returns a collection view of the values associated in the index with specified key.
     * Returned collection is read only and may not be modified.
     * If there are no values associated with the key empty collection is returned.
     * @return
     * @param key
     * @throws StorageException
     */
    public Collection getItems(Object key) throws StorageException {
        WeakReference ref = (WeakReference) listsByKey_cache.get (new BtreeListByKey.Key (key));
        BtreeListByKey list = null;
        if (ref != null)
            list = (BtreeListByKey) ref.get();
        if (list == null) {
            list = new BtreeListByKey(this, key);
            listsByKey_cache.put (list, new WeakReference (list));
        }
        return list;
    }
    
   /** Like getItems, but if the values in the index are a key type, 
     * returns the objects associated with the keys.
     * @return
     * @param key
     * @param repos where to look objetcs up
     * @throws StorageException
     */
    public Collection getObjects(Object key, SinglevaluedIndex repos) throws StorageException {
        WeakReference ref = (WeakReference) listsByKey_cache.get (new BtreeListByKey.Key (key));
        BtreeListByKey list = null;
        if (ref != null)
            list = (BtreeListByKey) ref.get();
        if (list == null) {
            list = new BtreeListByKey(this, key);
            listsByKey_cache.put (list, new WeakReference (list));
        }
        return new BtreeListByKeyRepos(list, repos);
    }
    
    /** If true, the collection of values is constrained to hold
     * no more than one of any value.
     * @throws StorageException
     * @return
     */
    public boolean isUnique() {
        return uniqueValues;
    }

    /** Removes the first occurrence of the specified element in the list
     * of values associated with the specified key.
     * @return true if this index changed as a result of this call
     * @param key
     * @param value
     * @throws StorageException
     */
    public boolean remove(Object key, Object value) throws StorageException {
        beginWrite();
	try { 
	    byte[] keyBuffer, dataBuffer;
	    boolean result;

	    if ((keyBuffer = keyInfo.toBuffer(key)) == null) {
		throw new StorageBadRequestException(
		    MessageFormat.format(
		  "Invalid key type for this index: {0} received, {1} expected",
			    new Object[] {
				key.getClass().getName(),
				keyInfo.typeName()} ));
	    }

	    if ((dataBuffer = dataInfo.toBuffer(value)) == null) {
		throw new StorageBadRequestException(
		    MessageFormat.format(
	         "Invalid data type for this index: {0} received, {1} expected",
			    new Object[] {
				value.getClass().getName(),
				dataInfo.typeName()} ));
	    }

	    BtreePage root = pageSource.getPage(rootPageId, this);
	    result = root.remove(keyBuffer, dataBuffer);
            if (result)
                updateKeyModCount (key);
	    pageSource.unpinPage(root);
	    return result;
	} finally {
	    endWrite();
	}
    }
    
    public void add(Object key, Object data) throws StorageException {
        super.add (key, data);
        updateKeyModCount (key);
    }
    
    public boolean remove(Object key) throws StorageException {
        boolean result = super.remove (key);
        if (result)
            updateKeyModCount (key);
        return result;
    }
    
    /**
     * Returns list of {@link BtreeEntryImpl} key-value pairs, where key contains 
     * the queried prefix.
     */
    public synchronized Collection queryByKeyPrefix (Object prefix, SinglevaluedIndex primaryIndex) throws StorageException {
        if (keyType != Storage.EntryType.STRING) {
            throw new UnsupportedOperationException ("Key type must be EntryType.STRING");
        }
        
        List result = new LinkedList ();        
        byte [] prefixBytes = keyInfo.toBuffer (prefix);
        SearchResult location = getLocation (prefixBytes);
        if (location.entryNum == location.page.numEntries())
            BtreePage.getNext (null, location);
        
        while ((location.entryNum < location.page.numEntries()) &&
            SinglevaluedBtree.isPrefix (prefixBytes, location.page.getKey (location.entryNum))) {
            byte [] key = location.page.getKey (location.entryNum);
            Object keyObject = keyInfo.objectFromBuffer (key, primaryIndex);
            Collection data = getObjects (keyObject, primaryIndex);
            Object entry = new MapEntryImpl (keyObject, data);            
            result.add (entry);
            int size = data.size ();
            location.page.findNth (location, key, size, false);
        } // while
        return result;
    }
    
    private static boolean isPrefix (byte [] prefix, byte [] key) {
        if (prefix.length > key.length)
            return false;
        for (int x = 0; x < prefix.length; x++) {
            if (prefix [x] != key [x])
                return false;
        }
        return true;
    }
    
    /**
     * Checks if an instance of BtreeListByKey related to key is in cache, if so, increases
     * list's modification counter.
     */
    protected void updateKeyModCount (Object key) {
        WeakReference ref = (WeakReference) listsByKey_cache.get (new BtreeListByKey.Key (key));
        if (ref == null)
            return;
        BtreeListByKey list = null;
        list = (BtreeListByKey) ref.get ();
        if (list != null)
            list.increaseModCount ();        
    }
        
}
