/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;

/**
* This class exists only to provide data conversions between primitive
* types and byte arrays.
*/
public class Converter {

    /* We encode strings to UTF-8 */

    private static final String ENCODING = "UTF-8";

    /** write a byte to an array
    * @param array array to write to 
    * @param offset offset in array at which to write byte. On return, this
    * will point to the next byte in the array
    * @param data byte to write to array
    */
    public static final void writeByte(
            byte array[], IntHolder offset, byte data){
        offset.setValue(writeByte(array, offset.getValue(), data));
    }

    /** write a byte to an array
    * @param array array to write to 
    * @param offset offset in array at which to write byte
    * @param data byte to write to array
    * @return offset in array which follows the written byte
    */
    public static final int writeByte(byte array[], int offset, byte data) {
        array[offset++] = data;
        return offset;
    }

    /** write a short to an array
    * @param array array to write to 
    * @param offset offset in array at which to write short. On return, this
    * will point to the next byte in the array
    * @param data short to write to array
    */
    public static final void writeShort(
            byte array[], IntHolder offset, short data){
        offset.setValue(writeShort(array, offset.getValue(), data));
    }

    /** write a short to an array
    * @param array array to write to 
    * @param offset offset in array at which to write short
    * @param data short to write to array
    * @return offset in array which follows the written short
    */
    public static final int writeShort(byte array[], int offset, short data) {
        array[offset++] = (byte)(data >> 8);
        array[offset++] = (byte)data;
        return offset;
    }

    /** write an int to an array
    * @param array array to write to 
    * @param offset offset in array at which to write int. On return, this
    * will point to the next byte in the array
    * @param data int to write to array
    */
    public static final void writeInt(
            byte array[], IntHolder offset, int data){
        offset.setValue(writeInt(array, offset.getValue(), data));
    }

    /** write a int to an array
    * @param array array to write to 
    * @param offset offset in array at which to write int
    * @param data int to write to array
    * @return offset in array which follows the written int
    */
    public static final int writeInt(byte array[], int offset, int data) {
        array[offset++] = (byte)(data >> 24);
        array[offset++] = (byte)(data >> 16);
        array[offset++] = (byte)(data >> 8);
        array[offset++] = (byte)data;
        return offset;
    }

    /** write a long to an array
    * @param array array to write to 
    * @param offset offset in array at which to write long. On return, this
    * will point to the next byte in the array
    * @param data long to write to array
    */
    public static final void writeLong(
            byte array[], IntHolder offset, long data){
        offset.setValue(writeLong(array, offset.getValue(), data));
    }

    /** write a long to an array
    * @param array array to write to 
    * @param offset offset in array at which to write long
    * @param data long to write to array
    * @return offset in array which follows the written long
    */
    public static final int writeLong(byte array[], int offset, long data) {
        array[offset++] = (byte)(data >>> 56);
        array[offset++] = (byte)(data >>> 48);
        array[offset++] = (byte)(data >>> 40);
        array[offset++] = (byte)(data >>> 32);
        array[offset++] = (byte)(data >>> 24);
        array[offset++] = (byte)(data >>> 16);
        array[offset++] = (byte)(data >>> 8);
        array[offset++] = (byte)data;
        return offset;
    }

    /** write a string to an array.  This uses the same format as 
    * DataOutput.WriteUTF, that is, after converting the String to an array
    * of bytes in the Java UTF-8 encoding, write two bytes containing the 
    * array length followed by the array itself.
    * @param array array to write to
    * @param offset offset in array at which to write string. On return, this
    * will point to the next byte in the array
    * @param data string to write to array
    */
    public static final void writeString(
            byte array[], IntHolder offset, String data){
        offset.setValue(writeString(array, offset.getValue(), data));
    }

    /** write a string to an array.  This uses the same format as 
    * DataOutput.WriteUTF, that is, after converting the String to an array
    * of bytes in the Java UTF-8 encoding, write two bytes containing the 
    * array length followed by the array itself.
    * @param array array to write to
    * @param offset offset in array at which to write string
    * @param data string to write to array
    * @return offset in array which follows the written string
    */
    public static final int writeString(byte array[], int offset, String data) {
        byte enc[] = convertStringToUTF8(data);
        offset = writeShort(array, offset, (short)enc.length);
        if (enc.length > 0)
            System.arraycopy(enc, 0, array, offset, enc.length);
        return offset + enc.length;
    }
        
    /** convert a String to a byte array
    * @param str string to convert
    * @return string in UTF-8 format
    */
    public static final byte[] convertStringToUTF8(String str) {
        byte result[] = null;
        try {
            result = str.getBytes(ENCODING);
        } catch (UnsupportedEncodingException ex) {
            /* UTF-8 should always be supported */
            throw new RuntimeException("UTF-8 not supported!");
        }
        return result;
    }

    /** read a byte from an array
    * @param array array to read from 
    * @param offset offset in array from which to read byte.  After returning,
    * this points to the offset in the array past the data read.
    * @return data read from array
    */
    public static final byte readByte(byte array[], IntHolder offset) {
        int offst = offset.getValue();
        byte data = readByte(array, offst);
        offset.setValue(offst+1);
        return data;
    }

    /** read a byte from an array
    * @param array array to read from 
    * @param offset offset in array from which to read byte. 
    * @return data read from array
    */
    public static final byte readByte(byte array[], int offst) {
        byte data = array[offst];
        return data;
    }

    /** read a short from an array
    * @param array array to read from 
    * @param offset offset in array from which to read short.  After returning,
    * this points to the offset in the array past the data read.
    * @return data read from array
    */
    public static final short readShort(byte array[], IntHolder offset) {
        int offst = offset.getValue();
        short data = readShort(array, offst);
        offset.setValue(offst+2);
        return data;
    }

    /** read a short from an array
    * @param array array to read from 
    * @param offset offset in array from which to read short. 
    * @return data read from array
    */
    public static final short readShort(byte array[], int offst) {
        short data = (short)(((short)array[offst++]) & 0xFF);
        data <<= 8;
        data |= (((short)array[offst]) & 0xFF);
        return data;
    }

    /** read an int from an array
    * @param array array to read from 
    * @param offset offset in array from which to read int.  After returning,
    * this points to the offset in the array past the data read.
    * @return data read from array
    */
    public static final int readInt(byte array[], IntHolder offset) {
        int offst = offset.getValue();
        int data = readInt(array, offst);
        offset.setValue(offst + 4);
        return data;
    }

    /** read an int from an array
    * @param array array to read from 
    * @param offset offset in array from which to read int.  
    * @return data read from array
    */
    public static final int readInt(byte array[], int offst) {
        int data = (((int)array[offst++]) & 0xFF);
        for (int i = 0; i < 3; i++) {
            data <<= 8;
            data |= (((int)array[offst++]) & 0xFF);
        }
        return data;
    }

    /** read a long from an array
    * @param array array to read from 
    * @param offset offset in array from which to read long.  After returning,
    * this points to the offset in the array past the data read.
    * @return data read from array
    */
    public static final long readLong(byte array[], IntHolder offset) {
        int offst = offset.getValue();
        long data = readLong(array, offst);
        offset.setValue(offst + 8);
        return data;
    }

    /** read a long from an array
    * @param array array to read from 
    * @param offset offset in array from which to read long.  
    * @return data read from array
    */
    public static final long readLong(byte array[], int offst) {
        long data = (((long)array[offst++]) & 0xFF);
        for (int i = 0; i < 7; i++) {
            data <<= 8;
            data |= (((long)array[offst++]) & 0xFF);
        }
        return data;
    }

    /** read a string from an array.  This will read a string written 
    * by WriteString.
    * @param array array to read from
    * @param offset offset in array from which to read string.  After returning,
    * this points to the offset in the array past the data read.
    * @return string read from array
    */
    public static final String readString(byte array[], IntHolder offset) {
        return readString(array, offset.getValue(), offset);
    }

    /** read a string from an array.  This will read a string written 
    * by WriteString.
    * @param array array to read from
    * @param offset offset in array from which to read string
    * @return string read from array
    */
    public static final String readString(byte array[], int offset) {
        return readString(array, offset, null);
    }

    /** read a string from an array.  This will read a string written 
    * by WriteString.
    * @param array array to read from
    * @param offset offset in array from which to read string
    * @param newOffset if non-null, this will receive the offset 
    * following the string
    * @return string read from array
    */
    private static String readString(
                    byte array[], int offset, IntHolder newOffset) {
        String str;

        short arrSize = readShort(array, offset);
        if (arrSize == 0) {
            str = "";
        }
        else {
            byte enc[] = new byte[arrSize];
            System.arraycopy(array, offset+2, enc, 0, arrSize);
            str = convertUTF8ToString(enc);
        }
        if (newOffset != null)
            newOffset.setValue(offset + 2 + arrSize);
        return str;
    }

    /** convert a byte array back to a string
    * @param buffer array of bytes in UTF-8 format to convert
    * @return string 
    */
    public static final String convertUTF8ToString(byte buffer[]) {
        String result = null;
        try {
        result = new String(buffer, ENCODING);
        } catch (UnsupportedEncodingException ex) {
            /* UTF-8 should always be supported */
            throw new RuntimeException("UTF-8 not supported!");
        }
        return result;
    }

}
    
